/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.input.SAXBuilder;


/**
 * The XMLSchemaParser is to save the parent-children relationships in
 * an XML schema file into a hashtable, then used by SchemaTree to create an
 * SchemaTree using insertRoot() and insertChildren().
 * The MyDTDParser was implemented in the same way. However this is not a good
 * way for the job at all. The XMLSchemaImporter was developed later, which
 * creates a SchemaTree directly from an input XML Schema file.
 *
 * The XMLSchemaParser is for parsing the schema files named xxx.xml in
 * SCIA_WORK_DIR/schemas, not
 * for those named xxx.xsd for which the XMLSchemaImporter based on XML Schema
 * parser is provided later.
 *
 * It is assumed that input XML Schema file has a docuemnt root
 * named schemaName, and the schema has a single root element, named as
 * schemaName + Type.
 *
 * It is also assumed that no dt:type, please change to type if dt:type is used.
 *
 *@author Guilian Wang
 */
public class XMLSchemaParser {
    // hash table to hold the children of each valid node in the schema tree
    static Hashtable eleAttrHash = new Hashtable();

    // the tag of the Schema element, also tree root 
    static String schemaRootTag;

    // the root element of the XML Schema document
    static Element schemaRootElem;

    public XMLSchemaParser(String xmlSchemaFile) {
        SAXBuilder builder = new SAXBuilder();

        try {
            // For reading file from jar file, only using relative path from 
            //CLASSPATH
            //InputStream is = this.getClass().getResourceAsStream(filePath);
            InputStream is = new FileInputStream(xmlSchemaFile);
            Document doc = builder.build(is);
            schemaRootElem = doc.getRootElement();

            if (schemaRootElem != null) {
                schemaRootTag = schemaRootElem.getAttributeValue("name");

                // hash children of each node of the schema tree
                hashChildren(schemaRootElem, true);

                // print out to the screen the schema tree
                //printSchemaTree(schemaRootElem, true, "  ");
            } else {
                if (SCIA.debug_on) {
                    System.err.println("schemaRootElem is null");
                }
            }
        } catch (Exception e) { // might indicate a well-formedness error 
            e.printStackTrace();
        }
    }

    /**
     * Store element-subelement and element-attribute relationships into
     * a hashtable for SchemaTree to create a schema tree.
     *
     * @para current The current Element is handled.
     * @para isSchemaDocRoot The boolean value of whether the current Element
     * is the shema file document root or not.
     */
    public void hashChildren(Element current, boolean isSchemaDocRoot) {
        String name = current.getName(); // element, elementType, schema, etc.
        String tag = current.getAttributeValue("name"); // tag
        String t = current.getAttributeValue("type");
        List children = current.getChildren();

        //if (name.equals("element") && t.equals("Item")) {
        List attrs = current.getAttributes();

        //System.out.println(tag + "'s attributes = " + attrs);
        String min = current.getAttributeValue("minOccurs");

        //if (min != null)
        //System.out.println(tag + "'s minOccurs = " + min);
        //}
        if (name.equalsIgnoreCase("sequence") ||
                (name.equalsIgnoreCase("complexType") && (tag == null))) {
            // a node is sequence or no-name-complexType
            // we don't want it in schema tree
            // add no-name-complexType or sequence's children as valid 
            // ancestor's children
            if (children != null) {
                Element ancestor = (Element) current.getParent();

                //System.out.println("ancestor = " + ancestor);
                String aName = ancestor.getAttributeValue("name");

                while (aName == null) {
                    ancestor = (Element) ancestor.getParent();
                    aName = ancestor.getAttributeValue("name");
                }

                List aChildren = (List) eleAttrHash.get(aName);

                if ((aChildren != null) && (aChildren.size() > 1)) {
                    // parent has more than one child

                    /********eleAttrHash.remove(pName);
                             if(SCIA.debug_on) System.err.println("before addAll");
                             Iterator iterator = children.iterator();
                             while (iterator.hasNext()) {
                             Element child = (Element) iterator.next();
                             child.setParent(parent);
                             pChildren.add(child);
                             }
                             if(SCIA.debug_on) System.err.println("after addAll");
                             eleAttrHash.put(pName, pChildren); */
                } else { // the valid ancestor has no more than one child

                    String key = aName;
                    Element aParent = (Element) ancestor.getParent();

                    if (aParent != null) {
                        String aPName = aParent.getAttributeValue("name");

                        while (aPName == null) {
                            aParent = (Element) aParent.getParent();
                            aPName = aParent.getAttributeValue("name");
                        }

                        key = aPName + "." + aName;
                    }

                    eleAttrHash.remove(key);
                    eleAttrHash.put(key, children);
                }

                //handle each child
                Iterator iterator = children.iterator();

                while (iterator.hasNext()) {
                    Element child = (Element) iterator.next();
                    hashChildren(child, false);
                }
            }
        } else {
            //System.out.println("not sequence or no-name-complexType: " + name);
            String type = current.getAttributeValue("type");

            if (type == null) {
                type = current.getAttributeValue("dt:type");
            }

            if (children != null) {
                //hash its children
                if (!isSchemaDocRoot) { // skip schemaRootElem

                    if (tag != null) {
                        String key = tag;
                        Element ancestor = (Element) current.getParent();

                        if (ancestor != null) {
                            String aName = ancestor.getAttributeValue("name");

                            while (aName == null) {
                                ancestor = (Element) ancestor.getParent();
                                aName = ancestor.getAttributeValue("name");
                            }

                            key = aName + "." + tag;

                            eleAttrHash.put(key, children);
                        }
                    }
                }

                //handle each child
                Iterator iterator = children.iterator();

                while (iterator.hasNext()) {
                    Element child = (Element) iterator.next();
                    hashChildren(child, false);
                }
            }
        }
    }

    public void printSchemaTree(Element subTreeRoot, 
    // the root of tree to print
    boolean isDocRoot, 
    //subTreeRoot is document root
    String spaces) {
        /**********************************************************************
        % Method Name :  XMLSchemaParser::printSchemaTree  (public)
        % File :         XMLSchemaParser.java
        %
        % Description :  print out the tree structure on screen. But it has not
        %                updated to cover more general type definition cases as
        %                SchemaTree::insertChildren does
        %
        **********************************************************************/

        //System.out.println("node hashtable is " + eleAttrHash);
        String current = subTreeRoot.getAttributeValue("name");
        String type = subTreeRoot.getAttributeValue("type");

        if (type == null) {
            type = subTreeRoot.getAttributeValue("dt:type");

            //System.out.println("type is " + type);
        }

        List children = new ArrayList();

        if (current != null) {
            String key = current;
            Element ancestor = (Element) subTreeRoot.getParent();

            if (ancestor != null) {
                String aName = ancestor.getAttributeValue("name");

                while (aName == null) {
                    ancestor = (Element) ancestor.getParent();
                    aName = ancestor.getAttributeValue("name");
                }

                key = aName + "." + current;
            }

            //System.out.println("key = " + key);
            children = (List) eleAttrHash.get(key);

            //System.out.println("current not null");
        }

        if ((children != null) && !children.isEmpty()) {
            // the subTreeRoot has children
            //System.out.println("current children not null");
            if (SCIA.debug_on) {
                System.err.println(spaces + current + "  " + type);
            }

            //System.out.println(current + " has children");
            Iterator iterator = children.iterator();
            spaces = spaces + "    ";

            while (iterator.hasNext()) {
                Element child = (Element) iterator.next();

                if (!(child.getName().equalsIgnoreCase("Attribute"))) {
                    if (!(child.getAttributeValue("name") == null)) {
                        printSchemaTree(child, false, spaces);
                    }
                } else {
                    //System.out.println(" child is attribute = " + child.getName());
                }
            }
        } else { // no children, go to its type definition to get children

            String currentNodeType;

            //System.out.println("current not null, but no children");
            if (isDocRoot) { // for docRoot, to get to the schema root by name+Type 
                currentNodeType = schemaRootTag + "Type";

                if (SCIA.debug_on) {
                    System.err.println(spaces + schemaRootTag + "  " +
                        currentNodeType);
                }

                //System.out.println(current + " is doc root, has no children");
                //schemaRootTag = currentNodeType;
            } else {
                currentNodeType = type;

                if ((current == null) &&
                        subTreeRoot.getName().equalsIgnoreCase("element")) {
                    if (SCIA.debug_on) {
                        System.err.println(spaces + currentNodeType + "  " +
                            currentNodeType);
                    }
                } else {
                    if (SCIA.debug_on) {
                        System.err.println(spaces + current + "  " +
                            currentNodeType);
                    }
                }
            }

            String key = schemaRootTag + "." + currentNodeType;
            children = (List) eleAttrHash.get(key);

            if ((children != null) && !children.isEmpty()) {
                spaces = spaces + "    ";

                Iterator iterator = children.iterator();

                while (iterator.hasNext()) {
                    Element child = (Element) iterator.next();

                    if (!(child.getName().equalsIgnoreCase("Attribute"))) {
                        //System.out.println(current + "'s child is not" +
                        // has name " + child.getName());
                        printSchemaTree(child, false, spaces);
                    }
                }
            }
        }
    }

    public static void main(String[] args) {
        if (args.length == 0) {
            if (SCIA.debug_on) {
                System.err.println("There is no input XML file!");
            }

            return;
        }

        XMLSchemaParser xmlSchemaParser = new XMLSchemaParser(args[0]);
    }
}
