/**
 *    '$RCSfile: NamedOntModel.java,v $'
 *
 *     '$Author: bowers $'
 *       '$Date: 2005/11/08 00:29:57 $'
 *   '$Revision: 1.9 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms;

import com.hp.hpl.jena.ontology.*; 
import com.hp.hpl.jena.rdf.model.*; 
import java.util.*;

/**
 *  This class encapsulates/wraps an ontology model. 
 *  TODO: still developing the interface for this class.
 */
public class NamedOntModel implements Comparable { 

    /** 
     * Creates a named ont model for the ontology given in filePath
     * @param filePath The path to the file to load. 
     * We assume the ontology is already classified.
     */
    public NamedOntModel(String filePath) throws Exception {
	_ontModel = ModelFactory.createOntologyModel(OntModelSpec.OWL_DL_MEM, null);
	_ontModel.read("file:" + filePath);
	initialize();
    }


    /**
     * Initializes a named ont model wrapper for the ontology.
     */
    private void initialize() {
	Iterator iter = _ontModel.listOntologies();
	assert iter.hasNext();
	// grab the first:
	_ontology = (Ontology)iter.next();
	_name = new String(_ontology.getLabel(null));
	if(_name == null)
	    _name = new String(_ontology.getLocalName());
    }

    /**
     * @return The namespace for the ontology
     */
    public String getNameSpace() {
	return _ontology.getNameSpace();
    }

    /** 
     * @return The root classes, i.e., without a named superclass, for this ontology.
     */
    public Iterator getRootClasses(boolean sorted) {
	Vector classes = new Vector();
	// get all the classes
 	for(Iterator iter = _ontModel.listNamedClasses(); iter.hasNext();) {
 	    OntClass c = (OntClass)iter.next();
 	    if(!classes.contains(c))
 		classes.add(c);
 	}
	Vector results = new Vector();
	boolean foundResult = true;
	// check each class to see if a root
	for(Iterator iter = _ontModel.listNamedClasses(); iter.hasNext();) {
	    OntClass c = (OntClass)iter.next();
	    // get the direct superclasses and see if they are one of the named classes
	    for(Iterator sups = c.listSuperClasses(true); sups.hasNext() && foundResult;) {
		OntClass s = (OntClass)sups.next();
		if(classes.contains(s))
		    foundResult = false;
	    }
	    if(foundResult)
		results.add(new NamedOntClass(c));
	    foundResult = true;
	}
	if(sorted)
	    Collections.sort(results);
	return results.iterator();
    }


    /**
     * gets a list of the named classes in the ontology
     * @param sorted Return sorted list if true
     * @return A sorted list of named ontology classes
     */
    public Iterator getNamedClasses(boolean sorted) {
	Vector results = new Vector();
 	for(Iterator iter = _ontModel.listNamedClasses(); iter.hasNext();) {
 	    OntClass c = (OntClass)iter.next();
 	    if(!results.contains(c))
 		results.add(new NamedOntClass(c));
 	}
	if(sorted)
	    Collections.sort(results);
 	return results.iterator();
    }


    /**
     * gets a list of the named properties in the ontology
     * @param sorted Return sorted list if true
     * @return A sorted list of named ontology classes
     */
    public Iterator getNamedProperties(boolean sorted) {
	Vector results = new Vector();
 	for(Iterator iter = _ontModel.listObjectProperties(); iter.hasNext();) {
 	    OntProperty p = (OntProperty)iter.next();
 	    if(!results.contains(p))
 		results.add(new NamedOntProperty(p));
 	}
	for(Iterator iter = _ontModel.listDatatypeProperties(); iter.hasNext();) {
	    OntProperty p = (OntProperty)iter.next();
	    if(!results.contains(p))
		results.add(new NamedOntProperty(p));
	}
	if(sorted)
	    Collections.sort(results);
 	return results.iterator();
    }


    /**
     *
     */
    public String toString() {
	return getName();
    }


    /**
     * @return The name of the model
     */
    public String getName() {
	return _name;
    } 



    public int compareTo(Object obj) {
	String str1 = toString();
	String str2 = obj.toString();
	return str1.compareTo(str2);
    }


    /* PRIVATE MEMBERS */

    private OntModel _ontModel;	   // the model
    private Ontology _ontology;	   // the ontology in the model
    private String _name;	   // the name of the ontology, or the namespace


} //NamedOntModel
