/**
 *    '$RCSfile: OntologyCatalog.java,v $'
 *
 *     '$Author: bowers $'
 *       '$Date: 2005/10/18 23:29:14 $'
 *   '$Revision: 1.9 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms;

/**
 * 
 */

import java.util.*;
import org.kepler.sms.util.*;
import javax.swing.*;

public class OntologyCatalog { 
    
    /**
     * This is a singleton class, so the constructor is "hidden"
     */
    protected OntologyCatalog() {
	initialize();
    }

    /**
     * Initialize the catalog. Reads from the index file (index.xml).
     * @see org.kepler.sms.util.OntologyConfiguration
     */
    protected void initialize() {
	OntologyConfiguration config = OntologyConfiguration.instance();

	for(Iterator iter = config.getFilePathNames(); iter.hasNext();) {
	    String ontoFilePath = (String)iter.next();
	    NamedOntModel mdl;
	    try {
		mdl = new NamedOntModel(ontoFilePath);
		_namedOntModels.add(mdl);
		if(config.isLibraryOntology(ontoFilePath))
		    _libraryModels.add(mdl);
	    } catch(Exception e) {
		JFrame f = new JFrame();
		String errmsg = "Cannot locate ontology: " + ontoFilePath;
		JOptionPane.showMessageDialog(f, errmsg, "Configuration Error", JOptionPane.WARNING_MESSAGE);
		f.dispose();
	    }
	}
    }

    /**
     * This method must be called to create/obtain an instance of the catalog
     * @return The unique instance of this class
     */
    public static OntologyCatalog instance() {
	if (_catalog == null)
	    _catalog = new OntologyCatalog();
	return _catalog;
    }    

    /**
     * @return The set of named ontology models in the catalog
     */
    public Iterator getNamedOntModels() {
	return _namedOntModels.iterator();
    }

    /**
     * This method must be called to create/obtain an instance of the catalog
     * @return The unique instance of this class
     */
    public Iterator getLibraryNamedOntModels() {
	return _libraryModels.iterator();
    }

    /**
     * @return The set of names for ontologies
     */
    public Iterator getOntologyNames() {
	Vector results = new Vector();
	for(Iterator iter = getNamedOntModels(); iter.hasNext();) {
	    NamedOntModel m = (NamedOntModel)iter.next();
	    results.add(m.getName());
	}
	return results.iterator();
    }

    public Iterator getLibraryOntologyNames() {
	Vector results = new Vector();
	for(Iterator iter = getLibraryNamedOntModels(); iter.hasNext();) {
	    NamedOntModel m = (NamedOntModel)iter.next();
	    results.add(m.getName());
	}
	return results.iterator();	
    }

    /**
     * @return The name of the ontology for the given namespace. The namespace can optionally end in #.
     * @param namespace The namespace of the desired ontology
     */
    public String getOntologyName(String namespace) {
	for(Iterator iter = getNamedOntModels(); iter.hasNext();) {
	    NamedOntModel m = (NamedOntModel)iter.next();
	    String nspace = m.getNameSpace() + "#";
	    if(m.getNameSpace().equals(namespace) || nspace.equals(namespace))
		return m.getName();
	}
	return null;
    }

    /**
     * @return The named ontology class for the given semantic type or
     * null if none exists.
     * @param st The semantic type to search for
     */
    public NamedOntClass getNamedOntClass(SemanticType st) {
	String conceptId = st.getConceptId();
	String [] parts = conceptId.split("#");
	// make sure we have a valid semantic type
	assert parts.length == 2; 
	// search for a matching model and class
	for(Iterator iter = getNamedOntModels(); iter.hasNext();) {
	    NamedOntModel m = (NamedOntModel)iter.next();
	    if(m.getNameSpace().equals(parts[0])) { 
		for(Iterator iter2 = m.getNamedClasses(false); iter2.hasNext();) {
		    NamedOntClass c = (NamedOntClass)iter2.next();
		    if(c.getLocalName().equals(parts[1]))
			return c;
		}
	    }
	}
	return null;
    }


    /**
     * @return The named ontology class for the given semantic type or
     * null if none exists.
     * @param namespace The namespace of the class
     * @param localName The local name (id) of the class within the namespace
     *
     * TODO: Pick up the class restrictions ??? 
     */
    public NamedOntClass getNamedOntClass(String namespace, String localName) {
	assert namespace != null && localName != null;
	// search for a matching model and class
	for(Iterator iter = getNamedOntModels(); iter.hasNext();) {
	    NamedOntModel m = (NamedOntModel)iter.next();
	    String m_nspace = m.getNameSpace() + "#";
	    if(m.getNameSpace().equals(namespace) || m_nspace.equals(namespace)) { 
		for(Iterator iter2 = m.getNamedClasses(false); iter2.hasNext();) {
		    NamedOntClass c = (NamedOntClass)iter2.next();
		    if(c.getLocalName().equals(localName))
			return c;
		}
	    }
	}
	return null;
    }
    

    /* PRIVATE MEMBERS */

    private static OntologyCatalog _catalog = null; // singleton instance
    private Vector _namedOntModels = new Vector();  // a list of the managed ontology models
    private Vector _libraryModels = new Vector(); // a subset of namedOntModels for actor libraries

    private String KEPLER = System.getProperty("KEPLER");
    private String ONTO_PATH = KEPLER + "/configs/ptolemy/configs/kepler/ontologies/"; 
}
