/* I/O port for collection actors.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp;

import org.nddp.tokens.ClosingDelimiterToken;
import org.nddp.tokens.CollectionToken;
import org.nddp.tokens.DataToken;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.tokens.OpeningDelimiterToken;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.Token;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;

/**
 * A subclass of TypedIOPort that implements the 
 * {@link org.nddp.CollectionPort} interface.  This class should be used
 * in actors derived from {@link org.nddp.AtomicCoactor} wherever a 
 * TypedIOPort would be used in a collection-unaware actor.
 * This class class provides easy-to-use factories, simplified 
 * get and send methods, and support for automatically recognizing tokens
 * derived from {@link org.nddp.tokens.CollectionToken}, including delimiters 
 * and metadata tokens.  Note that, unlike TypedIOPort, an instance of 
 * CollectionIOPort must be an input port or output port, but cannot be both.
 * <p>
 * <b>Usage:</b>  
 * <ul>
 * <li>A CollectionIOPort must be declared as a public field of the containing
 * actor.</li>
 * <li>A CollectionIOPort should be constructed by calling one of the factory 
 * method from within the constructor of the containing actor.</li>
 * <li>All collection-processing actors should use this class rather than Ptolemy's
 * TypedIOPort class.</li>
 * </ul>
 * 
 *  @author Timothy M. McPhillips
 */

public class CollectionIOPort extends TypedIOPort implements CollectionPort {
        

    /** 
     * Constructs an instance of CollectionIOPort--<b>do not use</b>.  This 
     * public constructor is provided solely to support instantiation of ports
     * added to actors by the user at run time.  The port so constructed is an
     * input port by default.
     * 
     * @param container ComponentEntity containing this port.
     * @param name Name of this port.
     */
	public CollectionIOPort(ComponentEntity container, String name) 
		throws IllegalActionException, NameDuplicationException {
	
	    // create an input port using the 4-parameter private constructor
		this(container, name, false, false);
   	}

    /** 
     * Constructs an instance of CollectionIOPort using the containing 
     * {@link org.nddp.Coactor} as the handler for delimiters and 
     * metadata.  This constructor is private to enforce use of factory 
     * methods for greater clarity.
     * 
     * @param container ComponentEntity containing this port.
     * @param name Name of this port.
     * @param isInput Flag indicating if this is an input port.
     * @param isOutput Flag indicating if this is an output port.
     */
	protected CollectionIOPort(ComponentEntity container, String name, 
        boolean isInput, boolean isOutput) throws IllegalActionException, 
        NameDuplicationException {
     
	    // call the private 5-argument constructor
    		this(container, name, isInput, isOutput, null);	
    }

	/** 
     * Constructs an instance of CollectionIOPort.  This constructor is private
     * to enforce use of the factory methods for greater clarity.
     * 
     * @param container ComponentEntity containing this port.
     * @param name Name of this port.
     * @param isInput Flag indicating if this is an input port.
     * @param isOutput Flag indicating if this is an output port.
     * @param nondataHandler Reference to an object that implements the 
     * {@link CollectionIOPort.NondataHandler} interface 
     * to handle delimiter and metadata tokens received on this port.
     * 
     * @exception NullPointerException If the actor, name, or nondataHandler
     * parameters are null.
     */
	private CollectionIOPort(ComponentEntity container, String name, 
        boolean isInput, boolean isOutput, NondataHandler nondataHandler)
        throws IllegalActionException, NameDuplicationException {
     
	    // call the superclass constructor
    		super(container, name, isInput, isOutput);	
    		
    		// validate the parameters
    		if (name == null) {
    		    throw new NullPointerException("Null name not allowed.");
    		}
    		
    		// validate the parameters
    		if (name.equals("")) {
    		    throw new IllegalActionException("Empty name not allowed.");
    		}
    		
    		// configure the port to process all types of tokens
       	setTypeEquals(BaseType.GENERAL);

        // set ouput ports to multiports
        setMultiport(isOutput);
       	
       	// store the reference to the metadata handler
    		_nondataHandler = nondataHandler;
    		
    		// store the reference to the containing actor
    		_container = container;
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
    /** 
     * Constructs and returns an instance of CollectionIOPort configured for 
     * use as an input port, with the containing actor serving as the non-data
     * handler for this port.  Note that an actor derived from 
     * {@link org.nddp.AtomicCoactor} must be passed as the first argument
     * (rather than any ComponentEntity), because 
     * {@link org.nddp.AtomicCoactor} implements 
     * {@link CollectionIOPort.NondataHandler} and thus may serve as the 
     * nondata handler.
     * 
     * @param actor Instance of {@link org.nddp.AtomicCoactor} 
     * containing this port.
     * @param name Name of this port.
     * @return The new CollectionIOPort.
     */
	public static CollectionIOPort createInputPort(ComponentEntity container, 
        String name) throws IllegalActionException, NameDuplicationException {
    	
	    if (container == null) {
	        throw new NullPointerException("Null container not allowed.");
	    }
	    
	    if (! (container instanceof Coactor)) {
	    		throw new ClassCastException("Container must implement CollectionActor interface.");
	    }
	    
	    // create and return an input port using the private 4-argument 
	    // constructor
    		return new CollectionIOPort(container, name, true, false);
    }

    /** 
     * Constructs and returns an instance of CollectionIOPort configured 
     * for use as an input port. 
     * 
     * @param container The actor or other ComponentEntity containing this port.
     * @param name Name of this port.
     * @param nondataHandler Reference to an object that implements the 
     * {@link CollectionIOPort.NondataHandler} interface and will handle 
     * {@link org.nddp.tokens.CollectionToken} objects received on this port.
     */
	public static CollectionIOPort createInputPort(ComponentEntity container, 
        	String name, NondataHandler nondataHandler) 
		throws IllegalActionException, NameDuplicationException {
    	
	    if (container == null) {
	        throw new NullPointerException("Null container not allowed.");
	    }
	    
	    // create and return an input port using the private 
	    // 5-argument constructor
		return new CollectionIOPort(container, name, true, false, 
	        nondataHandler);
    }
	
    /** 
     * Constructs and returns an instance of CollectionIOPort configured for 
     * use as an output port.
     *  
     * @param container The actor or other ComponentEntity containing this port.
     * @param name Name of this port.
     * @return The new CollectionIOPort.
     */
	public static CollectionIOPort createOutputPort(ComponentEntity container, 
        String name) throws IllegalActionException, NameDuplicationException {
    	
	    if (container == null) {
	        throw new NullPointerException("Null container not allowed.");
	    }

	    // craete and return an output port using the private 
	    // 4-argument constructor
    		return new CollectionIOPort(container, name, false, true);
    }
    
    /** 
     * Reads next token receieved by this input port.  Calls handler methods
     * in the {@link CollectionIOPort.NondataHandler} object registered for
     * this port and then returns null if a non-data token is received. 
     * Returns the token if a data token is received.  The method blocks
     * until a token arrives. 
     *
     * @return The next data token received by the port or null if a non-data
     * token is received next.
     * @exception IllegalActionException If this port is not an input port.
     */
	public Token get() throws IllegalActionException {
	    return get(true);
	}
	
    /** 
     * Reads next token receieved by this input port.  Calls handler methods
     * in the {@link CollectionIOPort.NondataHandler} object registered for
     * this port and then returns null if a non-data token is received. 
     * Returns the token if a data token is received.  The method blocks
     * until a token arrives. 
     * 
     * @param unwrapData Indicates whether data tokens wrapped in 
     * instances of {@link org.nddp.tokens.DataToken} should be unwrapped
     * and returned.  If false, the DataToken itself is
     * returned.
     * 
     * @return The next data token received by the port or null if a non-data
     * token is received next.
     * @exception IllegalActionException If this port is not an input port.
     */
    public Token get(boolean unwrapData) throws IllegalActionException { 
    	 
        	if (! isInput()) {
        	    throw new IllegalActionException("Not an input port.");
        	}
            
	    	// read the next token from channel 0 of the port
		Token token = (Token)get(0);
		
        NondataHandler collectionHandler;
        if (_nondataHandler != null) {
            collectionHandler = _nondataHandler;
        } else {
            collectionHandler = ((Coactor)_container).collectionHandler();
        }

        if (token instanceof CollectionToken) {
		    
		    if (unwrapData && token instanceof DataToken) {
		        return ((DataToken)token).getValue();
		    }

	    		if (token instanceof OpeningDelimiterToken) {
	    		 	collectionHandler.handleOpeningDelimiter(
    		 	        (OpeningDelimiterToken)token);
	    		    return null;
	    		}
	    			
	    		if (token instanceof ClosingDelimiterToken) {
	    		    collectionHandler.handleClosingDelimiter(
    		            (ClosingDelimiterToken)token);
	    		    return null;
	    		}
	    		
	    		if (token instanceof MetadataToken) {
	    		    collectionHandler.handleMetadataToken(
    		            (MetadataToken)token);
	    		    return null;
	    		}
	    		
	    		if (token instanceof LoopTerminationToken) {
				collectionHandler.handleLoopTerminationToken(
    		            (LoopTerminationToken)token);
	    		    return null;
	    		}
	    		    
	    		if (token instanceof ExceptionToken) {
				collectionHandler.handleExceptionToken(
			        (ExceptionToken)token);
				return null;
	    		}
		}
		
	    return token;
    }
    
    protected NondataHandler nondataHandler() {
        return _nondataHandler;
    }
    
    /** 
     * Sends a token on all channels connected to this output port.
     * 
     * @param token The token to send.
     * 
     * @exception IllegalActionException If this port is not an output port.
     * @exception NullPointerException If <code>token</code> is null.
     */
    public void send(Token token) throws IllegalActionException {
    
        	// Test preconditions
        	if (! isOutput()) {
        	    throw new IllegalActionException("Not an output port.");
        	}
        	
   		if (token == null) {
   		    throw new NullPointerException("Null token not allowed.");
   		}
   		
   		// send token on all channels
    		broadcast(token);
    }
    
    /** 
     * Sets the direction the port faces with respect to the actor when 
     * depicted in vergil.
     * <p>
     * 
     * @param direction One of the four cardinal directions: "north", "east",
     * "south", or "west."
     * 
     */
    public void setDirection(String direction) 
    		throws IllegalActionException, NameDuplicationException {
        
        if ( !( direction.equals("north") || direction.equals("east") ||
                direction.equals("south") || direction.equals("west"))) {
        
            throw new IllegalActionException("Port direction must be one of " +
                "north, east, south and west.");
        }
        
        StringAttribute portDirection;

        portDirection = (StringAttribute)getAttribute("_cardinal");
        
        if (portDirection == null) {
            portDirection = new StringAttribute(this, "_cardinal");
        }
            
        portDirection.setExpression(direction);
    }

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    /** Reference to the actor (or other ComponentEntity) containing 
     * this port. */
    private final ComponentEntity _container;
    
    /** Reference to the non-data handler for this port. */
    private final NondataHandler _nondataHandler;

    private static final long serialVersionUID = 1L;
	
	///////////////////////////////////////////////////////////////////
	////               public inner interfaces                     ////

    /** Interface implemented by objects that provide handlers for 
     * non-data tokens received by a {@link CollectionIOPort}. */
    public interface NondataHandler {

    		///////////////////////////////////////////////////////////////////
		////      public methods for CollectionIOPort.NondataHandler     //// 
	
    		/** 
        	 * Handler for closing delimiter tokens (see {@link org.nddp.tokens.ClosingDelimiterToken}).
        	 * 
        	 * @param token The closing delimiter token.
        	*/
    		void handleClosingDelimiter(ClosingDelimiterToken token) 
    			throws IllegalActionException;
    		
    		/** 
        	 * Handler for exception tokens (see {@link org.nddp.tokens.ExceptionToken}).
        	 * 
        	 * @param token The exception token.
        	*/    		
    		void handleExceptionToken(ExceptionToken token) 
    			throws IllegalActionException;

    		/** 
        	 * Handler for loop termination tokens (see {@link org.nddp.tokens.LoopTerminationToken}).
        	 * 
        	 * @param token The loop termination token.
        	*/    
    		void handleLoopTerminationToken(LoopTerminationToken token) 
    			throws IllegalActionException;    		
    		
        	/** 
        	 * Handler for metadata tokens (see {@link org.nddp.tokens.MetadataToken}).
        	 * 
        	 * @param token The metadata token.
        	*/    		
    		void handleMetadataToken(MetadataToken token)
    			throws IllegalActionException;
    		
    		/** 
        	 * Handler for opening delimiter tokens (see {@link org.nddp.tokens.OpeningDelimiterToken}).
        	 * 
        	 * @param token The opening delimiter token.
        	*/
    		void handleOpeningDelimiter(
	        OpeningDelimiterToken token) throws IllegalActionException;
    }
}
