/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ptolemy.actor.IOPort;
import ptolemy.data.ArrayToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.Token;
import ptolemy.kernel.util.IllegalActionException;

////////////////////////////////////////////////////////////////////////// 
//// InputPortBuffer 
 
/** 
  @author Timothy M. McPhillips
*/ 

public class OutputPortBuffer {
    
    public OutputPortBuffer(IOPort port) throws IllegalActionException {

        _port = port;
        
        // make sure port name indicates it is bound to collection contents
        Matcher portNameMatcher = OUTPUT_PORT_NAME_PATTERN.matcher(port.getName());
        if (! portNameMatcher.find()) {
            throw new IllegalActionException(
                "Port name does not match expected pattern.");
        }

        String subcollectionTypeString = portNameMatcher.group(2);
        String typeName = portNameMatcher.group(3);
        String cardinality = portNameMatcher.group(4);
        String count = portNameMatcher.group(6);
        String options = portNameMatcher.group(7);
        
//        System.out.println("subcollectionType = " + subcollectionTypeString);
//        System.out.println("typeName = " + typeName);
//        System.out.println("cardinality = " + cardinality);
//        System.out.println("count = " + count);
//        System.out.println("options = " + options);
        
        // make sure type name matches a registered data type
        if (! DataTypes.isDefined(typeName)) {
            throw new IllegalActionException(
                "No data type " + typeName + ".");
        }
        
        // look up the data type
        _dataType = DataTypes.valueOfString(typeName);

        // make sure subcollection type name matches a registered data type
        if (subcollectionTypeString != null) {
            
            if (CollectionTypes.isDefined(subcollectionTypeString)) {
                _subcollectionType = CollectionTypes.valueOfString(subcollectionTypeString);
            } else {
                throw new IllegalActionException("No collection type " + subcollectionTypeString);
            }                        
        }
        
        if (cardinality == null) {
            _minimum = 1;
            _maximum = 1;
            _bundlesTokens = false;
        } else if (cardinality.equals("?")) {
            _minimum = 0;
            _maximum = 1;
            _bundlesTokens = false;
        } else if (cardinality.equals("+")) {
            _minimum = 1;
            _maximum = 0; 
            _bundlesTokens = true;
        } else if (cardinality.equals("*")) {
            _minimum = 0;
            _maximum = 0;
            _bundlesTokens = true;
        }

        if (count != null) {
            int tokenCount = Integer.parseInt(count);
            _minimum = tokenCount;
            _maximum = tokenCount;
            _bundlesTokens = true;
        }
        
        if (options != null) {
            
            // look for metadata key
            Matcher keyMatcher = METADATA_KEY_PATTERN.matcher(options);
            if (keyMatcher.find()) {
                _metadataKey = keyMatcher.group(1);
            }
        }
    }
    
    public Class dataType() {
        return _dataType;
    }
    
    private void addTokenToCollection(CollectionManager collectionManager, Token token) {

        // handle object tokens and data tokens differently
        if (token instanceof ObjectToken) {
            collectionManager.addDomainObject((DomainObject)((ObjectToken)token).getValue());
        } else {
            collectionManager.addDataToken(token);
        }
    }

    public void flush(CollectionManager collectionManager) throws IllegalActionException {

        int tokenCount = 0;
        
        if (_metadataKey != null) {

            // read the next token
            Token token = _port.getInside(0);

            // make sure token is of declared type
             validateDataTokenType(token);
            
            collectionManager.addMetadata(_metadataKey, token);  
            tokenCount++;
        
        } else {
        
            for (int channel = 0; channel < _port.getWidthInside(); channel++) {
            
                // process each token one at a time
                while (_port.hasTokenInside(channel)) {
                        
                    // read the next token
                    Token token = _port.getInside(channel);
         
                    if (token instanceof ArrayToken) {
                        
                        Token[] tokenArray = ((ArrayToken)token).arrayValue();
    
                        for (int j = 0; j < tokenArray.length; j++) {
                            validateDataTokenType(tokenArray[j]);
                            tokenCount++;
                            validateTokenCountMaximum(tokenCount);
                            addTokenToCollection(collectionManager, tokenArray[j]);
                        }
                        
                    } else {
                        
                        validateDataTokenType(token);
                        tokenCount++;
                        validateTokenCountMaximum(tokenCount);
                        addTokenToCollection(collectionManager, token);
                    }
                }
            }
        }
        
        validateTokenCountMinimum(tokenCount);
    }

    private void validateTokenCountMaximum(int tokenCount) throws IllegalActionException {
        if (_maximum > 0 && tokenCount > _maximum) {
            throw new IllegalActionException("Too many tokens (" + tokenCount + ") produced by port " + _port.getName());
        }
    }

    private void validateTokenCountMinimum(int tokenCount) throws IllegalActionException {
        if (tokenCount < _minimum) {
            throw new IllegalActionException("Too few tokens (" + tokenCount + ") produced by port " + _port.getName());
        }
    }
    
    public void validateDataTokenType(Token token) throws IllegalActionException {
        
        Object object = token;
        
        if (token instanceof ObjectToken) {
            object = ((ObjectToken)token).getValue();
        }
        
        if (! _dataType.isInstance(object)) {
            throw new IllegalActionException("Incorrect token sent on port " + _port.getName());
        }        
    }
        
    public Class subcollectionType() {
        return _subcollectionType;
    }
    
    public String metadataKey() {
        return _metadataKey;
    }
    
    public IOPort port() {
        return _port;
    }
    
    public void clear() throws IllegalActionException {
      
        // drain any tokens left in port from previous (unsuccessful) firings
        for (int channel = 0; channel < _port.getWidthInside(); channel++) {
            while (_port.hasTokenInside(channel)) {
                _port.getInside(channel);
            }
        }
    }
        
    private IOPort _port;
    private Class _dataType = null;
    private Class _subcollectionType = null;
    private int _minimum;
    private int _maximum;
    private String _metadataKey = null;
    private boolean _bundlesTokens = false;
    
    private static final Pattern OUTPUT_PORT_NAME_PATTERN = 
          Pattern.compile("((\\p{Alnum}+)/)?(\\p{Alnum}+)([\\?\\+\\*])?(\\{(\\p{Digit}+)\\})?\\s*(\\[([^\\]]+)\\])?"); 

    private static final Pattern METADATA_KEY_PATTERN = 
        Pattern.compile("key\\s*=\\s*(\\p{Alnum}+)"); 
}