/* Actor that terminates workflow segment inside do-while construct.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors;

import org.nddp.CollectionHandler;
import org.nddp.CollectionPortBuffer;
import org.nddp.Collection;
import org.nddp.CollectionManager;
import org.nddp.CollectionIOPort;
import org.nddp.CollectionTypes;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.util.Parameters;

import ptolemy.data.BooleanToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.Variable;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="general"
 */
public class EndLoop extends CollectionTransformer {

	public EndLoop(CompositeEntity container, String name) 
		throws NameDuplicationException, IllegalActionException  {

    	    // call the superclass constructor
        	super(container, name);

        	loopback = CollectionIOPort.createOutputPort(this, "loopback");
		loopCollectionType = Parameters.stringParameter(this, 
		        "loopCollectionType", "");
		whileExpression = Parameters.stringParameter(this, 
		        "whileExpression", "true");
		maxLoopCount = Parameters.intParameter(this, "maxLoopCount", 1);
		exitOnException = Parameters.booleanParameter(this, 
		        "exitOnException", true);
		_whileExpression = new Variable(this, "_whileExpression");
		_whileExpression.setTypeEquals(BaseType.BOOLEAN);
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////
	
	public Parameter exitOnException;
   	public CollectionIOPort loopback;
	public Parameter loopCollectionType;
	public Parameter maxLoopCount;
	public Parameter whileExpression;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
	/** 
    	 * Initializes state of actor at start of workflow execution.
    	 */
	public void initialize() throws IllegalActionException {
 
	    // call superclass constructor
        	super.initialize();

    		_setInputPort(input);
        	_loopCollection = null;
        	_terminationToken = null;
        	_bufferedPort.clear();
    }  	
	
  	public boolean postfire() throws IllegalActionException {
  	    
  	    if (_terminationToken != null) {
  	        loopback.send(_terminationToken);
  	        _terminationToken = null;
  	    }
  	    
  	    return true;
  	}

	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

    public CollectionIOPort _getDefaultOutputPort() {	
		return output;
    }
    
    public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException {
	    
	    if (collectionManager.collection() == _loopCollection) {
	        
			_loopCount++;
	        	
	        	if (_continueLoopOk(collectionManager)) { 
	
	  	        _bufferedPort.flushToPort(loopback);
	  	        collectionManager.setOutputPort(loopback);
	        	
	    		} else { 
	
	  	        	_bufferedPort.flushToPort(output);
	  	        	collectionManager.setOutputPort(output);
	        	    	_terminateLoop();
	        	}
	    }
	}
   	
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) {
	    
	    	Collection collection = collectionManager.collection();
   	    
	    if (_loopCollection == null && 
            _loopCollectionType.isInstance(collection) && 
	        collection != _terminatedLoopCollection) {

	        _loopCollection = collection;
	        _loopCollectionParent = 
	            collectionManager.parentCollectionManager().collection();
	        _loopCount = 0;
	    }

    		if (_loopCollection != null) {
	        	collectionManager.setOutputPort(_bufferedPort);
    		}
	    
        return  CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
   	}
   	
    public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
  	    
		if (parameter == loopCollectionType) {
        	
		    _loopCollectionType = CollectionTypes.valueOfToken(value);
		
		} else if (parameter == whileExpression) {
		
		    _whileExpressionString = Parameters.stringValue(value);
		
		} else if (parameter == maxLoopCount) {
        		
		    _maxLoopCount = Parameters.intValue(value);
        		
        		if (_maxLoopCount < 1) {
        		    throw new IllegalActionException(
        		        "maxLoopCount must be greater than 0");
        		}
        		
		} else if (parameter == exitOnException) {
		 
		    _exitOnException = Parameters.booleanValue(value);
		    
        	} else {
        	    
			super._handleParameterChange(parameter, value);
        	}
    }

	///////////////////////////////////////////////////////////////////
	////                         private methods                   ////
    
   	private boolean _continueLoopOk(CollectionManager collectionManager) 
		throws IllegalActionException {

   	    	String expandedExpressionString = 
   	    	    collectionManager.expandMetadataSymbols(
   	    	            _whileExpressionString);
		_whileExpression.setExpression(expandedExpressionString);
     
		boolean expressionValue = 
		    ((BooleanToken)_whileExpression.getToken()).booleanValue();
        
        return expressionValue == _loopCount < _maxLoopCount && 
        		(!_exitOnException || !collectionManager.containsException());
   	}
   	
   	private void _terminateLoop() {
        	_terminationToken = 
        	    new LoopTerminationToken(_loopCollectionParent);
    		_terminatedLoopCollection = _loopCollection;
    		_loopCollection = null;
   	}  	    

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
	private CollectionPortBuffer _bufferedPort = new CollectionPortBuffer();
	private boolean _exitOnException;
	private Collection _loopCollection;
	private Collection _loopCollectionParent;
	private Class _loopCollectionType;  
	private int _loopCount;
	private int _maxLoopCount;
	private Collection _terminatedLoopCollection;
	private LoopTerminationToken _terminationToken;
	private Variable _whileExpression;
	private String _whileExpressionString;
}