/* Helper functions for reading and writing to an IOPort.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.util;

import ptolemy.actor.IOPort;
import ptolemy.actor.NoTokenException;
import ptolemy.data.ArrayToken;
import ptolemy.data.BooleanToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.util.IllegalActionException;

////////////////////////////////////////////////////////////////////////// 
//// Ports 

/** 
 @author Timothy M. McPhillips
 */

public class Port {

    // private constructor prevents instantation
    private Port() {}
    
    // determine if port is connected and has a token ready for reading
    public static boolean hasToken(IOPort port, int channel) throws IllegalActionException {
      
        return (port.getWidth() > channel && port.hasToken(channel));
    }
    
    public static boolean getBoolean(IOPort port, int channel) throws IllegalActionException {
        return getBoolean(port, channel, true, false);
    }

    public static boolean getBoolean(IOPort port, int channel, boolean defaultValue) throws IllegalActionException {
        return getBoolean(port, channel, false, defaultValue);        
    }
    
    private static boolean getBoolean(IOPort port, int channel, boolean mandatory, boolean defaultValue) 
        throws IllegalActionException {
        
        if (hasToken(port,channel)) {
            
            // receive one ObjectToken token on the taxonAbbreviations port            
            BooleanToken token = (BooleanToken) port.get(channel);
            
            // extract to the taxon abbreviations map
            return token.booleanValue();
            
        } else {
            
            if (mandatory) {
                throw new IllegalActionException("Required token not received on port " + port);
            } else {
                return defaultValue;
            }
        }        
    }

    // read an object token from the port and channel specified and return the contained object
    public static ObjectToken getObjectToken(IOPort port, int channel, boolean mandatory) 
        throws NoTokenException, IllegalActionException {

        if (hasToken(port,channel)) {
            
            // receive one ObjectToken token on the taxonAbbreviations port            
            ObjectToken token = (ObjectToken) port.get(channel);
            
            // extract to the taxon abbreviations map
            return token;
            
        } else {
            
            if (mandatory) {
                throw new IllegalActionException("Required token not received on port " + port);
            } else {
                return null;
            }
        }
    }
    
    // read an object token from the port and channel specified and return the contained object
    public static StringToken getStringToken(IOPort port, int channel, boolean mandatory) 
        throws NoTokenException, IllegalActionException {

        if (hasToken(port,channel)) {

            StringToken token = (StringToken) port.get(channel);
            
            return token;
            
        } else {
            
            if (mandatory) {
                throw new IllegalActionException("Required token not received on port " + port);
            } else {
                return null;
            }
        }
    }
    
    // read an object token from the port and channel specified and return the contained object
    public static String getString(IOPort port, int channel, boolean mandatory) 
        throws NoTokenException, IllegalActionException {

        String string = null;
        
        StringToken token = getStringToken(port, channel, mandatory);
        
        if (token != null) {
            string = token.stringValue();    
        }
        
        return string;
    }
    
    // read an object token from the port and channel specified and return the contained object
    public static Object getObject(IOPort port, int channel, boolean mandatory) 
        throws NoTokenException, IllegalActionException {

        Object object = null;
        
        ObjectToken token = getObjectToken(port, channel, mandatory);
        
        if (token != null) {
            object = token.getValue();
        }

        return object;
    }

    public static Token[] getTokenArray(IOPort port, int channel, boolean mandatory) throws IllegalActionException {
    
        if (hasToken(port,channel)) {

            // receive one ArrayToken on the treeArrayPort
            ArrayToken arrayToken = (ArrayToken)port.get(channel);
    
            // extract array of ObjectToken from ArrayToken
            Token[] tokenArray = (Token[])arrayToken.arrayValue(); 

            // return the token array
            return tokenArray;
            
        } else {
            
            if (mandatory) {
                throw new IllegalActionException("Required token not received on port " + port);
            } else {
                return null;
            }
        }
    }
}
