/* Class representing an environment for running an external process.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.util;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.Random;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;

import org.nddp.exceptions.ExternalEnvironmentException;

public class ProcessEnvironment {

    public ProcessEnvironment(String[] variableArray) {
        super();
        if (variableArray == null) {
            _variables = null;
        } else {
	        _variables = (String[]) variableArray.clone();
        }	        
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
    public void copyFile(String sourceFilePath, String destinationFileName) 
    		throws IOException {
        
	    File inputFile = new File(sourceFilePath);
        File outputFile = new File(_directory + "/" + 
			destinationFileName);

        FileReader in = new FileReader(inputFile);
        FileWriter out = new FileWriter(outputFile);
        int c;

        while ((c = in.read()) != -1)
           out.write(c);

        in.close();
        out.close();
    }

    public FileWriter createInputFile(String filename) 
    		throws ExternalEnvironmentException {
        
        if (_directory == null) {
            throw new ExternalEnvironmentException("No working directory");
        }
        
	    try {
		    File infile = new File(_directory, filename);
		    infile.createNewFile();
            return new FileWriter(infile);
	    } catch (IOException ex) {
	        throw new ExternalEnvironmentException(
                "Error writing input file " + filename);
	    }
    }
    
    public void createWorkingDirectory(String name) {
        String directoryName = name + Long.toHexString(_random.nextLong());
        _directory = new File(directoryName);
        _directory.mkdir();
    }
    
    public void deleteFile(String fileName) {
        
        File fileToDelete = new File(_directory + "/" + fileName);
        fileToDelete.delete();
    }

    public void deleteWorkingDirectory() {
        	if ( _directory != null ) {
			deleteDirectory(_directory);
        	}
        	_directory = null;
    }

    public String readFile(String fileName) 
    		throws ExternalEnvironmentException {

        File file;
        
		// open the file
        if ( _directory == null ) {
             file = new File(fileName);
        } else {
			file = new File(_directory, fileName);
        }
        
		BufferedReader bufferedReader;
		
        try {
            bufferedReader = new BufferedReader(new FileReader(file));
            
        } catch (FileNotFoundException e) {
            
           throw new ExternalEnvironmentException(
               "Error reading file " + fileName);
        }
        
        // TODO handle int overflow on conversion from long
		StringBuffer fileContents = new StringBuffer((int) file.length());
		
		try {
			// read file line by line, adding each line to the file 
		    // collection as a data token
			String line;
			while ( (line = bufferedReader.readLine()) != null) {
				fileContents.append(line + "\n");
			}
	
			// close the file
			bufferedReader.close();
		
		} catch (IOException ex) {
            throw new ExternalEnvironmentException(
            		"Error reading file " + fileName);
       	}
		
        return fileContents.toString();
    }
    

    
    public ExternalProcess startProcess(String command, Writer outputHandler, 
        Writer errorHandler) throws IOException {
        
        return new ExternalProcess(command, outputHandler, errorHandler, 
            _variables, _directory);
    }

    public String workingDirectoryPath() {
        return _directory.getAbsolutePath();
    }

    ///////////////////////////////////////////////////////////////////
	////                         private methods                   ////

    private static boolean deleteDirectory(File dir) {
        if (dir.isDirectory()) {
            String[] children = dir.list();
            for (int i = 0; i < children.length; i++) {
                boolean success = deleteDirectory(new File(dir, children[i]));
                if (!success) {
                    return false;
                }
            }
        }
    
        return dir.delete();
    }
    
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

    private File _directory = null;
    private static final Random _random = new Random();
    private final String[] _variables;
}
