/* an actor that execute a query on a database.

@Copyright (c) 1998-2003 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the
above copyright notice and the following two paragraphs appear in all
copies of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

                                                PT_COPYRIGHT_VERSION 2
                                                COPYRIGHTENDKEY
@ProposedRating Red (ellen_zh)
@AcceptedRating Red (ellen_zh)
*/

package org.sdm.spa;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.BooleanToken;
import ptolemy.data.IntToken;
import ptolemy.data.StringToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Workspace;

//////////////////////////////////////////////////////////////////////////
//// DatabaseWriter
/**
This actor reads a SQL update string from its input and execute the 
update on the database specified by the <i>databaseURL<i> parameter.

FIXME: I am convinced that we should pass the db connection around if 
we need more than one db actors to connect to the same database. (the
alternative is to combine DatabaseReader and DatabaseWriter to one class.
but this makes the actor interface a little nasty...

@author  Yang Zhao
@version $Id: DatabaseWriter.java,v 1.3 2005/11/01 20:39:14 ruland Exp $
*/


public class DatabaseWriter extends TypedAtomicActor {

    /** Construct an actor with the given workspace.
     *  @param workspace The workspace.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
       public DatabaseWriter(Workspace workspace)
               throws IllegalActionException, NameDuplicationException {
           super(workspace);

           // Set the type of the input and output port.
           updateResult = new TypedIOPort(this, "updateResult", false, true);
           updateResult.setTypeEquals(BaseType.INT);
           
           updateSQL = new TypedIOPort(this, "updateSQL", true, false);
           updateSQL.setTypeEquals(BaseType.STRING);

           databaseDriver = new StringParameter(this, "databaseDriver");
           databaseDriver.setExpression("sun.jdbc.odbc.JdbcOdbcDriver");
           databaseDriver.addChoice("com.ibm.db2.jcc.DB2Driver");
           databaseDriver.addChoice("oracle.jdbc.driver.OracleDriver");
           databaseDriver.addChoice("org.objectweb.rmijdbc.Driver");
           
           databaseURL = new Parameter(this, "databaseURL", 
                             new StringToken("jdbc:odbc:"));
           databaseURL.setTypeEquals(BaseType.STRING);
           
           username = new StringParameter(this, "username");
           username.setExpression("username");
           password = new StringParameter(this, "password");
           password.setExpression("this.is.not.secure,it.is.for.testing.only");
           //password.setPersistent(false);
           //new PasswordStyle(password, "style");


           refresh = new Parameter(this, "refresh", new BooleanToken(false));
           refresh.setTypeEquals(BaseType.BOOLEAN);
       }
       
    /** Construct an actor with the given container and name.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public DatabaseWriter(CompositeEntity container, String name)
            throws IllegalActionException, NameDuplicationException {
        super(container, name);

        // Set the type of the input and output port.
        updateResult = new TypedIOPort(this, "updateResult", false, true);
        updateResult.setTypeEquals(BaseType.INT);
        
        updateSQL = new TypedIOPort(this, "updateSQL", true, false);
        updateSQL.setTypeEquals(BaseType.STRING);
        
        databaseDriver = new StringParameter(this, "databaseDriver");
        databaseDriver.setExpression("sun.jdbc.odbc.JdbcOdbcDriver");
        databaseDriver.addChoice("com.ibm.db2.jcc.DB2Driver");
        databaseDriver.addChoice("oracle.jdbc.driver.OracleDriver");
        databaseDriver.addChoice("org.objectweb.rmijdbc.Driver");
           
        databaseURL = new Parameter(this, "databaseURL", 
                          new StringToken("jdbc:odbc:"));
        databaseURL.setTypeEquals(BaseType.STRING);
        
        username = new StringParameter(this, "username");
        username.setExpression("username");
        password = new StringParameter(this, "password");
        password.setExpression("this.is.not.secure,it.is.for.testing.only");
        //password.setPersistent(false);
        //new PasswordStyle(password, "style");

        refresh = new Parameter(this, "refresh", new BooleanToken(false));
        refresh.setTypeEquals(BaseType.BOOLEAN);
    }

    ///////////////////////////////////////////////////////////////////
    ////                     ports and parameters                  ////

    public StringParameter databaseDriver;
    /** The URL of the database to query from. This parameter contains
     *  a StringToken.
     */
    public Parameter databaseURL;
    
    public StringParameter username;
    public StringParameter password;

    /** The flag that indicates whether to reconnect to the database
     *  between each reading.
     *  This is a boolean, and defaults to false.
     */
    public Parameter refresh;
    
    /** The input port for the SQL update string.
     */
    public TypedIOPort updateSQL;
    
    /** The output port for the update result.
     */
    public TypedIOPort updateResult;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    /** If the specified attribute is <i>databaseURL</i>, update the 
     *  database's name to connect to. If the attribute is <i>refresh<i>
     *  then update the flag.
     *  @param attribute The attribute that has changed.
     *  @exception IllegalActionException If the base class throws it.
     */
    public void attributeChanged(Attribute attribute)
            throws IllegalActionException {
        if(attribute == databaseDriver) {
            _driver = ((StringToken)databaseDriver.getToken()).stringValue(); 
        }else if (attribute == databaseURL) {
            StringToken urlToken = null;
            urlToken = (StringToken)databaseURL.getToken();
            if (urlToken == null) {
                _database = null;
            } else {
                _database = urlToken.stringValue();
            }
        } else if(attribute == username) {
            _username = ((StringToken)username.getToken()).stringValue();    
        } else if (attribute == password) {
            _password = ((StringToken)password.getToken()).stringValue();   
        }else if (attribute == refresh) {
        
        
            _refreshFlag = ((BooleanToken)refresh.getToken()).booleanValue();
        }
        super.attributeChanged(attribute);
    }

    /** Clone the actor into the specified workspace.
     *  @param workspace The workspace for the new object.
     *  @return A new actor.
     *  @exception CloneNotSupportedException If a derived class contains
     *   an attribute that cannot be cloned.
     */
    public Object clone(Workspace workspace)
            throws CloneNotSupportedException {
        DatabaseWriter newObject = (DatabaseWriter)super.clone(workspace);
        return newObject;
    }
    
    /** Read a string token for the query from the input port, 
     *  execute it on the database and output the query result.
     *  @exception IllegalActionException If there is error to 
     *  execute the query or if the base class throw it.
     */
        public void fire() throws IllegalActionException {
            if (_refreshFlag) { //construct the connection here.
                try {
                    if (_connection != null) {
                        _connection.close();
                    }
                    _connection = DriverManager.getConnection (
                                  _database, _username, _password);
                } catch (Exception e) {
                    _debug("problems connecting to "+_database);
                    throw new IllegalActionException(this, e, 
                              "Failed to connect to the database");
                }            
            }
            //FIXME: if we get both update and query sql, should
            // we do update first or query first? here we do update
            //first...
            if (updateSQL.hasToken(0)) {
                
                String update = ((StringToken)updateSQL.get(0)).stringValue();

                Statement stmt = null;
                try {
                    stmt = _connection.createStatement();
                } catch (Exception e) {
                    _debug("problems connecting to "+_database);
                    throw new IllegalActionException(this, e, 
                              "Failed to connect to the database");
                }
                try {
                    int result = stmt.executeUpdate(update);
                    updateResult.send(0, new IntToken(result));
                } catch (SQLException e) {
                    throw new IllegalActionException(this, e, 
                    "failed to execute the query");
                }
            }
        }
    
    /** Load the dadabase driver. Connect to the database if the refresh
     *  flag is false.
     *  @exception IllegalActionException Not thrown in this base class
     */
    public void initialize() throws IllegalActionException {
        super.initialize();
        _connection = null;
        //attributeChanged(databaseURL);
        try {
            Class.forName(_driver);
        } catch (Exception e) {
            _debug("Failed to load JDBC/ODBC driver: " + e.getMessage());
            return;
        }
        if (!_refreshFlag) { //construct the connection once here.
            try {
                _connection = DriverManager.getConnection (
                              _database, _username, _password);
            } catch (Exception e) {
                _debug("problems connecting to "+_database + " :" + e.getMessage());
                throw new IllegalActionException(this, e, 
                          "Failed to connect to the database");
            }            
        }
    }

    /** Return true if there is input token to read.
     *  Otherwise, return false.
     *  @return True if the actor is ready to fire.
     *  @exception IllegalActionException If there is no director.
     */
    public boolean prefire() throws IllegalActionException {
        if (updateSQL.hasToken(0)) {
            return true;
        }
        else return false;
    }
    
    /** Close connection if there is one.
     *  @exception IllegalActionException If an IO error occurs.
     */
    public void wrapup() throws IllegalActionException {
        try {
            if (_connection != null) {
               _connection.close();
            }
        } catch (SQLException ex) {
            throw new IllegalActionException(this, ex, "Failed to close");
        }
    }

    ///////////////////////////////////////////////////////////////////
    ////                         protected methods                /////



    ///////////////////////////////////////////////////////////////////
    ////                         protected members                 ////

    // The connection to the database.
    private Connection _connection = null;

    private String _driver = " ";
    // String for the name of the database.
    private String _database = " ";
    
    private String _username = " ";
    
    private String _password = " ";
    
    // Flag to indicate whether or not to refresh the data between readings.
    private boolean _refreshFlag = false;
}
