/* ParameterizedGlobusJob actor gets the inouts and the program to run
   from the user, creates an RSL string from the inputs, submits it as
   a Globus job, and outputs the results.

 Copyright (c) 2000-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.
 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.
 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.
                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
 */

package org.sdm.spa;

import java.util.StringTokenizer;

import org.nmiworkflow.GridJob;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;

//////////////////////////////////////////////////////////////////////////
//// ParameterizedGlobusJob
/**
   This actor takes a Globus RSL and hostname and submits a job to that
   host. The output is gathered and sent to the output port.
   <p>
   The <i>rsl</i> parameter is the Globus RSL that defines the job to be run.
   The <i>host</i> parameter is the hostname where the job will be executed.
   I should add more parameters so the user does not need to specify a full
   rsl. Rsl is annoying.

   @author Ilkay Altintas (Build on top of Steve Mock's GlobusJob.)
   @version $Id: RunJobGridClient.java,v 1.3 2005/11/01 20:39:14 ruland Exp $
 */


public class RunJobGridClient extends TypedAtomicActor {

  public RunJobGridClient(CompositeEntity container, String name)
          throws IllegalActionException, NameDuplicationException {
      super(container, name);


      certificate = new TypedIOPort(this, "certificate", true, false);
      certificate.setTypeEquals(BaseType.STRING);
      new Attribute(certificate, "_showName");

      inputFiles = new TypedIOPort(this, "inputFiles", true, false);
      inputFiles.setTypeEquals(BaseType.STRING);
      new Attribute(inputFiles, "_showName");

      outputFiles = new TypedIOPort(this, "outputFiles", false, true);
      outputFiles.setTypeEquals(BaseType.STRING);
      new Attribute(outputFiles, "_showName");

      result = new TypedIOPort(this, "result", false, true);
      result.setTypeEquals(BaseType.STRING);
      new Attribute(result, "_showName");

      host = new StringAttribute(this, "GlobusHost");
      executable = new StringAttribute(this, "Program name");
      version = new StringAttribute(this, "Program version");
      queueName = new StringAttribute(this, "Queue name");
      numberOfProcessors = new StringAttribute(this, "Number of processors");

      _attachText("_iconDescription",
                  "<svg>\n"
                  + "<rect x=\"0\" y=\"0\" "
                  + "width=\"60\" height=\"30\" "
                  + "style=\"fill:blue; stroke:blue; stroke-width:2\"/>\n"
                  + "<text x=\"10\" y=\"20\""
                  + "style=\"font-size:12; fill:white; font-family:SansSerif\">"
                  + "Globus</text>\n"
                  + "</svg>\n");

  }

  ///////////////////////////////////////////////////////////////////
  ////                     ports and parameters                  ////
  /** The input port, which is a multiport.
   */
  public TypedIOPort certificate;
  public TypedIOPort inputFiles;
  public TypedIOPort outputFiles;
  public TypedIOPort result;

  //The host name to run the job on
  public StringAttribute host;

  //The path of the executable to be run
  public StringAttribute executable;
  //The version of the executable program
  public StringAttribute version;
  // ???
  public StringAttribute queueName;
  //The number of processors to execute the job
  public StringAttribute numberOfProcessors;


  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////

  /** If the specified attribute is <i>fileOrURL</i> and there is an
   *  open file being read, then close that file and open the new one;
   *  if the attribute is <i>numberOfLinesToSkip</i> and its value is
   *  negative, then throw an exception.  In the case of <i>fileOrURL</i>,
   *  do nothing if the file name is the same as the previous value of
   *  this attribute.
   *  @param attribute The attribute that has changed.
   *  @exception IllegalActionException If the specified attribute
   *   is <i>fileOrURL</i> and the file cannot be opened, or the previously
   *   opened file cannot be closed; or if the attribute is
   *   <i>numberOfLinesToSkip</i> and its value is negative.
   */
  public void attributeChanged(Attribute attribute)
          throws IllegalActionException {

      super.attributeChanged(attribute);
  }

  /** Output the data lines into an array.
   *  @exception IllegalActionException If there's no director.
   */

  public void fire() throws IllegalActionException {
      super.fire();

      //get the proxy string fed into the input port
      Token token = null;
      try {
        token = certificate.get(0);
        if (token != null) {
          _proxy = token.toString();
        }
      }
      catch (Exception ex) {
        _proxy = "";
        _debug("Proxy is null.");
      }

      String hostString = host.getExpression();
      _debug("<HOSTNAME>" + hostString + "</HOSTNAME>");
      String exeString  = executable.getExpression();
      _debug("<EXECUTABLE>" + exeString + "</EXECUTABLE>");
      String versionString  = version.getExpression();
      _debug("<VERSION>" + versionString + "</VERSION>");
      String queueNameString  = queueName.getExpression();
      _debug("<QUEUE_NAME>" + queueNameString + "</QUEUE_NAME>");
      String numberOfProcString  = numberOfProcessors.getExpression();
      _debug("<NUMBER_OF_PROCESSORS>" + numberOfProcString + "</NUMBER_OF_PROCESSORS>");

      String argString = "(arguments=";
      // Number of processors is set to 1 by default.
      if (numberOfProcString != null) {
        argString += "-n " + numberOfProcString;
      } else {
        argString += "-n 1";
      }
      //argString += "
      argString += ")";

      /* Consume the input file token if there's one.
       */
      Token tokenFiles = null;
      String filesStream = null;
      try {
        tokenFiles = inputFiles.get(0);
        if (tokenFiles != null) {
          filesStream = new String(tokenFiles.toString());
          _debug("<INPUT_FILES>" + filesStream + "</INPUT_FILES>");
          filesStream = filesStream.substring(1, filesStream.length() - 1);
        }
        //Tokenize the filesStream to get the name of each file.
        StringTokenizer st = new StringTokenizer(filesStream, ";", false);
        while (st.hasMoreTokens()) {
          _debug("<FILE_PATH>" + st.nextToken() + "</FILE_PATH>");
        }
      }
      catch (Exception ex) {
        _debug("Input files string is null.");
      }

      String rslString = "&(executable=" + exeString + ")" + argString;
      _debug("<RSL>" + rslString + "</RSL>");

      _job = new GridJob(hostString, false);
      _result = _job.GlobusRun(rslString, _proxy);
      _result += _job.getOut();

      result.broadcast(new StringToken(_result));
      outputFiles.broadcast(new StringToken("/rmount/paci/sdsc/altintas/TEMP/test_actor.out;/rmount/paci/sdsc/altintas/TEMP/temp.txt;/rmount/paci/sdsc/altintas/TEMP/tmp.xml;/rmount/paci/sdsc/altintas/TEMP/temp.html"));
      /*
       * FIX ME: The output files will be converted to output the paths
       *         of the result output files.
       */

  }

  /** ...
   */
  public String getResult() {
      return _result;
  }

  /** Post fire the actor. Return false to indicated that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   */
  public boolean postfire() {
      return false;
  }

  /** Open the file or URL and read the first line, and use the
   *  first line to set the type of the output.
   *  @exception IllegalActionException If the file or URL cannot be
   *  opened, or if the first line cannot be read.
   */
  public void preinitialize() throws IllegalActionException {
      super.preinitialize();
  }

  ///////////////////////////////////////////////////////////////////
  ////                         protected members                 ////


  ///////////////////////////////////////////////////////////////////
  ////                         private methods                   ////


  ///////////////////////////////////////////////////////////////////
  ////                         private members                   ////

  private GridJob _job;
  private String _proxy;

  /** Result string Variable. */
  private String _result = new String("");

}
