/* A Web Service Actor to invoke web services
 * To invoke a web service following parameters
 * are needed.
 * -URL
 * -NameSpace
 * -Method
 * -User
 * -Password
 *
 * We are adding a new Trigger to the Web Service Adapter to
 * notify its completion.
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.sdm.spa;

import java.net.URL;

import javax.xml.namespace.QName;
import javax.xml.rpc.ParameterMode;

import org.apache.axis.client.Call;
import org.apache.axis.client.Service;
import org.apache.axis.encoding.XMLType;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.BooleanToken;
import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;
public class WebServiceActor extends TypedAtomicActor {

	public WebServiceActor(CompositeEntity aContainer, String aName)
		throws IllegalActionException, NameDuplicationException {

		super(aContainer, aName);
		result = new TypedIOPort(this, "result", false, true);
		result.setTypeEquals(BaseType.STRING);
		input = new TypedIOPort(this, "input", true, false);
		input.setTypeEquals(BaseType.STRING);

		trigger = new TypedIOPort(this, "trigger", false, true);
		trigger.setTypeEquals(BaseType.BOOLEAN);

		nameSpace = new StringAttribute(this, "nameSpace");
		locationUrl = new StringAttribute(this, "locationUrl");
		userName = new StringAttribute(this, "userName");
		password = new StringAttribute(this, "password");
		method = new StringAttribute(this, "method");
		paramName = new StringAttribute(this, "paramName");
		//input = new StringAttribute(this, "input");
	}

	///////////////////////////////////////////////////////////////////
	////                     ports and parameters                  ////

	// -- Part of the Actor

	public TypedIOPort result;
	public TypedIOPort input;
	public TypedIOPort trigger;

	// -- Public interface of the Actor
	public StringAttribute nameSpace;
	public StringAttribute locationUrl;
	public StringAttribute userName;
	public StringAttribute password;
	public StringAttribute method;
	public StringAttribute paramName;
	//public StringAttribute input;

	///////////////////////////////////////////////////////////////////
	////                         public methods                    ////

	/** Initialize the PN actor.
	 *  @exception IllegalActionException If the parent class throws it.
	 */
	public void initialize() throws IllegalActionException {
		super.initialize();
		_returnValue = true;

	}

	public void attributeChanged(Attribute attribute)
		throws IllegalActionException {
		if (attribute == nameSpace) {
			p_nameSpace = nameSpace.getExpression();

		} else if (attribute == locationUrl) {
			p_locationUrl = locationUrl.getExpression();

		} else if (attribute == userName) {
			p_userName = userName.getExpression();

		} else if (attribute == password) {
			p_password = password.getExpression();

		} else if (attribute == method) {
			p_method = method.getExpression();

		} else if (attribute == paramName) {
			p_paramName = paramName.getExpression();

		} else {
			super.attributeChanged(attribute);
		}
	}

	/** Fire the actor.
	 *  @exception IllegalActionException If there is no director.
	 */
	public void fire() throws IllegalActionException {
		String md = p_nameSpace + p_method + p_locationUrl;
		p_input = ((StringToken) (input.get(0))).stringValue();

		String res;
		if (toCache) {
			res = Util.getFromCache(md, p_input);
			_debug(" Broadcast from WSActor: Got Result from Cache:" + res);
			if (res != null) {
				result.broadcast(new StringToken(res));
				_reachedEND = true;
				return;
			}
		}

		try {
			Service service = new Service();
			Call call = (Call) service.createCall();
			URL url = new URL(p_locationUrl);
			call.setTargetEndpointAddress(url);
			call.setOperationName(new QName(p_nameSpace, p_method));
			call.addParameter(
				p_paramName,
				XMLType.XSD_STRING,
				ParameterMode.IN);
			call.setReturnType(XMLType.XSD_STRING);
			call.setUsername(p_userName);
			call.setPassword(p_password);
			call.setTimeout(new Integer(0));
			_debug(
				"Invoking:"
					+ p_method
					+ "("
					+ p_paramName
					+ ":"
					+ p_input
					+ ")"
					+ p_userName
					+ ":"
					+ p_password);
			Object ret = call.invoke(new Object[] { p_input });

			if (ret instanceof String) {
				res = (String) ret;
				_debug(" Broadcast from WSActor: Got Results:" + res);
				if (toCache) {
					Util.saveToCache(md,p_input,res)	;
				}

				result.broadcast(new StringToken(res));
			} else {
				_debug(" Broadcast from WSActor: Got No Result.");
			}

		} catch (Exception e) {
			_debug(" Broadcast Error from WSActor:" + e.getMessage());
			e.printStackTrace();
		}

		_reachedEND = true;
	}

	/** Post fire the actor. Return false to indicated that the
	 *  process has finished. If it returns true, the process will
	 *  continue indefinitely.
	 */
	public boolean postfire() throws IllegalActionException {
		if (_reachedEND) {
			trigger.broadcast(BooleanToken.TRUE);
			return true;
		}
		//endOfLoop.broadcast(BooleanToken.FALSE);
		return true;
	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	// -- Get private copies of the parameters
	private String p_nameSpace;
	private String p_locationUrl;
	private String p_userName;
	private String p_password;
	private String p_method;
	private String p_input;
	private String p_paramName;
	private boolean _returnValue = true;
	private boolean _reachedEND = false;
	private boolean toCache = Util.toCache;

}
