/**
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.srb;

import ptolemy.actor.NoTokenException;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.ArrayToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

import java.util.Vector;

import edu.sdsc.grid.io.MetaDataRecordList;
import edu.sdsc.grid.io.local.LocalFile;
import edu.sdsc.grid.io.srb.SRBFile;
import edu.sdsc.grid.io.srb.SRBFileSystem;
import edu.sdsc.grid.io.srb.SRBMetaDataSet;

//////////////////////////////////////////////////////////////////////////
//// SGetPhysicalLocation
/**
   Get the physical location of SRB files.

   @author Efrat Jaeger
   @version $Id: SGetPhysicalLocation.java,v 1.8 2005/09/17 00:21:52 brooke Exp $
   @category.name srb
   @category.name put

*/

public class SGetPhysicalLocation extends TypedAtomicActor {

    /** Construct a constant source with the given container and name.
     *  Create the <i>value</i> parameter, initialize its value to
     *  the default value of an IntToken with value 1.
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the entity cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public SGetPhysicalLocation(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException {
        super(container, name);

        SRBFileSystem = new TypedIOPort(this, "SRBFileSystem", true, false);
        SRBFileSystem.setTypeEquals(BaseType.GENERAL);
        new Attribute(SRBFileSystem, "_showName");

        logicalPath = new TypedIOPort(this, "logicalPath", true, false);
        logicalPath.setTypeEquals(new ArrayType(BaseType.STRING));
        new Attribute(logicalPath, "_showName");

        physicalPath = new TypedIOPort(this, "physicalPath", false, true);
        physicalPath.setTypeEquals(new ArrayType(BaseType.STRING));
        new Attribute(physicalPath, "_showName");

/*	    exitCode = new TypedIOPort(this, "exitCode", false, true);
        exitCode.setTypeEquals(BaseType.STRING);
        new Attribute(exitCode, "_showName");
*/
        _attachText("_iconDescription",
                "<svg>\n"
                + "<rect x=\"0\" y=\"0\" "
                + "width=\"158\" height=\"30\" "
                + "style=\"fill:white\"/>\n"
                + "<text x=\"7\" y=\"24\" "
                   + "style=\"font-size:12; fill:black; font-family:SansSerif\">"
                   + "SRB$</text>\n"
                + "<text x=\"41\" y=\"25\" "
                + "style=\"font-size:14; fill:blue; font-family:SansSerif\">"
                + "Physical Location</text>\n"
                + "</svg>\n");
    }

    public TypedIOPort SRBFileSystem;
    public TypedIOPort logicalPath;
    public TypedIOPort physicalPath;
//	public TypedIOPort exitCode;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
        /** Get the physical location of SRB logical file paths.
         *  @exception IllegalActionException If it is thrown if the SRB
         *  file cannot be accessed or the current directory cannot be
         *  broadcasted.
         */
    public void fire() throws IllegalActionException {

        SRBFile srbFile;
        LocalFile localFile;
        String localFilePath;
        String _exitCode = "";

        try {
//       	 make sure there is an alive connection.
            try {
                srbFileSystem.getHost();
            } catch (Exception ex) { // connection was closed.
                srbFileSystem = null;
                ObjectToken SRBConOT = null;
                try { // try to get a new connection in case the previous one has terminated.
                    SRBConOT = (ObjectToken) SRBFileSystem.get(0);
                } catch (NoTokenException ntex) {}
                if (SRBConOT != null) {
                    srbFileSystem = (SRBFileSystem) SRBConOT.getValue();
                }
            }
            if (srbFileSystem != null) {

                // Getting the list of file to query for logical path.
                ArrayToken logFilesTokenArr = null;
                try {
                    logFilesTokenArr = (ArrayToken) logicalPath.get(0);
                } catch (Exception ex) {
                    _debug("logFilesTokenArr port is null.");
                }
                if (logFilesTokenArr != null) {
                    Token[] logFilesToken = logFilesTokenArr.arrayValue();
                    Vector physicalLocVec = new Vector();
                    for (int i=0; i<logFilesToken.length; i++) {

                        // srb file path.
                        String logFileStr = ((StringToken) logFilesToken[i]).stringValue();
                        _debug("<FILE_TO_PUT>" + logFileStr + "<FILE_TO_PUT>");

                        srbFile = new SRBFile(srbFileSystem, logFileStr);

                        if (srbFile.exists()) {
                            MetaDataRecordList[] record = srbFile.query(SRBMetaDataSet.PATH_NAME);

                            // extract the physical location
                            String physicalLocation = record[0].getStringValue(0);
                            physicalLocVec.add(new StringToken(physicalLocation));
                        }
                        else _exitCode += srbFile.getAbsolutePath() + "does not exist.\n";
                    }
                    if (physicalLocVec.size() > 0) {
                        Token[] physicalPathArr = new Token[physicalLocVec.size()];
                        physicalLocVec.toArray(physicalPathArr);
                        physicalPath.broadcast(new ArrayToken(physicalPathArr));
                    }
                    if (_exitCode.equals("")) {
                        _exitCode = "success";
                    }
    //        		exitCode.broadcast(new StringToken(_exitCode));
                }
            } else throw new IllegalActionException(this,
                    "No SRB connection available in actor " + this.getName() + ".");
        } catch (Exception ex) {
            srbFile = null;
            srbFileSystem = SRBUtil.closeConnection(srbFileSystem);
            throw new IllegalActionException(this, ex.getMessage() + ". in actor " + this.getName() + ".");
        }
    }

    /** Initialize the srb file system to null.
     */
    public void initialize() throws IllegalActionException {
        super.initialize();
        srbFileSystem = null;
    }

    /** Disconnect from SRB.
     */
    public void wrapup() {
        srbFileSystem = SRBUtil.closeConnection(srbFileSystem);
    }

    private SRBFileSystem srbFileSystem = null;
}
