/* A listener that is for provenance logging purposes.

Copyright (c) 1998-2004 The Regents of the University of California.
All rights reserved.
Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

PT_COPYRIGHT_VERSION_2
COPYRIGHTENDKEY

*/

package ptolemy.actor;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.text.DateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import ptolemy.actor.CompositeActor;
import ptolemy.actor.ModelDependencyGraph;
import ptolemy.data.Token;
import ptolemy.data.expr.StringParameter;
import ptolemy.graph.DirectedGraph;
import ptolemy.graph.Element;
import ptolemy.gui.ComponentDialog;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.AbstractSettableAttribute;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.ChangeListener;
import ptolemy.kernel.util.ChangeRequest;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Nameable;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Settable;
import ptolemy.kernel.util.ValueListener;
import ptolemy.kernel.util.Workspace;

//////////////////////////////////////////////////////////////////////////
//// ProvenanceExecutionListener

/**
   Our initial version of the provenance execution listener.  It has
   a good portion of the functionality described on:
   http://kepler-project.org/Wiki.jsp?page=KeplerProvenanceFramework
   Note that this is still a work in progress.
   Also I am not sure extending AbstractSettableAttribute is appropriate but
   it allows us to leave Manager.java unmodified.

   @author Oscar Barney
   @version $Id: ProvenanceExecutionListener.java,v 1.6 2006/01/24 00:04:30 barney Exp $
*/
public class ProvenanceExecutionListener extends AbstractSettableAttribute
	              implements ExecutionListener, ChangeListener, TokenSentListener {
    /** Construct a provenance execution listener in the default
     *  workspace with an empty string as its name. The listener
     *  is added to the list of objects in the workspace. Increment the
     *  version number of the workspace.
     */
    public ProvenanceExecutionListener()
            throws IllegalActionException, NameDuplicationException {
        super();
        _addIcon();
        _init();
    }

    /** Construct a provenance execution listener in the workspace
     *  with an empty name.  The listener is added to the list of objects
     *  in the workspace.  Increment the version number of the workspace.
     *
     *  @param workspace The workspace for this object.
     */
    public ProvenanceExecutionListener(Workspace workspace) 
            throws IllegalActionException, NameDuplicationException {
        super(workspace);
        _addIcon();
        _init();
    }

    /** Construct a provenance execution listener in the given
     *  container with the given name.  The container argument must
     *  not be null, or a NullPointerException will be thrown.
     *  If the name argument is null, then the name is set to the
     *  empty string. Increment the version number of the workspace.
     *
     *  @param container Container of the director.
     *  @param name Name of this listener.
     *  @exception IllegalActionException If the listener is not compatible
     *   with the specified container.  May be thrown in a derived class.
     *  @exception NameDuplicationException If the container is not a
     *   CompositeActor and the name collides with an entity in the container.
     */
	public ProvenanceExecutionListener(CompositeEntity container, String name)
            throws IllegalActionException, NameDuplicationException {
        super(container, name);
        _addIcon();
        _init();
    }

    ///////////////////////////////////////////////////////////////////
    ////                         parameters                        ////

    /** A parameter to hold the name of the workflow user.
     */
    public StringParameter ranBy;

    /** A parameter to hold the name of the institution the
     *  user works for.
     */
    public StringParameter institutionName;

    /** A parameter to label the name of this run or experiment.
     */
    public StringParameter runName;

    /** A parameter to keep track of the date.
     */
    public StringParameter date;

    /** A parameter to set the level of detail we want to save.
     */
    public StringParameter levelOfDetail;

    /** A parameter that allows the user to specify the desired
     *   output format.
     */
    public StringParameter outputFormat;

    /** A parameter to set the destination of the saved information.
     */
    public StringParameter cacheDestination;

    /** A parameter to set the path for saved files.
     */
    public StringParameter fileSavePath;

    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////

    //The following 6 functions are used to implement AbstractSettableAttribute.
    //This class is implemented because of the function validate which seems
    //to be the only time that the manager lets attributes know that it is
    //about to run the model.  Without using this function changes to the 
    //manager would be needed to let the ProvenanceExecutionListener know
    //that the model is about to fire and that it should register with the
    //appropriate listeners.  NOTE: Implementing AbstractSettableAttribute
    //is not the cleanest solution because the ProvenanceExecutionListener
    //really is not "settable" but it does allow us to minimaze the changes 
    //to Ptolemy II.
    public void addValueListener(ValueListener listener) {
        //do nothing
        //System.out.println("addValList Called");
    }

    public String getExpression() {
        //System.out.println("getExp Called");
        return "not really a settable attribute";
    }

    public Settable.Visibility getVisibility() {
        //System.out.println("getVis Called");
        return Settable.NONE;
    }

    public void removeValueListener(ValueListener Listener) {
        //do nothing
        //System.out.println("remValList Called");
    }

    public void setVisibility(Settable.Visibility visibility) {
        //do nothing
        //System.out.println("setVis Called");
    }

    //This function is called when the manager is about to run the model
    //and allows us to register with the manager before the run happens.
    //We would do this in the constructor but it seems that when the object
    //is created there is no manager yet.
    public void validate() {
        initialize();

        CompositeActor topLevel = (CompositeActor)getContainer();
        //System.out.println("CONT: " + topLevel);
        Manager manager = topLevel.getManager();
        //System.out.println("MAN: " + manager);

        if(!_registered){
            manager.addExecutionListener(this);
            manager.addChangeListener(this);

            Collection nodes = _graph.nodes();
            Iterator portIterator = nodes.iterator();
            while(portIterator.hasNext()) {
                IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
                if(port.isOutput()) {
                    port.addTokenSentListener(this);
                }
            }
            _registered = true;
        }
    }

    /** Override the base class to check validity of parameter entries.
     * 
     *  @param attribute The attribute that changed.
     *  @exception IllegalActionException If the function is not recognized.
     */
    public void attributeChanged(Attribute attribute)
            throws  IllegalActionException {
        String temp = "";
        if (attribute == levelOfDetail) {
            temp = levelOfDetail.stringValue();
            if (!temp.equals("Verbose-all") && !temp.equals("Verbose-some")
                    && !temp.equals("Medium") && !temp.equals("Error Log")) {
                throw new IllegalActionException(this,
                        "Unrecognized level of detail: " + temp);
            }
        } else if(attribute == outputFormat) {
            temp = outputFormat.stringValue();
            if (!temp.equals("text") && !temp.equals("HTML")
                    && !temp.equals("XML")) {
                throw new IllegalActionException(this,
                        "Unrecognized output format: " + temp);
            }
        } else if(attribute == cacheDestination) {
            temp = cacheDestination.stringValue();
            if (!temp.equals("NO CACHE") && !temp.equals("localhost")
                    && !temp.equals("To Database") && !temp.equals("To SRB")) {
                throw new IllegalActionException(this,
                        "Unrecognized cache destination: " + temp);
            }
        } else {
            super.attributeChanged(attribute);
        }
    }

    /** Report that an actor is done firing and how long it took to
     *  complete.  In SDF this event is now used but in PN it doesnt make 
     *  as much sense because the threads are sometimes blocked so 
     *  the total time would not really correspond to the amount of work
     *  done.
     *
     *  @param actor The actor that is now done.
     *  @param millis The time the actor took to fire.
     */
    public void actorDoneFiring(Actor actor, long millis) {
        String actorName = ((Nameable)actor).getFullName();
        if(_format.equals("text")) {
            _errorLog.println("Actor: " + actorName + " is done firing\n\t" +
                    actorName + " took " + ((double)millis)/1000 + " seconds to complete");
        } else if (_format.equals("HTML")) {
            _errorLog.println("\t<p>");
            _errorLog.println("\t\tActor: <b>" + actorName + 
                    "</b> is done firing\n\t\t<br> \n\t\tIt took " + ((double)millis)/1000 + 
                    " seconds to complete");
            _errorLog.println("\t</p>");
        } else if (_format.equals("XML")) {
            _errorLog.println("\t<Event type=\"Actor_event\">Actor: " + 
                    actorName + " is done firing.  Time: " + ((double)millis)/1000 + 
                            " seconds.</Event>");
        } else {
            throw new InternalErrorException("Uncaught unrecognized format error");
        }
    }

    /** Report that an actor is about to iterate.  We put this information
     *  in the error execution log.
     * 
     *  @param actor The actor that is now done.
     *  @param iterationCount The number of time the actor will fire.
     */
    public void actorAboutToFire(Actor actor, int iterationCount) {
        if(_format.equals("text")) {
            _errorLog.println("Actor: " + ((Nameable)actor).getFullName() + 
                    " is about to fire " + iterationCount + " time(s).");
        } else if (_format.equals("HTML")) {
            _errorLog.println("\t<p>");
            _errorLog.println("\t\tActor: <b>" + ((Nameable)actor).getFullName() + 
                    "</b> is about to fire <b>" + iterationCount + "</b> time(s).");
            _errorLog.println("\t</p>");
        } else if (_format.equals("XML")) {
            _errorLog.println("\t<Event type=\"Actor_event\">Actor: " + 
                    ((Nameable)actor).getFullName() + " is about to fire " + 
                    iterationCount + " times.</Event>");
        } else {
            throw new InternalErrorException("Uncaught unrecognized format error");
        }
        if(_dataLog != null) {
            if(_format.equals("text")) {
                _dataLog.println("Actor: " + ((Nameable)actor).getFullName() 
                            + " is about to fire " + iterationCount + " time(s).");
            } else if (_format.equals("HTML")) {
                _dataLog.println("\t<p>");
                _dataLog.println("\t\tActor: <b>" + ((Nameable)actor).getFullName() 
                            + "</b> is about to fire <b>" + iterationCount + "</b> time(s).");
                _dataLog.println("\t</p>");
            } else if (_format.equals("XML")) {
                _dataLog.println("\t\t</Iteration>");
                _dataLog.println("\t</Actor>");
                //TODO get read of the two things before this if you are the first actor
                _dataLog.println("\t<Actor name=\"" + ((Nameable)actor).getFullName() + "\">");
                _dataLog.println("\t\t<Params>");
                _dataLog.println("\t\t\t<Parameter name=\"name\">value</Parameter>");
                _dataLog.println("\t\t\t............");
                _dataLog.println("\t\t</Params>");
                _dataLog.println("\t\t<Iteration number=\"" + iterationCount + "\">");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
        }
    }

    /** React to a change request has been successfully executed.
     *  This method is called after a change request
     *  has been executed successfully.
     *  
     *  @param change The change that has been executed, or null if
     *   the change was not done via a ChangeRequest.
     */
    public void changeExecuted(ChangeRequest change) {
        //update graph for every graph change not EVERY change
        if(change.getSource().toString().indexOf("ActorGraph") != -1) {
                _graph = ModelDependencyGraph.generateDependencyGraph
                    ((CompositeActor)_container);
        }
        //TODO if we are firing do we want to make a new log when change is made?
        if(_firing) {
            if(_format.equals("text")) {
                _errorLog.println("Succesful change while execution in progress: " + 
                        change.getDescription());
                _errorLog.println("\nChange made by: " + change.getSource());
            } else if (_format.equals("HTML")) {
                _errorLog.println("\t<p>\n");
                _errorLog.println("\t\tSuccesful change while execution in progress: " + 
                        change.getDescription());
                _errorLog.println("\t\t \n<br>\n");
                _errorLog.println("\t\tChange made by: " + change.getSource() + "\n");
                _errorLog.println("\t</p>\n");
            } else if (_format.equals("XML")) {
                _errorLog.println("<Change source=\"" + change.getSource() +"\">");
                _errorLog.println(change.getDescription() + "</Change>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }

            //System.out.println("Succesful change while execution in progress: " 
            //        + change.getDescription());
            //System.out.println("Change made by: " + change.getSource());
        }        
    }

    /** React to a change request has resulted in an exception.
     *  This method is called after a change request was executed,
     *  but during the execution an exception was thrown.
     *  
     *  @param change The change that was attempted or null if
     *   the change was not done via a ChangeRequest.
     *  @param exception The exception that resulted.
     */
    public void changeFailed(ChangeRequest change, Exception exception) {
        if (_errorLog != null) {
            if(_format.equals("text")) {
                _errorLog.println("Change failed: " + change.getDescription());
                _errorLog.println("\n Change made by: " + change.getSource());
            } else if (_format.equals("HTML")) {
                _errorLog.println("\t<p>\n");
                _errorLog.println("\t\tChange failed: " + change.getDescription());
                _errorLog.println("\t\t \n<br>\n");
                _errorLog.println("\t\tChange made by: " + change.getSource() + "\n");
                _errorLog.println("\t</p>\n");
            } else if (_format.equals("XML")) {
                _errorLog.println("<Change_failed source=\"" + change.getSource() +"\">");
                _errorLog.println(change.getDescription() + "</Change_failed>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
        }
    }

    /** Get the Provenance listener ready to record the provenance
     *  information for the next run.  This function is called by the manager.  
     */
    public void initialize() { 
        _graph = ModelDependencyGraph.generateDependencyGraph
                ((CompositeActor)_container);
//TODO is this try still needed?
        try{
            _format = outputFormat.stringValue();
            _owner = ranBy.stringValue();
            _institution = institutionName.stringValue();
            _note = runName.stringValue();
            _date = date.stringValue();
            _detail = levelOfDetail.stringValue();
        } catch (IllegalActionException ex){
            throw new InternalErrorException(ex);
        }
        _updateLogOutput();
    
    }

    /** An actor could call this function to record in the log that it
     *  has done something with an external file.
     *  
     *  @param file The file that the action was done to.
     *  @param action The action done to the file.
     *  @param actor The actor doing the action.
     *  @param location The location of this file.
     */
    public void externalFileEvent
            (String file, String action, Actor actor, String location) {
        if(_dataLog != null) {
            if(_format.equals("text")) {
                _dataLog.println("FILE CREATED:\n" + file + " was " + action + 
                        " by " + actor + " to " + location);
                _dataLog.println("at time: " + new Date() + "\n");
            } else if (_format.equals("HTML")) {
                _dataLog.println("\t<p>");
                _dataLog.println(file + " was " + action + " by " + actor + " to " + location);
                _dataLog.println("\t\t<br>\n\t\tat time: " + new Date());
                _dataLog.println("\t</p>");
            } else if (_format.equals("XML")) {
                _dataLog.println("\t<Event type=\"File Event\">" + 
                        file + " was " + action + " by " + actor + " to " + location);
                _dataLog.println("\t<Time>" + new Date() + "</Time>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
        }
    }

    //TODO i think that we no longer need the if error log != null now that
    //we have seperate the smart reruns stuff from this file!

    //TODO get read of the multiple print statements in favor of one big one
    // with \n added for new line... will save function calls
    /** Report an execution failure.   This method will be called
     *  when an exception or error is caught by a manager.
     *  Exceptions are reported this way when the run() or startRun()
     *  methods of the manager are used to perform the execution.
     *  If instead the execute() method is used, then exceptions are
     *  not caught, and are instead just passed up to the caller of
     *  the execute() method.  Those exceptions are not reported
     *  here (unless, of course, the caller of the execute() method does
     *  so).
     *
     *  @param manager The manager controlling the execution.
     *  @param throwable The throwable to report.
     */
    public void executionError(Manager manager, Throwable throwable) {
        if(_errorLog != null && _registered) {
            if(_format.equals("text")) {
                _errorLog.println("Execution error:");
                throwable.printStackTrace(_errorLog);
                _errorLog.println("at time: " + new Date());
            } else if (_format.equals("HTML")) {
                _errorLog.print("\t<h2 align=\"center\"><font size=\"3\">");
                _errorLog.println("Execution error:</font> </h3>\n\t<p>");
                throwable.printStackTrace(_errorLog);
                _errorLog.println("\t\t<br>\n\t\tat time: " + new Date());
                _errorLog.println("\t</p>");
            } else if (_format.equals("XML")) {
                _errorLog.print("\t<ERROR> \n\t");
                throwable.printStackTrace(_errorLog);
                _errorLog.println("\t</ERROR>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
        }
    }

//TODO record how long it took as well!
    /** Report that the current execution has finished and
     *  the wrapup sequence has completed normally. The number of successfully
     *  completed iterations can be obtained by calling getIterationCount()
     *  on the manager.
     *
     *  @param manager The manager controlling the execution.
     */
    public void executionFinished(Manager manager) {
        if(_errorLog != null && _registered) {
            if(_format.equals("text")) {
                _errorLog.print("Completed execution with "
                    + manager.getIterationCount() + " iterations");
                _errorLog.println(" at time: " + new Date() + "\n");
            } else if (_format.equals("HTML")) {
                _errorLog.println("\t<p>");
                _errorLog.println("\t\tCompleted execution with "
                    + manager.getIterationCount() + " iterations");
                _errorLog.println("\t\t<br>\n\t\tat time: " + new Date());
                _errorLog.println("\t</p>");
            } else if (_format.equals("XML")) {
                _errorLog.println(
                        "\t<Event type=\"Execution_finished\">Completed execution with "
                        + manager.getIterationCount() + " iterations</Event>");
                _errorLog.println("\t<Time>" + new Date() + "</Time>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
            if(_dataLog != null) {
                if(_format.equals("text")) {
                    _dataLog.print("=> Completed execution with "
                        + manager.getIterationCount() + " iterations");
                    _dataLog.println("at time: " + new Date() + "\n");
                } else if (_format.equals("HTML")) {
                    _dataLog.println("\t<p>");
                    _dataLog.println("\t\tCompleted execution with "
                        + manager.getIterationCount() + " iterations");
                    _dataLog.println("\t\t<br>\n\t\tat time: " + new Date());
                    _dataLog.println("\t</p>");
                } else if (_format.equals("XML")) {
                    _dataLog.println(
                            "\t<Event type=\"Execution_finished\">Completed execution with "
                            + manager.getIterationCount() + " iterations");
                    _dataLog.println("\t<Time>" + new Date() + "</Time>");
                } else {
                    throw new InternalErrorException("Uncaught unrecognized format error");
                }
            }
        }
    }

    /** Report that the manager has changed state.
     *  To access the new state, use the getState() method of Manager.
     *
     *  @param manager The manager controlling the execution.
     *  @see Manager#getState()
     */
    public void managerStateChanged(Manager manager) {
        Manager.State state = manager.getState();

        if (state == Manager.PREINITIALIZING) {
            CompositeActor topLevel = (CompositeActor)getContainer();
            //if the container of this is null then it has been removed and we should unregister.
            if(topLevel == null){
                //manager.removeExecutionListener(this);
                //dont do this anymore because of concurent modification exception. Let it unregister itself
                manager.removeChangeListener(this);
                _registered = false;
                //Unregister all ports because this listener has been removed.
                Collection nodes = _graph.nodes();
                Iterator portIterator = nodes.iterator();
                while(portIterator.hasNext()) {
                    IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
                    if(port.isOutput()) {
                        port.removeTokenSentListener(this);
                    }
                }
            }
        }

        //if there is an error log print to it
        if(_errorLog != null && _registered) {
            if (state == Manager.ITERATING) {
                _firing = true;
                String message = state.getDescription() + " iteration number " + 
                        manager.getIterationCount();
                if(_format.equals("text")) {
                    _errorLog.println("Manager event: " + message);
                    _errorLog.println("at time: " + new Date());
                } else if (_format.equals("HTML")) {
                    _errorLog.println("\t<p>");
                    _errorLog.println("\t\tManager event: " + message);
                    _errorLog.println("\t\t<br>\n\t\tat time: " + new Date());
                    _errorLog.println("\t</p>");
                } else if (_format.equals("XML")) {
                    _errorLog.println("\t<Event type=\"Manager_event\">" + 
                            state.getDescription() + "</Event>");
                } else {
                    throw new InternalErrorException("Uncaught unrecognized format error");
                }
                
                if(_dataLog != null && _format.equals("XML")) {
                    _dataLog.println("<RUN>");
                }
            } else {
                if (state == Manager.WRAPPING_UP) {
                    _firing = false;  //to keep track of changes made when paused!
                    if(_dataLog != null){
                        if(_format.equals("XML")){
                            _dataLog.println("</RUN>");
                        } else if(_format.equals("HTML")){
                            _dataLog.println("\t<p>\n\t\t<b>Done with run</b>\n\t</p>");
                        }
                    }
                }

                if(_format.equals("text")) {
                    _errorLog.println("Manager event: " + state.getDescription());
                } else if (_format.equals("HTML")) {
                    _errorLog.println("\t<p>");
                    _errorLog.println("\t\tManager event: " + state.getDescription());
                    _errorLog.println("\t</p>");
                } else if (_format.equals("XML")) {
                    _errorLog.println("\t<Event type=\"Manager_event\">" + 
                            state.getDescription() + "</Event>");
                } else {
                    throw new InternalErrorException("Uncaught unrecognized format error");
                }
            }
        }
    } 

    /** This function is the implementation of a TokenSentListener.  IOPort
     *  will call this function when a token is sent so we can store it.
     *  
     *  @param event The event representing the token(s) sent.
     */
    public void tokenSentEvent(TokenSentEvent event) {
        IOPort port = event.getPort();
        int channel = event.getChannel();
        Token token = event.getToken();
        Token[] tokenArray = event.getTokenArray();
        int vectorLength = event.getVectorLength();
        
        //TODO make level name a constant thing
        
        //If the detail level is Verbose-some check to see if this token 
        //should be saved.
        if(!_detail.equals("Verbose-some") || (_detail.equals("Verbose-some") &&
                _saveStuffForActors.contains(port.getContainer().getFullName())) ){
            if(vectorLength != -1) {
                if(channel != -1) {
                    _tokenArraySent(channel, tokenArray, vectorLength, port);
                } else {
                    //find channels because it is a broadcast
                    Receiver[][] farReceivers;
                    farReceivers = port.getRemoteReceivers();
                    for (int i = 0; i < farReceivers.length; i++) {
                        _tokenArraySent(i, tokenArray, vectorLength, port);
                    }
                }
            } else {
                //only one token was sent
                if(channel != -1) {
                    _tokenSent(channel, token, port);
                } else {
                    //find channels because it is a broadcast
                    Receiver[][] farReceivers;
                    farReceivers = port.getRemoteReceivers();
                    for (int i = 0; i < farReceivers.length; i++) {
                        _tokenSent(i, token, port);
                    }
                }
            }
         }
    }
    
    //TODO init in init?
    /** The key is a string representing the workflow instance
     *  and the value is a PrintStream. The idea is that we will
     *  have 
     */
    protected HashMap _wfStates = new HashMap();
    protected HashMap _dataStates = new HashMap();
    protected HashMap _errorStates = new HashMap();
    //TODO could combine all of these into one map and change the keys slightly for each type.
    protected HashMap _dataTokenStates = new HashMap();  //for the file that will contain the printed out tokens
    protected HashMap _dataFileNames = new HashMap();  //so we can get the name of the file back so we can put it in the HTML correctly

    ///////////////////////////////////////////////////////////////////
    ////                         private variables                 ////
    
    //A map of actor names to true false values to indicate
    //which actors you would like to save information for in 
    //verbose some mode.
    private HashMap _actorVerboseSomeMap = new HashMap();
    
    //Stream variables to hold the current ouput streams for each type
    //of provenance output.
    private PrintStream _dataLog;
    private PrintStream _wfLog;
    private PrintStream _errorLog;

    //The output format.
    private String _format;

    //Who ran the workflow.
    private String _owner;

    //Institution this workflow is being run by.
    private String _institution;

    //A note about this workflow.
    private String _note;
    
    //Date this workflow is being run.
    private String _date;

    //Level of detail that we will be saving information at.
    private String _detail;

    private NamedObj _container;

    //The graph representing the model we are running.
    private DirectedGraph _graph;

    //Actors that we will save information for in verbose some mode.
    private HashSet _saveStuffForActors = new HashSet();
    
    //A counter that makes sure we dont give two files that hold tokens
    //the same name.
    private int _tokenLinkNo; 
    
    //Where to put files generated to hold tokens seperately;
    private String _dataLogPath;
    
    //A flag to tell the provenance execution listener if it is 
    //registered with the manager or not.
    private boolean _registered;
    
    //A flag to keep track of whether or not manager is now firing
    private boolean _firing;

    ///////////////////////////////////////////////////////////////////
    ////                      private methods                    //////
	 
    /**  Add an XML graphic as a hint to UIs for rendering the director.
     */
    private void _addIcon() {
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"-50\" y=\"-15\" "
                + "width=\"100\" height=\"30\" "
                + "style=\"fill:blue\"/>\n" +
                "</svg>\n");
    }

    /**  Function to create provenance file headers.
     */ 
    private String _fileHeader() {
        String output = "";
        if(_format.equals("text")) {
            output = "Ran By: " + _owner + "\n";
            output += "Institution: " + _institution + "\n";
            output += "Run Name/Note: " + _note + "\n";
            output += "Date: " + _date + "\n"; 
        } else if (_format.equals("HTML")) {
            output += "</head>\n<body>\n";
            output += "\t<h2 align=\"center\"><font size=\"5\">" + _note + "</font> </h2>\n";
            output += "\t<h3 align=\"center\"><font size=\"4\"> By " + _owner + " at " + _institution  + " </font> </h3>\n";
            output += "\t<h4 align=\"center\"><font size=\"3\"> " + _date + " </font> </h4>\n";
        } else if (_format.equals("XML")) {
            output = "<element User_Name=\"" + _owner + "\"/>\n";
            output += "<element  User_Institution=\"" + _institution + "\"/>\n";
            output += "<element Run_Name=\"" + _note + "\"/>\n";
            output += "<element Time_Date=\"" + _date + "\"/>\n"; 
        } else {
            throw new InternalErrorException("Uncaught unrecognized format error");
        }
        return output;
    }

    /** Initialize the provenance execution listener. 
     */
    private void _init()
            throws IllegalActionException, NameDuplicationException {

        Workspace workspace = workspace();
        _container = getContainer();

        //TODO make this enable caching for individual actors        
        //enableCacheing = new Parameter(this, "Enable Cacheing");
        //enableCacheing.setTypeEquals(BaseType.BOOLEAN);
        //enableCacheing.setExpression("false");

        ranBy = new StringParameter(this, "Your Name");
        ranBy.setExpression("");

        institutionName = new StringParameter(this, "Your Institution");
        institutionName.setExpression("");

        runName = new StringParameter(this, "Name or Note to associate with the run");
        runName.setExpression("");

        Date d = new Date();
        date = new StringParameter(this, "Date to associate with the run");
        date.setExpression(d.toString());

        levelOfDetail = new StringParameter(this, "Level of Detail");
        levelOfDetail.setExpression("Error Log");
        levelOfDetail.addChoice("Verbose-all");
        levelOfDetail.addChoice("Verbose-some");
        levelOfDetail.addChoice("Medium");
        levelOfDetail.addChoice("Error Log");

        outputFormat = new StringParameter(this, "Output Format");
        outputFormat.setExpression("text");
        outputFormat.addChoice("text");
        outputFormat.addChoice("HTML");
        outputFormat.addChoice("XML");

        //TODO how to put stuff in .kepler under $HOME? when we select no cache or localhost
        cacheDestination = new StringParameter(this, "Cache Destination");
        cacheDestination.setExpression("localhost");
        cacheDestination.addChoice("NO CACHE");
        cacheDestination.addChoice("localhost");
        cacheDestination.addChoice("To Database");
        cacheDestination.addChoice("To SRB");

        fileSavePath = new StringParameter(this, "Path to saved files (put a / on the end)");
        fileSavePath.setExpression("/tmp/");

        _graph = null;
        _registered = false;
        //so that we dont name the files that hold token outputs the same thing
        _tokenLinkNo = 0;
        _dataLogPath = "";
    }

    /**  Used when we create a link to another page that displays
     *  the contents of large tokens.
     */ 
    private String _getTokenLinkNo() {
        _tokenLinkNo++;
        return Integer.toString(_tokenLinkNo);
    }
    
    /**  Used to output the model in different formats.
     */ 
    public String _outputCompositeActorsContents(NamedObj container) {
        if(_format.equals("text")) {
            return _outputCompositeActorsContentsToString(container, 0);
        } else if (_format.equals("HTML")) {
            return _outputCompositeActorsContentsToHTML(container, 0);
        } else if (_format.equals("XML")) {
            //save the moml
            return container.exportMoML(container.getName());
        } else {
            return "unknown format";
        }
    }

    /**  Output the model contents in string format.
     */
    public String _outputCompositeActorsContentsToString(NamedObj container, int level) {
        String output = new String();
        String paramString = "";
        String paramName = "";
        Iterator obj = container.containedObjectsIterator();
        Object t;
        if(level == 0) {
            output += _fileHeader() + "\n";
        }
        while(obj.hasNext()) {
            t = obj.next();
            if (t instanceof AbstractSettableAttribute) {
                paramString = t.toString();
                paramName = ((NamedObj)t).getFullName();
                if (paramName.lastIndexOf("vergil") == -1 && 
                        paramName.lastIndexOf("_windowProperties") == -1 &&
                        (level >= 0 || paramName.lastIndexOf("_doc") == -1)) {
                    for(int i = 0; i < level; i++) {
                        output += "\t";
                    }
                    output += "Parameter: " + paramString + "\n";
                }
            } else if (t instanceof CompositeActor) {
                for(int i = 0; i < level; i++) {
                    output += "\t";
                }
                output += "Composite Actor: "  + t.toString() + "\n" 
                    +	_outputCompositeActorsContentsToString((NamedObj)t, level+1);
            } else if (t instanceof Actor) {
                for(int i = 0; i < level; i++) {
                    output += "\t";
                }
                output += "Actor: "  + t.toString() + "\n";
                //do we want expert parameters? how to filter these out?
                List result = ((NamedObj)t).attributeList(AbstractSettableAttribute.class);
                Iterator params = result.iterator();
                while(params.hasNext()) {
                    NamedObj param = (NamedObj)params.next();
                    paramString = param.toString();
                    paramName = param.getFullName();
                    if (paramName.lastIndexOf("_windowProperties") == -1 &&
                        (level >= 0 || paramName.lastIndexOf("_doc") == -1)) {
                        for(int i = 0; i < level+1; i++) {
                            output += "\t";
                        }
                        output += "* " +paramString + "\n";
                    }
                }
            } else if (t instanceof Director) {
                for(int i = 0; i < level; i++) {
                    output += "\t";
                }
                output += "Director: " + t.toString() + "\n";
                //do we want expert parameters? how to filter these out?
                List result = ((NamedObj)t).attributeList(AbstractSettableAttribute.class);
                Iterator params = result.iterator();
                while(params.hasNext()) {
                    for(int i = 0; i < level+1; i++){
                        output += "\t";
                    }
                    output += "* " + params.next().toString() + "\n";
                }
            }
        }
        return output;		 
    }

    /**  Output the model contents in HTML format.
     */ 
    public String _outputCompositeActorsContentsToHTML(NamedObj container, int level) {
        String output = new String();
        Iterator obj = container.containedObjectsIterator();
        Object t;
        if(level == 0){
            output += _fileHeader();
        }     
        for(int i = 0; i < level+1; i++) {
            output += "\t";
        }
        output += "<ul>\n";
        while(obj.hasNext()) {
            t = obj.next();
            if (t instanceof AbstractSettableAttribute) {
                String paramString = t.toString();
                if (paramString.lastIndexOf("vergil") == -1 && 
                        paramString.lastIndexOf("_windowProperties") == -1) {
                    for(int i = 0; i < level+2; i++){
                        output += "\t";
                    }
                    output += "<li>Parameter: " + paramString + "</li>\n";
                }
            } else if (t instanceof CompositeActor) {
                for(int i = 0; i < level+2; i++) {
                    output += "\t";
                }
                output += "<li>Composite Actor:  "  + t.toString() + "</li>\n" 
                    +   _outputCompositeActorsContentsToHTML((NamedObj)t, level+1);
            } else if (t instanceof Actor){
                for(int i = 0; i < level+2; i++){
                    output += "\t";
                }
                output += "<li>Actor:  "  + t.toString() + "</li>\n";
                //do we want expert parameters? how to filter these out?
                List result = ((NamedObj)t).attributeList(AbstractSettableAttribute.class);
                Iterator params = result.iterator();
                for(int i = 0; i < level+2; i++){
                    output += "\t";
                }
                output += "<ul>\n";
                while(params.hasNext()) {
                    String paramString = params.next().toString();
                    if (paramString.lastIndexOf("_windowProperties") == -1) {
                        for(int i = 0; i < level+3; i++) {
                            output += "\t";
                        }
                        output += "<li>" + paramString + "</li>\n";
                    }
                }
                for(int i = 0; i < level+2; i++) {
                    output += "\t";
                }
                output += "</ul>\n";
            } else if (t instanceof Director) {
                for(int i = 0; i < level+1; i++) {
                    output += "\t";
                }
                output += "\t<li>Director: " + t.toString() + "</li>\n";
                //do we want expert parameters? how to filter these out?
                List result = ((NamedObj)t).attributeList(AbstractSettableAttribute.class);
                Iterator params = result.iterator();
                for(int i = 0; i < level+2; i++) {
                    output += "\t";
                }
                output += "<ul>\n";
                while(params.hasNext()) {
                    for(int i = 0; i < level+3; i++){
                        output += "\t";
                    }
                    output += "<li>" + params.next().toString() + "</li>\n";
                }
                for(int i = 0; i < level+2; i++) {
                    output += "\t";
                }
                output += "</ul>\n";
            }
        }
        for(int i = 0; i < level+1; i++) {
            output += "\t";
        }
        output += "</ul>\n";
        return output;       
    }

    /**  Output the graph that represents this model.
     */ 
    private String _outputGraph() {
        if(_graph == null) {
            return "no graph";
        }

        if(_format.equals("text")) {
            return "\nGraph representation of the workflow: \n" + _graph.toString();
        } else if (_format.equals("HTML")) {
            return "\n" + _outputGraphToHTML();
        } else if (_format.equals("XML")) {
            //save the moml
            return "";  //because moml already has its own graph like thing
        } else {
            return "unknown graph format";
        }
    }

    /**  Output the graph that represents this model to HTML.
     */
    private String _outputGraphToHTML() {
        if(_graph == null){
            return "no graph";
        }

        String result = "\t<h3 align=\"left\"><font size=\"4\">";
        result += "Graph representation of the workflow: </font> </h3>\n";

        //now add the nodes
        result += "\t<h2 align=\"left\"><font size=\"3\">Nodes:</font> </h3>\n"; 
        Collection list = _graph.nodes();
        Iterator itr = list.iterator();
        result += "\t<ol>\n";
        while(itr.hasNext()) {
            result += "\t\t<li> " + itr.next() + " </li>\n";
        }
        result += "\t</ol>\n";
        //now add the edges
        result += "\t<h2 align=\"left\"><font size=\"3\">Edges:</font> </h3>\n";
        list = _graph.edges();
        itr = list.iterator();
        result += "\t<ol>\n";
        while(itr.hasNext()) {
            result += "\t\t<li> " + itr.next() + " </li>\n";
        }
        result += "\t</ol>\n";
        return result;
    }

    /**  Output the title for the Error/Exec log.
     */
    private String _printErrorTitle() {
        if(_format.equals("text")) {
            return "Error/Execution Log";
        } else if (_format.equals("HTML")) {
            return "<html>\n<head>\n\t<title>Error/Execution Log</title>";
        } else if (_format.equals("XML")) {
            return "<File_type>Error/Execution Log</File_type>";
        } else {
            return "unknown format";
        }
    }

    /**  Output the title for the WF log.
     */
    private String _printWFTitle(){
        if(_format.equals("text")) {
            return "Workflow Log\n";
        } else if (_format.equals("HTML")) {
            return "<html>\n<head>\n\t<title>Workflow Log</title>\n";
        } else if (_format.equals("XML")) {
            //do nothing because we have the moml title
            return "";
        } else {
           return "unknown format\n";
        }
    }

    /** Report that an array of tokens was sent from fromPort
     *  on the channel indicated.
     *
     *  @param channelIndex The channel the token was sent on.
     *  @param tokens The tokens that were sent.
     *  @param vectorLength The number of tokens from the array
     *                  that were actually sent.
     *  @param fromPort The token was sent from this port.
     */
    private void _tokenArraySent(int channelIndex, Token[] tokens, 
            int vectorLength, IOPort fromPort) {
        for(int i = 0; i < vectorLength; i++) {
            _tokenSent(channelIndex, tokens[i], fromPort);
        }
    }

    /** Report that a token was sent from fromPort on the
     *  channel indicated.
     *
     *  @param channelIndex The channel the token was sent on.
     *  @param token The token that was sent.
     *  @param fromPort The token was sent from this port.
     */
    private void _tokenSent(int channelIndex, Token token, 
                                         IOPort fromPort) {
         if(_dataLog != null) {
            if(_format.equals("text")) {
                _dataLog.println("Token: " + token); 
                _dataLog.println("sent from port " + fromPort  + " on channel " + 
                        channelIndex + "\n");
            } else if (_format.equals("HTML")) {
                _dataLog.println("\t<p>");
                //TODO if token has a large toString make a link to a page that has the output on that token on it
                String temp = token.toString();
                String[] tempParts = temp.split("\n");
                if(temp.length() < 150 && tempParts.length < 5) {
                    _dataLog.println("\t\tToken: <b>");
                    for(int i = 0; i < tempParts.length; i++) {
                        _dataLog.println("\t\t<br> \n\t\t" + tempParts[i]);
                    }
                     _dataLog.println("\t\t</b>");
                } else {
                    PrintStream log = (PrintStream) _dataTokenStates.get(_dataLog);
                    String fileName = ((String)_dataFileNames.get(_dataLog)).
                            replaceAll(".html$", "_TOKENS.html");
                    if(log == null) {
                        try {
                            //file that holds data such as tokens sent to each actor
                            File tokenFile = new File(fileName);
                            if(!tokenFile.createNewFile()) {
                                //TODO throw appropriate ptolemy error
                                System.err.println("could not create a token log file.  may already exist.");
                            } else {
                                log = new PrintStream(new FileOutputStream(tokenFile));
                                _dataTokenStates.put(_dataLog, log);
                                tokenFile.deleteOnExit();  //TODO remove this after testing
                                log.println(_printWFTitle() + _fileHeader() + 
                                    "\n\t<h3 align=\"center\"><font size=\"3\">Individual Token Log</font> </h3>");
                            }
                        } catch (IOException e) {
                            System.err.println("An error occured trying to create the log file.\n" + e);
                            //TODO: throw appropriate ptolemy error here
                        }
                    }
                    String num = _getTokenLinkNo();
                    log.println("\t<p>");
                    log.println("\t\t<a name =\"" + num + "\"></a>");
                    log.println("\t\t<br> \n\t\tLink No: <b>" + num + "</b>");
                    log.println("\t\t<br> \n\t\tToken: <b>");
                    //TODO maybe deal with tabs as well???
                    for(int i = 0; i < tempParts.length; i++) {
                        log.println("\t\t<br> \n\t\t" + tempParts[i]);
                    }
                     log.println("\t\t</b>");
                     log.println("\t\t<br> \n\t\tsent from port <b>" + fromPort  + "</b> on channel <b>" 
                                    + channelIndex +"</b>");
                    log.println("\t</p>");
                    
                    _dataLog.println("\t\t<br> \n\t\tLink No: <b>" + num + "</b>"); 
                    //figure out the length of the label
                    int labelLength = temp.length();
                    if(labelLength > 50) {
                        labelLength = 50;
                    }                        
                    _dataLog.println("\t\tToken: <b><a href=\"" + fileName + "#" + num + "\" >" + 
                            temp.substring(0, labelLength) + "...</a></b>");
                }
                _dataLog.println("\t\t <br>\n\t\tsent from port <b>" + fromPort  + 
                        "</b> on channel <b>" + channelIndex + "</b>");
                _dataLog.println("\t</p>");
            } else if (_format.equals("XML")) {
                //NOTE:  didnt do the <Output_Port name="...> thing because this way you get to 
                //see the order things were sent.  and it was way easier
                //TODO make sure this does something reasonable for complex tokens
                _dataLog.print("\t\t\t\t<Token_Sent type=\"" + token.getType() + "\"");
                _dataLog.println(" fromPort=\"" + fromPort + "\" channel=\"" + 
                        channelIndex + ">");
                _dataLog.println("\t\t\t\t\t" + token);
                _dataLog.println("\t\t\t\t</Token_Sent>");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
        }
    }
    
    /**  This function creates and maintains the log files that we have
     *  created.  When you run a model that is different from any that have
     *  been run before it will create a new file to output this information to.
     *  If you then change back so that the model is identical to one that has
     *  been run before this function, when called in initialize, will then set the
     *  streams to output to the files that were created when this model was
     *  run the first time.  This function also controls how many logs we are
     *  going to write to (just error or error and wf, etc).
     */
    private void _updateLogOutput() {
        //find current state
        //tell it level -1 so it gives us an accurate state w/o _doc forexample
        String state = _outputCompositeActorsContentsToString(_container,-1);
        //using graph of ports for state here is good because we can now tell if 
        //an actor adds extra ports.  if actors as nodes we wont notice this
        if(_graph != null) {
            state = state + "\n" + _graph.toString();
        }

        try {
            File dataFile;   //file that holds data such as tokens sent to each actor
            File wfFile;   //file that holds the workflow specification
            File errorFile;  //file that holds the error and execution log
            DateFormat df = DateFormat.getDateInstance(DateFormat.SHORT);
            DateFormat tf = DateFormat.getTimeInstance(DateFormat.LONG);
            Date d = new Date();                
            String name = _note+"_"+_owner+"_";
            name +=  df.format(d) + "_" + tf.format(d);
            name = name.replaceAll(" ","");
            name = name.replaceAll("/","-");
            
            //TODO... for local host this is suppose tobe .kepler or something like that
            //TODO set path based on the destinat
            String path = "";
            try {
                path = fileSavePath.stringValue();
            } catch (IllegalActionException ex) {
                throw new InternalErrorException(ex);
            }
            if(_format.equals("text")) {
                dataFile = new File(path + name + "_data.txt");
                wfFile = new File(path + name + "_wf.txt");
                errorFile = new File(path + name + "_error-exec.txt");
            } else if (_format.equals("HTML")) {
                dataFile = new File(path + name + "_data.html");
                wfFile = new File(path + name + "_wf.html");
                errorFile = new File(path + name + "_error-exec.html");
            } else if (_format.equals("XML")) {
                dataFile = new File(path + name + "_data.xml");
                wfFile = new File(path + name + "_wf.xml");
                errorFile = new File(path + name + "_error-exec.xml");
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }

            if(_detail.equals("Verbose-all") || _detail.equals("Verbose-some")) {
                //output to all three files
                //compare that state with what we have already seen
                _wfLog = (PrintStream) _wfStates.get(state);
                if(_wfLog == null) {
                    if(!wfFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create wf log file.  may already exist.");
                    } else {
                        _wfLog = new PrintStream(new FileOutputStream(wfFile));
                        _wfStates.put(state, _wfLog);
                        _wfLog.print(_printWFTitle());
                        //write parameters and associated graph to wf file    
                        _wfLog.println(_outputCompositeActorsContents(_container) + _outputGraph());
                    }
                }
                _dataLog = (PrintStream) _dataStates.get(state);

                if(_dataLog == null) {
                    if(!dataFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create data log file.  may already exist.");
                    } else{
                        _dataLog = new PrintStream(new FileOutputStream(dataFile));
                        _dataStates.put(state, _dataLog);
                        if(_format.equals("text")) {
                            _dataLog.println("Data Log");
                        } else if (_format.equals("HTML")) {
                            _dataFileNames.put(_dataLog, dataFile.getAbsolutePath());  //save the file names of the data log for html
                            _dataLog.println("<html>\n<head>\n\t<title>Data Log</title>");
                        } else if (_format.equals("XML")) {
                            _dataLog.println("<File_type>Data Log</File_type>");
                        } else {
                            //throw error... shouldnt ever get here
                        }
                        _dataLog.println(_fileHeader());
                        
                    }
                }
                _errorLog = (PrintStream) _errorStates.get(state);

                if(_errorLog == null) {
                    if(!errorFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create error log file: " + errorFile + ".  May already exist.");
                    } else {
                        _errorLog = new PrintStream(new FileOutputStream(errorFile));
                        _errorStates.put(state, _errorLog);
                        _errorLog.println(_printErrorTitle());
                        _errorLog.println(_fileHeader());
                    }
                }

                if (_detail.equals("Verbose-some")) {
                    Collection nodes = _graph.nodes();
                    Iterator portIterator = nodes.iterator();
                    while(portIterator.hasNext()) {
                        IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
                        String actorName = port.getContainer().getFullName();
                        if(port.isOutput()) {
                            if(!_actorVerboseSomeMap.containsKey(actorName)){
                                _actorVerboseSomeMap.put(actorName, new Boolean(false));
                            }
                        }
                    }
                        
                    NameTrueFalseQuery panel = new NameTrueFalseQuery(_actorVerboseSomeMap, "Save provenance for:");
                    
                    ComponentDialog dialog = new ComponentDialog(null,
                                    "Select actors", panel);
                
                    //next we put all nonCacheable actors in this set for later use
                    _saveStuffForActors = new HashSet();
                    //also we will save the boolean values for next time the menu is displayed
                    Iterator actorIterator = _actorVerboseSomeMap.keySet().iterator();
                    while(actorIterator.hasNext()){
                        String actorName = (String)actorIterator.next();
                        Boolean temp = new Boolean(panel.getBooleanValue(actorName));
                        _actorVerboseSomeMap.put(actorName, temp);
                        if(temp.booleanValue() == true){
                            _saveStuffForActors.add(actorName);
                        }    
                    }
                    
                    //TODO if they press cancel do something intelligent
                    if (!(dialog.buttonPressed().equals("OK"))) {
                        // Restore original parameter values?
                        panel.restore();
                    }
                }
            } else if (_detail.equals("Medium")) {
                //compare that state with what we have already seen
                _wfLog = (PrintStream) _wfStates.get(state);
                if(_wfLog == null){
                     if(!wfFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create wf log file.  may already exist.");
                    } else {
                        _wfLog = new PrintStream(new FileOutputStream(wfFile));
                        _wfStates.put(state, _wfLog);
                        _wfLog.print(_printWFTitle());
                        //write parameters and associated graph to wf file    
                        _wfLog.println(_outputCompositeActorsContents(_container) + _outputGraph());
                    }
                }
                _dataLog = null;
                _errorLog = (PrintStream) _errorStates.get(state);
                if(_errorLog == null) {
                    if(!errorFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create error log file.  may already exist.");
                    } else {
                        _errorLog = new PrintStream(new FileOutputStream(errorFile));
                        _errorStates.put(state, _errorLog);
                        _errorLog.println(_printErrorTitle());
                        _errorLog.println(_fileHeader());
                    }
                }
            } else if (_detail.equals("Error Log")) {
                //compare that state with what we have already seen
                _wfLog = null;
                _dataLog = null;
                _errorLog = (PrintStream) _errorStates.get(state);
                if(_errorLog == null) {
                    if(!errorFile.createNewFile()) {
                        //TODO throw appropriate ptolemy error
                        System.err.println("could not create error log file: " + errorFile + ".  May already exist.");
                    } else {
                        _errorLog = new PrintStream(new FileOutputStream(errorFile));
                        _errorStates.put(state, _errorLog);
                        _errorLog.println(_printErrorTitle());
                        _errorLog.println(_fileHeader());
                    }
                }
            } else {
                throw new InternalErrorException("Uncaught unrecognized format error");
            }
            //TODO delete when done testing or add option to do this
            wfFile.deleteOnExit();
            dataFile.deleteOnExit();
            errorFile.deleteOnExit();
        } catch (IOException e) {
            System.err.println("An error occured trying to create the log file.");
            //TODO: throw appropriate ptolemy error here
        } 
    }

}
