/**
 *    '$RCSfile: EcogridFactoryAuthClient.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2007-09-07 18:02:37 $'
 *   '$Revision: 1.2 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2007 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.ecogrid.client;

import java.net.URL;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.stub.EcoGridAuthInterfaceLevelOnePortType;
import org.ecoinformatics.ecogrid.stub.EcogridAuthLoginRequestElementType;
import org.ecoinformatics.ecogrid.stub.service.EcoGridAuthInterfaceLevelOneServiceGridLocator;
import org.ecoinformatics.ecogrid.stub.service.EcoGridQueryLevelOneServiceGridLocator;
import org.globus.ogsa.utils.GridServiceFactory;
import org.gridforum.ogsi.Factory;
import org.gridforum.ogsi.LocatorType;
import org.gridforum.ogsi.OGSIServiceGridLocator;

/**
 *      @author     tao 
 *      This class is the client which can access ecogrid
 *      authentication web service. The web service type should be
 *      factory-instance pattern. There are two main methods in this class -
 *      login_action and logout_action login_action will return a session id and
 *      logout_action will close the session
 */
public class EcogridFactoryAuthClient
{
  private URL serviceURL = null;
  private EcoGridAuthInterfaceLevelOnePortType ecogrid = null;

  /**
   *  The main program for the EcogridFactoryAuthClient class
   *
   *@param  args  The command line arguments
   */
  public static void main(String[] args)
  {
    String opMode = "";
    String userName = "";
    String passWord = "";
    String sessionId = "";
    String gsh_str = "";
    String retStr;

    if(args.length <= 0)
    {
      PrintAuthUsage();
      System.exit(0);
    }
    else
    {
      opMode = args[0];

      if(opMode.compareToIgnoreCase("login") == 0)
      {
        if(args.length != 4)
        {
          PrintAuthUsage();
          System.exit(0);
        }
        userName = args[1];
        passWord = args[2];
        gsh_str = args[3];
      }
      else if(opMode.compareToIgnoreCase("logout") == 0)
      {
        if(args.length != 3)
        {
          PrintAuthUsage();
          System.exit(0);
        }
        sessionId = args[1];
        gsh_str = args[2];
      }
      else
      {
        PrintAuthUsage();
        System.exit(0);
      }
    }

    try
    {
      URL GSH = new URL(gsh_str);

      // create a client object
      EcogridFactoryAuthClient client = new EcogridFactoryAuthClient(GSH);

      if(opMode.compareToIgnoreCase("login") == 0)
      {
        client.login_action(userName, passWord);
      }
      else if(opMode.compareToIgnoreCase("logout") == 0)
      {
        client.logout_action(sessionId);
      }
    }
    catch(Exception e)
    {
      // System.out.println("Exception in main : "+ e.getMessage());
      e.printStackTrace();
    }

  }
  //main

  /**
   *  Constructor of client. It will create the web service instance from
   *  factory then create the porttype. The port type will be called by other
   *  public method
   *
   *@param  serviceGSH
   *@exception  Exception  Description of the Exception
   */
  public EcogridFactoryAuthClient(URL serviceGSH)
    throws Exception
  {
    //this.serviceURL = serviceGSH;
    EcogridUtils.setDebug(true);
    // Get a reference to the EcoGridQueryLevelOneService Factory
    long startFactory = System.currentTimeMillis();
    OGSIServiceGridLocator gridLocator = new
        OGSIServiceGridLocator();
    Factory factory = gridLocator.getFactoryPort(serviceGSH);
    GridServiceFactory ecogridQueryFactory = new
        GridServiceFactory(factory);
    long endFacotry = System.currentTimeMillis();
    EcogridUtils.debugMessage("The time to create factory is ============= " +
        (endFacotry - startFactory) / 1000, 30);
    // Create a new EcoGridQueryLevelOneService instance and get a reference
    // to its getEcoGridQueryLevelOneServicePort
    LocatorType locator = ecogridQueryFactory.createService();
    EcoGridAuthInterfaceLevelOneServiceGridLocator loc = new
        EcoGridAuthInterfaceLevelOneServiceGridLocator();
    ecogrid = loc.getEcoGridAuthInterfaceLevelOnePort(locator);
    org.apache.axis.client.Stub stub = (org.apache.axis.client.Stub)ecogrid;
    stub.setTimeout(10000000);
    long endInstance = System.currentTimeMillis();
    EcogridUtils.debugMessage("The time to create instance is =========== " +
        (endInstance - endFacotry) / 1000, 30);
  }

  /**
   *  Method to login web service and get an sessionid
   *
   *@param  UserName
   *@param  PassWord
   *@return
   *@throws  Exception
   */
  public String login_action(String UserName, String PassWord)
    throws Exception
  {
    String retStr = null;

    EcogridAuthLoginRequestElementType loginReq = new EcogridAuthLoginRequestElementType();
    loginReq.setUserName(UserName);
    loginReq.setPassWd(PassWord);
    if(ecogrid != null)
    {
      retStr = ecogrid.login(loginReq);
    }

    EcogridUtils.debugMessage("The sessionid is " + retStr, 30);
    return retStr;
  }

  /**
   *  Method to logout.
   *
   *@param  SessionId
   *@throws  Exception
   */
  public void logout_action(String SessionId)
    throws Exception
  {
    String retStr = null;
    if(ecogrid != null)
    {
      retStr = ecogrid.logout(SessionId);
    }
  }

  /**
   *  Method to desctory the web service instance. This method will be called
   *  when logout happend.
   *
   *@exception  Exception  Description of the Exception
   */
  public void destory()
    throws Exception
  {
    if(ecogrid != null)
    {
      ecogrid.destroy();
    }
  }

  /**
   */
  static void PrintAuthUsage()
  {
    System.out.println("Usage: java org.ecoinformatics.ecogrid.client.EcogridAuthClient login userName passWord GSH");
    System.out.println("Usage: java org.ecoinformatics.ecogrid.client.EcogridAuthClient logout sessionId GSH");
  }

}

