/**
 *    '$RCSfile: RegistryServiceClient.java,v $'
 *
 *     '$Author: leinfelder $'
 *       '$Date: 2009-03-18 20:31:01 $'
 *   '$Revision: 1.6 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2007 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.ecogrid.client;

import java.io.File;
import java.io.FileReader;
import java.io.Reader;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.registry.stub.AddRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.ListRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.ListResponseElement;
import org.ecoinformatics.ecogrid.registry.stub.QueryRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.QueryResponseElement;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryArrayType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryEntryTypeDocumentType;
import org.ecoinformatics.ecogrid.registry.stub.RegistryServiceLocator;
import org.ecoinformatics.ecogrid.registry.stub.RegistryServicePortType;
import org.ecoinformatics.ecogrid.registry.stub.RemoveRequestElement;
import org.ecoinformatics.ecogrid.registry.stub.UpdateRequestElement;
import org.ecoinformatics.ecogrid.registry.util.RegistryEntryTypeParser;


/**
 * Client for the ecogrid registry
 */
public class RegistryServiceClient
{
  private String serviceURL = null;
  private RegistryServicePortType ecogrid = null;

  final static int ADD_REG = 1;
  final static int UPDATE_REG = 2;
  final static int REMOVE_REG = 3;
  final static int GET_ALL = 4;
  final static int QUERY_REG = 5;

  /**
   *  The main program for the RegistryServiceClient class
   *
   *@param  args  The command line arguments
   */
  public static void main(String[] args)
  {
    String opMode = "";
    int intOpCode = 0;
    String sessionId = "";
    String localFilename = "";
    String gsh_str = "";
    String regName = "";
    String retStr;
    String queryField = "";
    String queryString = "";
    FileReader reader;
    String docid = null;
    if(args.length < 2)
    {
      PrintRegistryClientUsage();
      System.exit(0);
    }

    opMode = args[0];

    if(opMode.compareToIgnoreCase("add") == 0)
    {
      intOpCode = ADD_REG;
      if(args.length != 4)
      {
        PrintRegistryClientUsage();
        System.exit(0);
      }
      sessionId = args[1];
      localFilename = args[2];
      gsh_str = args[3];
    }
    else if(opMode.compareToIgnoreCase("update") == 0)
    {
      intOpCode = UPDATE_REG;
      if(args.length != 5)
      {
        PrintRegistryClientUsage();
        System.exit(0);
      }
      sessionId = args[1];
      docid = args[2];
      localFilename = args[3];
      gsh_str = args[4];
    }
    else if(opMode.compareToIgnoreCase("remove") == 0)
    {
      intOpCode = REMOVE_REG;
      if(args.length != 4)
      {
        PrintRegistryClientUsage();
        System.exit(0);
      }
      sessionId = args[1];
      docid = args[2];
      gsh_str = args[3];
    }
    else if(opMode.compareToIgnoreCase("list") == 0)
    {
      intOpCode = GET_ALL;
      if(args.length != 3)
      {
        PrintRegistryClientUsage();
        System.exit(0);
      }
      sessionId = args[1];
      gsh_str = args[2];

    }
    else if(opMode.compareToIgnoreCase("query") == 0)
    {
      intOpCode = QUERY_REG;
      if(args.length != 5)
      {
        PrintRegistryClientUsage();
        System.exit(0);
      }
      sessionId = args[1];
      queryField = args[2];
      queryString = args[3];
      gsh_str = args[4];
    }
    else
    {
      PrintRegistryClientUsage();
      System.exit(0);
    }

    try
    {
      // create a client object
      RegistryServiceClient client = new RegistryServiceClient(gsh_str);


      switch (intOpCode)
      {
        case ADD_REG:
          reader = new FileReader(new File(localFilename));
          String newId = client.add(sessionId, reader);
          System.out.println("The new id is " + newId);
          break;
        case UPDATE_REG:
          reader = new FileReader(new File(localFilename));
          client.update(sessionId, docid, reader);
          break;
        case REMOVE_REG:

          client.remove(sessionId, docid);
          break;
        case GET_ALL:

          RegistryEntryType[] entry1 = client.list(sessionId);
          print(entry1);
          break;
        case QUERY_REG:
        	RegistryEntryType[] entry = client.query(sessionId, queryField, queryString);
        	print(entry);
        	break;
      }

    }
    catch(Exception e)
    {
      System.out.println("Exception in main : " + e.getMessage());

      if(EcogridUtils.printStackTraceByEnv())
      {
        e.printStackTrace();
      }
    }

  }
  //main

  /**
   * @deprecated Please use the add() methos that uses a RegistryEntryType object param
   */
  public String add(String sessionId, Reader entryReader)
  throws Exception {
	    RegistryEntryType entry = RegistryEntryTypeParser.parseXML(entryReader);
	  return add(sessionId, entry);
  }
  /**
   *  Adds a feature to the RegistryEntry attribute of the RegistryServiceClient
   *  object
   *
   *@param  sessionId      The feature to be added to the RegEntry attribute
   *@param  entry          The feature to be added to the RegEntry attribute
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  public String add(String sessionId, RegistryEntryType entry)
    throws Exception
  {
    AddRequestElement request =
        new AddRequestElement(sessionId, entry);
    String docid = ecogrid.add(request);
    return docid;
  }

  /**
   * @deprecated Please use the method that takes a RegistryEntryType param
   * @param sessionId
   * @param docid
   * @param reader
   * @return
   * @throws Exception
   */
  public String update(String sessionId, String docid, Reader reader)
  throws Exception
{
  RegistryEntryType entry = RegistryEntryTypeParser.parseXML(reader);
  return update(sessionId, docid, entry);
}
  /**
   * update the registry entry with a new entry
   * 
   *@param  sessionId      Description of the Parameter
   *@param  docid          Description of the Parameter
   *@param  reader         Description of the Parameter
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  public String update(String sessionId, String docid, RegistryEntryType entry)
    throws Exception
  {
    UpdateRequestElement update = 
    	new UpdateRequestElement(sessionId, docid, entry);
    String returnValue = ecogrid.update(update);
    return returnValue;
  }

  /**
   * Delete an entry from the registry
   * 
   *@param  sessionId      Description of the Parameter
   *@param  docid        Description of the Parameter
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  public String remove(String sessionId, String docid)
    throws Exception
  {
    RemoveRequestElement remove =
        new RemoveRequestElement(sessionId, docid);
    String returnValue = ecogrid.remove(remove);
    return returnValue;
  }

  /**
   * return all registry entries
   * 
   *@param  sessionId      Description of the Parameter
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  public RegistryEntryType[] list(String sessionId)
    throws Exception
  {
    ListRequestElement allSviceRsqtElement =
        new ListRequestElement(sessionId);
    ListResponseElement getAllRspnsElement =
        ecogrid.list(allSviceRsqtElement);

    if(getAllRspnsElement != null)
    {
      RegistryEntryArrayType regArray = getAllRspnsElement.getListReturn();
      //System.out.println("regArray: " + regArray);
      if(regArray != null)
      {
    	  RegistryEntryType[] regEntries = regArray.getRegistryEntry();
        return regEntries;
      }
      else
      {
        System.out.println("returning null for some reason.");
        return null;
      }
    }
    else
    {
      System.out.println("getAllRspnsElement  is null");
      return null;
    }

  }


  /**
   * query the registry for a specific string
   * 
   *@param  sessionId      Description of the Parameter
   *@param  queryField     Description of the Parameter
   *@param  queryString    Description of the Parameter
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  public RegistryEntryType[] query(String sessionId, String queryField, String queryString)
    throws Exception
  {
    QueryRequestElement request =
        new QueryRequestElement(sessionId, queryField, queryString);
    QueryResponseElement response = ecogrid.query(request);
    if(response != null)
    {
      RegistryEntryArrayType regArray = response.getQueryReturn();
      if(regArray != null)
      {
        return regArray.getRegistryEntry();
      }
      else
      {
        return null;
      }
    }
    else
    {
      return null;
    }

  }

  /**
   *  Constructor for the RegistryServiceClient object
   *
   *@param  serviceGSH     Description of the Parameter
   *@exception  Exception  Description of the Exception
   */
  public RegistryServiceClient(String serviceGSH)
    throws Exception
  {
    this.serviceURL = serviceGSH;
    RegistryServiceLocator loc = 
    	new RegistryServiceLocator();

    ecogrid = loc.getAdd(new java.net.URL(serviceURL));

    org.apache.axis.client.Stub stub = (org.apache.axis.client.Stub)ecogrid;

    stub.setTimeout(10000000);
  }


  /**
   * print the client usage stats
   */
  private static void PrintRegistryClientUsage()
  {
    System.out.println("Usage: java RegistryServiceClient add session_id local_file GSH");
    System.out.println("Usage: java RegistryServiceClient update session_id  docid local_file GSH");
    System.out.println("Usage: java RegistryServiceClient remove session_id docid GSH");
    System.out.println("Usage: java RegistryServiceClient list session_id GSH");
    System.out.println("Usage: java RegistryServiceClient query session_id query_field query_string GSH");
  }

  /**
   * change a reader to a string
   * 
   *@param  entry          Description of the Parameter
   *@return                Description of the Return Value
   *@exception  Exception  Description of the Exception
   */
  private String transformReaderToString(Reader entry)
    throws Exception
  {
    if(entry == null)
    {
      throw new Exception("The regitry entry is null");
    }
    String xml = "";
    int end = 0;
    char[] content = new char[3 * 1024];
    end = entry.read(content);
    while(end != -1)
    {
      String tmp = new String(content, 0, end);
      xml = xml + tmp;
      end = entry.read(content);
    }
    return xml;
  }

  /**
   * print an entry
   * 
   *@param  entry  Description of the Parameter
   */
  private static void print(RegistryEntryType[] entry)
  {
    System.out.println("start of print");
    if(entry == null)
    {
      System.out.println("The Entry list is null");
    }
    else
    {
      for(int i = 0; i < entry.length; i++)
      {
    	System.out.println("--- Registry Entry ---");
    	RegistryEntryType oneEntry = entry[i];
        String serviceName = oneEntry.getServiceName();
        System.out.println("The service name is " + serviceName);
        System.out.println("The service group is " + oneEntry.getServiceGroup());
        System.out.println("The service classification is " + oneEntry.getServiceClassification());
        System.out.println("The service description is " + oneEntry.getDescription());        
        System.out.println("The service ID is " + oneEntry.getId());        
        String serviceType = oneEntry.getServiceType();
        System.out.println("The service type is " + serviceType);
        String endPoint = oneEntry.getEndPoint();
        System.out.println("The endPoint is " + endPoint);
        RegistryEntryTypeDocumentType[] typeList = oneEntry.getDocumentType();
        if(typeList != null)
        {
          for(int j = 0; j < typeList.length; j++)
          {
        	RegistryEntryTypeDocumentType type = typeList[j];
            String namespace = type.getNamespace();
            System.out.println("namespace is " + namespace);
            String label = type.getLabel();
            System.out.println("Label is " + label);
          }
        }
        String[] providerList = oneEntry.getProvider();
        if(providerList != null)
        {
          for(int k = 0; k < providerList.length; k++)
          {
            String provider = providerList[k];
            System.out.println("The provider is " + provider);
          }
        }
      }
    }
  }

}

