/**
 *  '$RCSfile: EcoReg.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: rspears $'
 *     '$Date: 2004-06-10 13:08:18 $'
 * '$Revision: 1.3 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
 
package org.ecoinformatics.ecogrid.registry.impl;

import java.util.Vector;

import org.ecoinformatics.ecogrid.registry.EcoRegInterface;
import org.ecoinformatics.ecogrid.registry.EcoRegConstants;
import org.ecoinformatics.ecogrid.registry.RegEntry;

// For Generic XML Stuff
import java.io.*;

/**
 * Implementation of a registry each records contains "name", "registry name", and "registry address"
 * @author  Rod Spears
 */
public class EcoReg implements EcoRegInterface {
  
  protected final String ENTRY   = "entry";
  protected final String REGNAME = "regname";
  protected final String REGADDR = "regaddr";
  
  protected String   _docPath = null;
  protected Vector   _items   = new Vector();
       
  /**
   * Default Constructor
   */
  public EcoReg() {
  
  }
  
   
  /**
   * Creates an entry in the registry, "name" must be unique
   * @param aName name of service
   * @param aRegName urn name of service
   * @param aRegAddr web address of the service
   * @return status code from EcoRegInterface's error codes
   */
  public int registerEntry(String aName, String aRegName, String aRegAddr)
  {
    if (_items == null) return EcoRegConstants.kEmptyListError;
    
    RegEntry entry = getEntry(aName);
    if (entry == null)
    {
      entry = new RegEntry(aName, aRegName, aRegAddr);
      _items.add(entry);
    }
    else
    {
      return EcoRegConstants.kDupEntryError;
    }
    
    return EcoRegConstants.kNoError;
  }

  /**
   * Updates existing entry the registry database
   * @param aName name of service
   * @param aRegName urn name of service
   * @param aRegAddr web address of the service
   * @return status code from EcoRegInterface's error codes
   */
  public int updateEntry(String aName, String aRegName, String aRegAddr)
  {
    if (_items == null) return EcoRegConstants.kEmptyListError;

    RegEntry entry = getEntry(aName);
    if (entry != null)
    {
      entry.setRegName(aRegName);
      entry.setRegAddr(aRegAddr);
      return EcoRegConstants.kNoError;
    }
    
    return EcoRegConstants.kNotFoundError;
  }

  /**
   * Removes an entry in the registry, "name" must be unique
   * @param aName name of service
   * @return status code from EcoRegInterface's error codes
   */
  public int removeEntry(String aName)
  {
    if (_items == null) return EcoRegConstants.kEmptyListError;
    
    for (int i=0;i<_items.size();i++)
    {
      RegEntry entry = (RegEntry)_items.elementAt(i);
      if (entry.getName().equals(aName)) 
      {
        System.out.println("Item removed: ["+aName+"]");
        _items.removeElementAt(i);
        return EcoRegConstants.kNoError;
      }
    }
    return EcoRegConstants.kNotFoundError;
  }

  /**
   * Looks up an entry by name
   * @param aName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntry(String aName)
  {
    //System.out.println("getEntry: ["+_items.size()+"]");
    if (_items == null) return null;
    
    for (int i=0;i<_items.size();i++)
    {
      RegEntry entry = (RegEntry)_items.elementAt(i);
      //System.out.println("["+entry.getName()+"]["+aName+"]");
      if (entry.getName().equals(aName)) 
      {
        return (RegEntry)_items.elementAt(i);
      }
    }
    return null;
  }

  /**
   * Looks up a entry by Registration name (URN)
   * @param aRegName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntryByRegName(String aRegName) 
  {
    if (_items == null) return null;
    
    for (int i=0;i<_items.size();i++)
    {
      RegEntry entry = (RegEntry)_items.elementAt(i);
      if (entry.getRegName().equals(aRegName)) 
      {
        return (RegEntry)_items.elementAt(i);
      }
    }
    return null;
  }
  
  /**
   * Returns an array list of all registry entries
   * @return returns an XML string of all the entries in the registry
   */
  public RegEntry[] getList()
  {
    if (_items != null && _items.size() == 0) return null;
    
    RegEntry[] array = new RegEntry[_items.size()];
    for (int i=0;i<_items.size();i++)
    {
      RegEntry entry = (RegEntry)_items.elementAt(i);
      array[i] = new RegEntry(entry);
    }
    return array;
  }

  /**
   * Returns an XML list of all registry entries
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLListing()
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.setLength(0);
    strBuf.append("<listing>\n");
    for (int i=0;i<_items.size();i++)
    {
      RegEntry entry = (RegEntry)_items.elementAt(i);
      entry.toXMLStrBuf(strBuf);
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Returns an XML string representing the entry
   * @param aName name of service
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByName(String aName)
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.setLength(0);
    strBuf.append("<listing>\n");
    RegEntry entry = getEntry(aName);
    if (entry != null)
    {
      entry.toXMLStrBuf(strBuf);
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Returns an XML string representing the entry
   * @param aRegName name of entry to be returned
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByRegName(String aRegName)
  {
    StringBuffer strBuf = new StringBuffer("");
    strBuf.append("<listing>\n");
    RegEntry entry = getEntryByRegName(aRegName);
    if (entry != null)
    {
      entry.toXMLStrBuf(strBuf);
    }
    strBuf.append("</listing>\n");
    return strBuf.toString();
  }

  /**
   * Closes registry - writes it to disk
   */
  public boolean closeRegistry()
  {
    try {
      /*FileOutputStream file = new FileOutputStream(_docPath);
      DataOutputStream dos = new DataOutputStream(file);
      dos.writeInt(_items.size());
      dos.writeUTF("\n");
      for (int i=0;i<_items.size();i++)
      {
        RegEntry entry = (RegEntry)_items.elementAt(i);
        entry.write(dos);
      }
      file.close();
       */
      PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(_docPath)));
      int num = _items == null ? 0 : _items.size();
      out.print(num);
      out.println();
      for (int i=0;i<num;i++)
      {
        RegEntry entry = (RegEntry)_items.elementAt(i);
        entry.write(out);
      }
      out.close();

      return true;
    }
    catch (Exception e) {
      System.out.println("closeRegistry - " + e);
    }
    return false;
  }

  /**
   * Opens registry by name (initially it is the file name)
   * @param aFileName name of DB to be opened
   * @return returns true for success, false for failure
   */
  public boolean openRegistry(String aFileName) {
    _docPath = aFileName;
    try {
      File file = new File(aFileName);
      if (file.exists())
      {
        System.err.println("OPened Registry at ["+aFileName+"].");
        FileInputStream fis   = new FileInputStream(aFileName);
        BufferedReader  reader = new BufferedReader(new InputStreamReader(fis));
        String line = reader.readLine();
        int num = Integer.parseInt(line);
        for (int i=0;i<num;i++) 
        {
          RegEntry entry = new RegEntry();
          entry.read(reader);
          _items.add(entry);
        }
        fis.close();
        return true;
      }
      else
      {
        System.err.println("Registry not found at ["+aFileName+"] creating registry.");
        return true;
      }
    }
    catch (Exception e) {
      System.out.println("openRegistry - " + e);
    }
    return false;
  }
  
}
