/**
 *  '$RCSfile: EcogridRegistryService.java,v $'
 *  Copyright: 2004 Regents of the University of California and the
 *              National Center for Ecological Analysis and Synthesis
 *  Purpose: To test the MetaCatURL class by JUnit
 *    Authors: @Rod Spears@
 *    Release: @release@
 *
 *   '$Author: rspears $'
 *     '$Date: 2004-06-10 13:08:19 $'
 * '$Revision: 1.4 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
package org.ecoinformatics.ecogrid.registry.impl;

import org.ecoinformatics.ecogrid.EcogridUtils;
import org.ecoinformatics.ecogrid.registry.RegEntry;

import org.ecoinformatics.ecogrid.registry.EcoRegInterface;

/**
 * Registry Service for registring Ecogrid Resources
 * @author  Rod Spears
 */
public class EcogridRegistryService implements EcoRegInterface {

  protected String               _registryPath = "reg.dat";
  protected EcoRegMetacat        _registry     = null;
  //protected EcogridJSPGlobusBean _regProxy     = null;
  
  /**
   * Constructor - opens database of of the registry automatically
   */
  public EcogridRegistryService()
  {
  try {
    EcogridUtils.setDebug(true);
    EcogridUtils.debugMessage("***********************\n Constructing EcogridRegistryService ");

    EcoRegMetacat registry = new EcoRegMetacat();
    EcogridUtils.debugMessage("Opening DB at Metacat");
    _registry = registry;
    _registry.openRegistry(_registryPath);

    //_regProxy = null;//new EcogridJSPBean();

    if (EcogridUtils.isDebugOn())
    {
      RegEntry[] items = getList();
      if (items != null) {
        EcogridUtils.debugMessage("Num of Entries in DB ["+items.length+"]");
      } else {
        EcogridUtils.debugMessage("No items were returned.");
      }
    }
  } catch (Exception e) {
    System.out.println(e);
    e.printStackTrace();
  }
  }

  /**
   * Makes sure the DB is persisted to disk
   */
  public void persist()
  {
    // this call for the flat file writes out 
    // the data but doesn't "close" it
    _registry.closeRegistry();
  }

  /**
   * Makes sure the DB gets persisted
   */
  public void finalize()
  {
    EcogridUtils.debugMessage("***********************\n Destroying EcogridRegistryService ");
    persist();
  }

  /**
   * Looks up an entry by name
   * @param aName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntry(String aName) 
  {
    return _registry.getEntry(aName);
  }
  
  /**
   * Looks up a entry by Registration name (URN)
   * @param aRegName name of entry to be returned
   * @return returns a RegEntry object
   */
  public RegEntry getEntryByRegName(String aRegName) 
  {
    return _registry.getEntryByRegName(aRegName);
  }
  
  /**
   * Creates an array of all the registry entries
   * @return returns an array of RegEntry objects
   */
  public RegEntry[] getList() 
  {
    return _registry.getList();
  }
  
  /**
   * Creates an entry in the registry, "name" must be unique
   * @param aName name of service
   * @param aRegName urn name of service
   * @param aRegAddr web address of the service
   * @return status code from EcoRegInterface's error codes
   */
  public int registerEntry(String aName, String aRegName, String aRegAddr) 
  {
    EcogridUtils.debugMessage("\n---- EcogridRegistryService ----");
    EcogridUtils.debugMessage("registerEntry - aName:    ["+aName+"]");
    EcogridUtils.debugMessage("registerEntry - aRegName: ["+aRegName+"]");
    EcogridUtils.debugMessage("registerEntry - aRegAddr: ["+aRegAddr+"]");
    EcogridUtils.debugMessage("---- EcogridRegistryService ----");

    int status = _registry.registerEntry(aName,  aRegName,  aRegAddr);
    persist();

    /*if (_regProxy != null)
    {
      EcogridUtils.debugMessage("---- Distrubuting ----");
      RegEntry[] entries = _registry.getList();
      for (int i=0;i<entries.length;i++)
      {
        if (entries[i].getRegName().equals("urn:EcogridRegistryService2"))
        {
          int retVal = _regProxy.registerEntry(entries[i]);
          EcogridUtils.debugMessage("Distributing registerEntry - " + entries[i].getRegAddr() + " status: " + retVal);
        }
      }
    }*/

    return status;
  }
  
  /**
   * Updates existing entry the registry database
   * @param aName name of service
   * @param aRegName urn name of service
   * @param aRegAddr web address of the service
   * @return status code from EcoRegInterface's error codes
   */
  public int updateEntry(String aName, String aRegName, String aRegAddr) 
  {
    EcogridUtils.debugMessage("\n---- EcogridRegistryService ----");
    EcogridUtils.debugMessage("updateEntry - aName:    ["+aName+"]");
    EcogridUtils.debugMessage("updateEntry - aRegName: ["+aRegName+"]");
    EcogridUtils.debugMessage("updateEntry - aRegAddr: ["+aRegAddr+"]");
    EcogridUtils.debugMessage("---- EcogridRegistryService ----");

    int status = _registry.updateEntry(aName,  aRegName,  aRegAddr);

    persist();

    /*if (_regProxy != null)
    {
      EcogridUtils.debugMessage("---- Distrubuting ----");
      RegEntry[] entries = _registry.getList();
      for (int i=0;i<entries.length;i++)
      {
        if (entries[i].getRegName().equals("urn:EcogridRegistryService2"))
        {
          int retVal = _regProxy.updateEntry(entries[i].getRegAddr(), entries[i].getRegName(), entries[i]);
          EcogridUtils.debugMessage("Distributing updateEntry - " + entries[i].getRegAddr() + " status: " + retVal);
        }
      }
    }*/

    return status;
  }

  /**
   * Removes an entry in the registry, "name" must be unique
   * @param aName name of service
   * @return status code from EcoRegInterface's error codes
   */
  public int removeEntry(String aName) 
  {
    EcogridUtils.debugMessage("\n---- EcogridRegistryService ----");
    EcogridUtils.debugMessage("removeEntry - aName:    ["+aName+"]");
    EcogridUtils.debugMessage("---- EcogridRegistryService ----");

    /*if (_regProxy != null)
    {
      EcogridUtils.debugMessage("---- Distrubuting ----");
      RegEntry[] entries = _registry.getList();
      for (int i=0;i<entries.length;i++)
      {
        if (entries[i].getRegName().equals("urn:EcogridRegistryService2"))
        {
          int retVal = _regProxy.removeEntry(entries[i].getRegAddr(), entries[i].getRegName(), aName);
          EcogridUtils.debugMessage("Distributing removeEntry - " + entries[i].getRegAddr() + " status: " + retVal);
        }
      }
    }*/

    EcogridUtils.debugMessage("*** removeEntry " + aName);
    int status = _registry.removeEntry(aName);
    persist();

    return status;
  }

  /**
   * Returns an XML list of all registry entries
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLListing()
  {
    return _registry.getXMLListing();
  }

  /**
   * Returns an XML string representing the entry
   * @param aName name of service
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByName(String aName)
  {
    return _registry.getXMLEntryByName(aName);
  }

  /**
   * Returns an XML string representing the entry
   * @param aRegName name of entry to be returned
   * @return returns an XML string of all the entries in the registry
   */
  public String getXMLEntryByRegName(String aRegName)
  {
    return _registry.getXMLEntryByRegName(aRegName);
  }

}
