/**
 *    '$RCSfile: RegistryViewer.java,v $'
 *
 *     '$Author: rspears $'
 *       '$Date: 2004-10-27 13:21:33 $'
 *   '$Revision: 1.4 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.ecoinformatics.ecogrid.registry.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;

import org.ecoinformatics.ecogrid.registry.stub.EcoRegInterfaceLevelOnePortType;
import org.ecoinformatics.ecogrid.registry.stub.EcogridRegEntryType;
import org.ecoinformatics.ecogrid.registry.stub.service.EcoRegInterfaceLevelOneServiceGridLocator;

/**
 * The Query Builder app which extends a JPanel. This enables it to be embedded in an applet or an application.
 */
public class RegistryViewer extends JPanel implements ListSelectionListener, EcogridProcessorListener
{
  private static final String REG_SERVICE  = "http://sasa.sdsc.edu:8080/ogsa/services/org/ecoinformatics/ecogrid/registry/EcogridRegistryService";
  
  // The preferred size of the window
  private static final int    PREFERRED_WIDTH  = 768;
  private static final int    PREFERRED_HEIGHT = 300;

  // Status Bar
  private JTextField          mStatusField = null;
  private JProgressBar        mProgressBar = new JProgressBar();
  private JMenuBar            mMenuBar     = null;
  private JFrame              mFrame       = null;
  private JTable              mTableView   = null;
  private JMenu               mFileMenu    = null;
  private JMenu               mEntriesMenu = null;
  
  private AddAction           mAddAction;
  private UpdateAction        mUpdateAction;
  private DeleteAction        mDeleteAction;
  private ViewDataAction      mViewDataAction;
  private LoginDlg            mLoginDlg = null;
  
  private Vector                          mRegList        = new Vector();
  private EcoRegInterfaceLevelOnePortType mRegistry       = null;
  private String                          mSessionId      = null;
  private RegistryModel                   mRegistryModel  = null;

  /**
   * Constructor 
   * @param gc
   */
  public RegistryViewer(GraphicsConfiguration gc)
  {
      try
      {
        mRegistry = createEcoRegInterfacePortType(new URL(REG_SERVICE));
      } catch (Exception e)
      {
          
      }
      mRegistryModel = new RegistryModel(this, mRegList);
      initialize(gc);
  }

  /**
   * 
   * @param aURL
   * @return
   * @throws Exception
   */
  public EcoRegInterfaceLevelOnePortType createEcoRegInterfacePortType(URL aURL) throws Exception
  {
    
     EcoRegInterfaceLevelOneServiceGridLocator loc = new EcoRegInterfaceLevelOneServiceGridLocator();
     return loc.getEcoRegInterfaceLevelOnePort(aURL);
  }

  /**
   * General Method for initializing the class
   *
   */
  private void initialize(GraphicsConfiguration gc)
  {
    setLayout(new BorderLayout());

    // set the preferred size of the demo
    setPreferredSize(new Dimension(PREFERRED_WIDTH, PREFERRED_HEIGHT));

    initializeUI(gc);

    // Note that
    // we again must do this on the GUI thread using invokeLater.
    SwingUtilities.invokeLater(new Runnable()
    {
      public void run()
      {
        showApp();
       }
    });
  }
  
  /**
   * Starts the login process
   * @param aUserName
   * @param aPassword
   * @return
   */
    protected void login(String aUserName, String aPassword)
    {
        mProgressBar.setIndeterminate(true);
        updateUI(false);
        setStatus("Logging in...");
        EcogridProcessor ep = new EcogridProcessor(mRegistry, aUserName, aPassword, this);
        ep.start();
    }
  
    /**
     * Starts the logout process
     * @param aSessionId
     * @return
     */
      private void logout(String aSessionId)
      {
          mProgressBar.setIndeterminate(true);
          updateUI(false);
          setStatus("Logging out...");
          EcogridProcessor ep = new EcogridProcessor(mRegistry, aSessionId, this);
          ep.start();
      }
    
  /**
   * Gets all the registry entries
   * @throws Exception
   */
  protected void getAllRegistries()
  {
      mProgressBar.setIndeterminate(true);
      updateUI(false);
      setStatus("Getting registry entries...");
      EcogridProcessor ep = new EcogridProcessor(mRegistry, mSessionId, 
                                                 EcogridProcessor.DO_GET_ALL, 
                                                 null, this);
      ep.start();

  }

  /**
   * Adds an enntry
   * @param aRegEntry
   */
  public void addItem(EcogridRegEntryType aRegEntry)
  {
    if (aRegEntry != null)
    {
        mProgressBar.setIndeterminate(true);
        updateUI(false);
        EcogridProcessor ep = new EcogridProcessor(mRegistry, mSessionId, 
                                                   EcogridProcessor.DO_ADD, 
                                                   aRegEntry, this);
        ep.start();
    }
  }

  /**
   * UPdates an item
   * @param aRegEntry
   */
  public void updateItem(EcogridRegEntryType aRegEntry)
  {
      if (aRegEntry != null)
      {
          mProgressBar.setIndeterminate(true);          
          updateUI(false);
          EcogridProcessor ep = new EcogridProcessor(mRegistry, mSessionId, 
                                                     EcogridProcessor.DO_UPDATE, 
                                                     aRegEntry, this);
          ep.start();
      } 
  }


  /**
   * Helper function for adding and updating, display the dialog.
   * @param aDoingAdd
   */
  public void doAddUpdateItem(boolean aDoingAdd)
  {
      SwingUtilities.invokeLater(new RVAppRunnable(this,  new Boolean(aDoingAdd))
      {
        public void run()
        {
            boolean doingAdd = ((Boolean)mObj).booleanValue();
            RegEntryDlg dlg = new RegEntryDlg(mFrame, mApp, doingAdd);
            dlg.setRegEntry(doingAdd ? null : (EcogridRegEntryType)mRegList.elementAt(mTableView.getSelectedRow()));
            dlg.show();
            if (!dlg.wasCancelled())
            {
              if (doingAdd)
              {
                addItem(dlg.getRegEntry());
              } else
              {
                updateItem(dlg.getRegEntry());
              }
            }
        }
      });
      
  }
  
  /**
   * 
   *
   */
  public void doDeleteSelectedItem()
  {
      if (JOptionPane.YES_OPTION == JOptionPane.showConfirmDialog(this, "Do you wish to remove the selected item?", "Delete Item", JOptionPane.YES_NO_OPTION))
      {
          EcogridRegEntryType regEntry = (EcogridRegEntryType)mRegList.elementAt(mTableView.getSelectedRow());
          if (regEntry != null)
          {
              mProgressBar.setIndeterminate(true);
              updateUI(false);
              EcogridProcessor ep = new EcogridProcessor(mRegistry, mSessionId, 
                                                         EcogridProcessor.DO_REMOVE, 
                                                         regEntry, this);
              ep.start();
          }
      }
  }
    
  /**
   * 
   *
   */
  public void viewDataForSelectedItem()
  {
/*      DataCacheItem item = (DataCacheItem)mCacheMgr.getItems().elementAt(mTableView.getSelectedRow());
      JFrame dataViewer = new JFrame();
      dataViewer.getContentPane().setLayout(new BorderLayout());
      JEditorPane editor = new JEditorPane();
      String dataStr = "No Data Available.";
      if (item.getData() != null) 
      {
          dataStr = new String(item.getData());
      }
      editor.setText(dataStr);
      JScrollPane scroller = new JScrollPane(editor);
      dataViewer.getContentPane().add(scroller, BorderLayout.CENTER);
      dataViewer.setBounds(0,0,600,400);
      dataViewer.show();*/
  }
 
    // *******************************************************
    // *************** Load UI ******************
    // *******************************************************
    public void initializeUI(GraphicsConfiguration gc)
    {
        mFrame = new JFrame(gc);
        JPanel top = new JPanel();
        top.setLayout(new BorderLayout());
        add(top, BorderLayout.NORTH);
        mMenuBar = createMenus();
        if (mMenuBar != null)
            top.add(mMenuBar, BorderLayout.NORTH);
        
        JPanel statusPanel = new JPanel(new BorderLayout());
        mStatusField = new JTextField("");
        mStatusField.setEditable(false);
        statusPanel.add(mStatusField, BorderLayout.CENTER);
        statusPanel.add(mProgressBar, BorderLayout.EAST);
        add(statusPanel, BorderLayout.SOUTH);
        
        // Create the table
        mTableView = new JTable(mRegistryModel);
        JScrollPane scrollpane = new JScrollPane(mTableView);
        mTableView.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        mTableView.setColumnSelectionAllowed(false);
        mTableView.setRowSelectionAllowed(true);
        mTableView.getSelectionModel().addListSelectionListener(this);
        add(scrollpane, BorderLayout.CENTER);
        
        DefaultTableCellRenderer renderer = new DefaultTableCellRenderer() {
            JLabel label = new JLabel();

            public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column)
            {
                if (!label.isOpaque())
                {
                    label.setOpaque(true);
                }
                label.setHorizontalAlignment(JLabel.CENTER);
                label.setText(value.toString());

                label.setBackground(isSelected? Color.BLUE : Color.WHITE);
                label.setForeground(isSelected? Color.WHITE : Color.BLACK);
                return label;
            }
        };  
        for (int i=0;i<mRegistryModel.HEADINGS.length;i++) {
            TableColumn column = mTableView.getColumn(mRegistryModel.HEADINGS[i]);
            column.setCellRenderer(renderer);
        }
    }
    
    /**
     * Create menus
     */
    public JMenuBar createMenus()
    {
        JMenuBar menuBar = null;
        menuBar = new JMenuBar();
        JMenuItem mi;
        mFileMenu = (JMenu) menuBar.add(new JMenu("File"));
        mFileMenu.setMnemonic('F');

        mi = createMenuItem(mFileMenu, "Exit", "x", "Exit Appication", true, null);
        mi.addActionListener(new ActionListener()
        {
            public void actionPerformed(ActionEvent ae)
            {
                if (mSessionId != null)
                {
                  logout(mSessionId);
                }
            }
        });
        mEntriesMenu = (JMenu) menuBar.add(new JMenu("Entries"));
        mEntriesMenu.setMnemonic('E');
        mi = createMenuItem(mEntriesMenu, "Add", "A", "Add", false, mAddAction = new AddAction(this));
        mi = createMenuItem(mEntriesMenu, "Update", "U", "Update", false, mUpdateAction = new UpdateAction(this));
        mEntriesMenu.addSeparator();
        mi = createMenuItem(mEntriesMenu, "Delete Item", "D", "Delete", false, mDeleteAction = new DeleteAction(this));
        
        //JMenu viewMenu = (JMenu) menuBar.add(new JMenu("View"));
        //viewMenu.setMnemonic('V');
        //mi = createMenuItem(viewMenu, "Data", "D", "Data", false, mViewDataAction = new ViewDataAction(this));
        
        mAddAction.setEnabled(true);

        return menuBar;
    }

    /**
     * @param b
     * @return
     */
    protected PropertyChangeListener createActionChangeListener(JMenuItem b)
    {
        return new ActionChangedListener(b);
    }
    
    /**
     * 
     * @author globus
     *
     * TODO To change the template for this generated type comment go to
     * Window - Preferences - Java - Code Generation - Code and Comments
     */
    private class ActionChangedListener implements PropertyChangeListener
    {
        JMenuItem menuItem;
        ActionChangedListener(JMenuItem mi)
        {
            super();
            this.menuItem = mi;
        }
        public void propertyChange(PropertyChangeEvent e)
        {
            String propertyName = e.getPropertyName();
            if (e.getPropertyName().equals(Action.NAME))
            {
                String text = (String) e.getNewValue();
                menuItem.setText(text);
            } else if (propertyName.equals("enabled"))
            {
                Boolean enabledState = (Boolean) e.getNewValue();
                menuItem.setEnabled(enabledState.booleanValue());
            }
        }
    }
    
  /**
   * Creates a generic menu item
   */
    public JMenuItem createMenuItem(JMenu aMenu,
                                    String aLabel,
                                    String aMnemonic,
                                    String aAccessibleDescription,
                                    boolean aEnabled,
                                    AbstractAction aAction)
    {
        JMenuItem mi = (JMenuItem) aMenu.add(new JMenuItem(aLabel));
        mi.setMnemonic(aMnemonic.charAt(0));
        mi.getAccessibleContext().setAccessibleDescription(aAccessibleDescription);
        mi.addActionListener(aAction);
        if (aAction != null)
        {
          aAction.addPropertyChangeListener(createActionChangeListener(mi));
          aAction.setEnabled(aEnabled);
        }
        //mi.setEnabled(aEnabled);
        return mi;
    }


    /**
     * Enables disables the UI
     * @param aEnable
     */
    protected void updateUI(boolean aEnable)
    {
        mFileMenu.setEnabled(aEnable);
        mEntriesMenu.setEnabled(aEnable);
        mTableView.setEnabled(aEnable);
    }
  
  /**
   * Bring up the PPApp demo by showing the frame (only applicable if coming up
   * as an application, not an applet);
   */
  public void showApp()
  {
      // put PPApp in a frame and show it
      JFrame f = getFrame();
      f.setTitle("Registry Editor");
      f.getContentPane().add(this, BorderLayout.CENTER);
      f.pack();

      Rectangle screenRect = f.getGraphicsConfiguration().getBounds();
      Insets screenInsets = Toolkit.getDefaultToolkit().getScreenInsets(
          f.getGraphicsConfiguration());

      // Make sure we don't place the demo off the screen.
      int centerWidth = screenRect.width < f.getSize().width ? screenRect.x : screenRect.x
          + screenRect.width / 2 - f.getSize().width / 2;
      int centerHeight = screenRect.height < f.getSize().height ? screenRect.y : screenRect.y
          + screenRect.height / 2 - f.getSize().height / 2;

      centerHeight = centerHeight < screenInsets.top ? screenInsets.top : centerHeight;

      f.setLocation(centerWidth, centerHeight);
      f.show();
      
      //if (mRegistry != null)
      //{
          // do the following on the gui thread
          SwingUtilities.invokeLater(new RVAppRunnable(this,  "")
          {
            public void run()
            {
                mLoginDlg = new LoginDlg(mFrame, mApp);
                mLoginDlg.show();
                if (!mLoginDlg.wasCancelled())
                {
                  mApp.login(mLoginDlg.getUsername(), mLoginDlg.getPassword());
                }
            }
          });
      /*} else {
          JOptionPane.showMessageDialog(
                  this,
                  "There was a problem connecting to the registry",
                  "Registry Problem",
                  JOptionPane.ERROR_MESSAGE); 
          setStatus("Not connected to the registry.");
      }*/
  }

  /**
   * Returns the frame instance
   */
  public JFrame getFrame()
  {
    return mFrame;
  }

  /**
   * Returns the menubar
   */
  public JMenuBar getMenuBar()
  {
    return mMenuBar;
  }

  /**
   * Set the status
   */
  public void setStatus(String s)
  {
    // do the following on the gui thread
    SwingUtilities.invokeLater(new RVAppRunnable(this, s)
    {
      public void run()
      {
        mApp.mStatusField.setText((String) mObj);
      }
    });
  }
  
  // *******************************************************
  // ******************   Runnables  ***********************
  // *******************************************************

  /**
   * Generic PPApp runnable. This is intended to run on the
   * AWT gui event thread so as not to muck things up by doing
   * gui work off the gui thread. Accepts a PPApp and an Object
   * as arguments, which gives subtypes of this class the two
   * "must haves" needed in most runnables for this demo.
   */
  class RVAppRunnable implements Runnable
  {

    protected RegistryViewer mApp;
    protected Object         mObj;

    public RVAppRunnable(RegistryViewer aApp, Object aObj)
    {
      this.mApp = aApp;
      this.mObj = aObj;
    }

    public void run()
    {
    }
  }

  //-----------------------------------------------------------------------------
  //-- RegistryModel
  //-----------------------------------------------------------------------------
  public class RegistryModel extends AbstractTableModel {
      
    public final String HEADINGS[] = { "Service Name", "Service Type", "End Point", "Classification"};
    //private EcogridRegistryService    mRegService    = null; 
    
    protected RegistryViewer mApp   = null;
    protected Vector         mItems = null;
    
    /**
     * 
     * @param aCities
     */
    public RegistryModel(RegistryViewer aApp, Vector aRegList)
    {
      mApp        = aApp;
      mItems      = aRegList;
    }
    
    public int getColumnCount() 
    {
        return HEADINGS.length;
    }

    public int getRowCount() 
    {
        return mItems.size();
    }

    public void deleteItemAt(int aInx)
    {
        mItems.removeElementAt(aInx);
        this.fireTableDataChanged();
    }
    
    public void deleteItem(EcogridRegEntryType aRegEntry)
    {
        mItems.removeElement(aRegEntry);
        this.fireTableDataChanged();
    }
    
    public void clear()
    {
        mItems.clear();
        this.fireTableDataChanged();
    }
    
    public void addItem(EcogridRegEntryType aRegEntry)
    {
      mItems.addElement(aRegEntry);
      this.fireTableDataChanged();
    }
    
    public void updateItem()
    {
      this.fireTableDataChanged();
    }
    
    public Object getValueAt(int row, int col)
    {
        EcogridRegEntryType item = (EcogridRegEntryType) mItems.elementAt(row);
        if (item != null)
            {
                switch (col) {
                    case 0 :
                        return item.getServiceName();
                    case 1 :
                        return item.getServiceType();
                    case 2 :
                        return item.getEndPoint();
                    case 3 :
                        return item.getServiceClassification();
                }
            }
            return "";
        }

    public String getColumnName(int column) {
        return HEADINGS[column];
    }

    public Class getColumnClass(int c) {
        return getValueAt(0, c).getClass();
    }

    public boolean isCellEditable(int row, int col) {
        return false;
    }

    public void setValueAt(Object aValue, int row, int column) { /*
                                                                  * data[row][column] =
                                                                  * aValue;
                                                                  */
    }
  }
  
  //--------------------------------------------
  //-------- Actions -------------
  //--------------------------------------------
  class AddAction extends AbstractAction
  {
    RegistryViewer mRegViewer;
    protected AddAction(RegistryViewer aRegViewer)
    {
      super("AddAction");
      this.mRegViewer = aRegViewer;
    }
    public void actionPerformed(ActionEvent e)
    {
      mRegViewer.doAddUpdateItem(true);
    }
  }
  
  class UpdateAction extends AbstractAction
  {
    RegistryViewer mRegViewer;
    protected UpdateAction(RegistryViewer aRegViewer)
    {
      super("UpdateAction");
      this.mRegViewer = aRegViewer;
    }
    public void actionPerformed(ActionEvent e)
    {
      mRegViewer.doAddUpdateItem(false);
    }
  }
  
  class DeleteAction extends AbstractAction
  {
    RegistryViewer mRegViewer;
    protected DeleteAction(RegistryViewer aRegViewer)
    {
      super("DeleteAction");
      this.mRegViewer = aRegViewer;
    }
    public void actionPerformed(ActionEvent e)
    {
      mRegViewer.doDeleteSelectedItem();
    }
  }
  
  class ViewDataAction extends AbstractAction
  {
    RegistryViewer mRegViewer;
    protected ViewDataAction(RegistryViewer aRegViewer)
    {
      super("ViewDataAction");
      this.mRegViewer = aRegViewer;
    }
    public void actionPerformed(ActionEvent e)
    {
      mRegViewer.viewDataForSelectedItem();
    }
  }
  
  

  
  //--------------------------------------------
  //-------- ListSelectionListener -------------
  //--------------------------------------------
  
  /**
   * Upates the "inspector" ui when an item in the list is clicked on
   */
  public void valueChanged(ListSelectionEvent e)
  {
      //if (!mProgressBar.isIndeterminate())
      //{
        if (e == null || !e.getValueIsAdjusting()) 
        {
            int selectedInx = mTableView.getSelectedRow();
            boolean isSelected = selectedInx != -1 && mRegistryModel.getRowCount() > 0;
            boolean isReg      = mSessionId != null;
            mAddAction.setEnabled(isReg);
            mUpdateAction.setEnabled(isSelected && isReg);
            mDeleteAction.setEnabled(isSelected && isReg);
            //mViewDataAction.setEnabled(isSelected);
        }
      //}
  } 
  
  //-----------------------------------------------------------------------------
  //-- EcogridProcessorListener
  //-----------------------------------------------------------------------------

  public void complete(EcogridProcessor aEP)
  {
      mProgressBar.setIndeterminate(false);
      updateUI(true);
      int status = aEP.getStatus();
      
      switch (aEP.getWorkType())
      {
          case EcogridProcessor.DO_NONE:
              break;
          
          case EcogridProcessor.DO_LOGIN:
              {
                  mSessionId = aEP.getSessionId();
                  if (mSessionId == null || aEP.getStatus() == EcogridProcessor.PROCESS_ERROR) 
                  {
                      JOptionPane.showMessageDialog(
                              this,
                              "There was a problem authenticating you as a user!",
                              "Login Problem",
                              JOptionPane.ERROR_MESSAGE);                
                      setStatus("Login failed.");                  
                  } else
                  {
                      setStatus("Login successful.");
                      this.getAllRegistries();
                  }
              }
              break;
          
          case EcogridProcessor.DO_LOGOUT:
              System.exit(0);
              break;
          
          case EcogridProcessor.DO_GET_ALL:
              {
                  if (aEP.getStatus() == EcogridProcessor.PROCESS_ERROR)
                  {
                      JOptionPane.showMessageDialog(
                              this,
                              "Error getting all the services.",
                              "Getting Entries Problem",
                              JOptionPane.ERROR_MESSAGE);
                  } else 
                  {
                      EcogridRegEntryType[] regEntries = aEP.getRegEntries();
                      mRegList.clear();
                      if (regEntries != null)
                      {
                          for (int i=0;i<regEntries.length;i++)
                          {
                              mRegList.addElement(regEntries[i]);   
                          }
                          setStatus("There were " + regEntries.length + " entries in the registry");
                          this.mRegistryModel.updateItem(); // refreshes the UI from the model
                      } else
                      {
                          setStatus("There were no entries in the registry");
                      }
                  }
              }
              break;
              
          case EcogridProcessor.DO_ADD:
              if (aEP.getStatus() != EcogridProcessor.PROCESS_ERROR)
              {
                  mRegistryModel.addItem(aEP.getRegEntry());
                  setStatus("Item added.");
              } else
              {
                  JOptionPane.showMessageDialog(
                          this,
                          "Error Adding the item: "+aEP.getStatusStr(),
                          "Add Problem",
                          JOptionPane.ERROR_MESSAGE);                
                  setStatus("Add failed.");                  
              } 
              break;
          
          case EcogridProcessor.DO_UPDATE:
              if (aEP.getStatus() != EcogridProcessor.PROCESS_ERROR)
              {
                  mRegistryModel.updateItem(); // notification
                  setStatus("Item updated.");
              } else
              {
                  JOptionPane.showMessageDialog(
                          this,
                          "Error Updating the item: "+aEP.getStatusStr(),
                          "Updating Problem",
                          JOptionPane.ERROR_MESSAGE);                
                  setStatus("Update failed.");                  
              } 
              break;
              
          case EcogridProcessor.DO_REMOVE:
              if (aEP.getStatus() != EcogridProcessor.PROCESS_ERROR)
              {
                  mRegistryModel.deleteItem(aEP.getRegEntry());
                  setStatus("Item removed.");
              } else
              {
                  JOptionPane.showMessageDialog(
                          this,
                          "Error removing the item: "+aEP.getStatusStr(),
                          "Remove Problem",
                          JOptionPane.ERROR_MESSAGE);                
                  setStatus("Remove failed.");                  
              } 
              break;
      }
  }
  
  //-----------------------------------------------------------------------------
  //-- Application MAIN
  //-----------------------------------------------------------------------------

  /**
   * PPApp Main. Called only if we're an application, not an applet.
   */
  public static void main(String[] args)
  {
    // Create PPApp on the default monitor
    RegistryViewer qbApp = new RegistryViewer(GraphicsEnvironment.getLocalGraphicsEnvironment()
        .getDefaultScreenDevice().getDefaultConfiguration());
  }

}

