/**
 *    Purpose: A Class that implements put service for SRB data mgmt system.
 *  Copyright: 2004 Regents of the University of California and
 *             San Diego Supercomputer Center
 *    Authors: Bing Zhu
 *             Sept 14, 2004
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package org.ecoinformatics.ecogrid.srb.put;

import java.util.Vector;
import java.io.*;

import org.ecoinformatics.ecogrid.stub.EcoGridPutInterfaceLevelOnePortType;
import org.ecoinformatics.ecogrid.EcoGridErrorCode;
import org.ecoinformatics.ecogrid.EcogridObjType;
import org.ecoinformatics.ecogrid.srb.SRBuri;

import org.globus.ogsa.impl.ogsi.GridServiceImpl;
import org.globus.ogsa.impl.ogsi.PersistentGridServiceImpl;


import java.rmi.RemoteException;

import edu.sdsc.grid.io.srb.*;
import edu.sdsc.grid.io.*;

import java.net.URI;
import java.net.URISyntaxException;

import org.apache.xerces.parsers.DOMParser;
import org.xml.sax.InputSource;
import org.w3c.dom.*;

import org.ecoinformatics.ecogrid.EcogridUtils;

/**
 * The class implements ecogrid query to srb query
 */
//public class SrbPutImpl extends GridServiceImpl implements EcoGridPutInterfaceLevelOnePortType
public class SrbPutImpl extends PersistentGridServiceImpl implements EcoGridPutInterfaceLevelOnePortType
{
   SRBuri srbURI;
   String srbHost, srbPort, srbUserName, srbPasswd, srbParentCollection, srbMdasDomainHome, srbDefaultResource;
   String srbFilename;

   int uploadObjType;

   public SrbPutImpl()
   {
      super("Sput SRB Service");
   }

   // The SRB config file is $TOMCAT_HOME/webapps/ogsa/WEB-INF/etc/SrbConfig.xml
   void GetSrbConfig()
   {
      Document doc = EcogridUtils.OpenEcogridConfigFile("srb");
      if(doc == null)  // maybe the file doesn't exist.
        return;

      String tmpStr;

      Element rootElement = doc.getDocumentElement();

      // default value for debug in 'EcogridUtils' is false.
      tmpStr = EcogridUtils.getEcogridConfigValue(doc,"debugOn");
      if((tmpStr != null) && (tmpStr.length() > 0))
      {
         if(tmpStr.compareToIgnoreCase("yes") == 0)
           EcogridUtils.setDebug(true);
      }
   }

   public void put(byte[] objData, java.lang.String objectId, int objType, java.lang.String sessionId) throws RemoteException
   {
      // read config info dynamically from ecogrid config fil.
      GetSrbConfig();

      EcogridUtils.debugMessage("server side srb put(): sessionID is " + sessionId);

      uploadObjType = objType;

      if(objType == EcogridObjType.DATA)
      {
          // Debug code: check if bytes are good
          /*
          try
          {
             FileOutputStream fout = new FileOutputStream("/home/bzhu/jakarta-tomcat-4.1.27/logs/fun.fun");
             fout.write(objData, 0, objData.length);
             fout.close();
          }
          catch(Exception e)
          {
             throw new RemoteException(e.getMessage());
          }
          */

          UploadSrbData(objData, objectId);

          EcogridUtils.debugMessage("UploadSrbData() succeeded.");
      }
      else if(objType == EcogridObjType.METADATA)
      {
          UploadSrbMetadata(objData, objectId);

          EcogridUtils.debugMessage("UploadSrbMetadata succeeded.");
      }
      else
      {
          throw new RemoteException("Unknown Ecogrid data type. The data type is 'data' or 'metadata'.");
      }
   }

   SRBAccount CreateSrbAccountByURI(String srbUri) throws RemoteException
   {
       SRBAccount srbAccount= null;
       try
       {
          srbURI = new SRBuri(srbUri);
       }
       catch(Exception e)
       {
          throw new RemoteException(e.getMessage());
       }

       srbHost = srbURI.getSrbHost();
       if(srbURI.getSrbPort() > 0)
          srbPort = Integer.toString(srbURI.getSrbPort());
       else
          srbPort = "7614";
       srbUserName = srbURI.getSrbUserName();
       srbPasswd = srbURI.getSrbPassword();
       srbParentCollection = srbURI.getSrbParentCollection();
       srbMdasDomainHome = srbURI.getSrbMdasDomainHome();
       srbDefaultResource = srbURI.getSrbDefaultResource();
       srbFilename = srbURI.getSrbFilename();

       // if it is for uploading metadat, we put a dummy resource if it is null.
       if((srbDefaultResource == null)||(srbDefaultResource.length() == 0))
         srbDefaultResource = "dummy";

       try
       {
          srbAccount = new SRBAccount(srbHost, Integer.parseInt(srbPort), srbUserName, srbPasswd,
                              srbParentCollection, srbMdasDomainHome, srbDefaultResource);
       }
       catch(Exception e)
       {
          throw new RemoteException(e.getMessage());
       }

       return srbAccount;
   }

   void UploadSrbMetadata(byte[] bytes, String srbUri) throws RemoteException
   {
       EcogridUtils.debugMessage("UploadSrbMetadata()");

       String xmlString = new String(bytes);

       // get all 3-paired metadata
       DOMParser parser = null;
       Document doc = null;
       try
       {
          parser = new DOMParser();
          StringReader inReader = new StringReader(xmlString);
          InputSource inSource = new InputSource(inReader);
          parser.parse(inSource);
          doc = parser.getDocument();
       }
       catch(Exception e)
       {
          throw new RemoteException(e.getMessage());
       }

       // use dom parser???
       Vector attVector, valVector, unitVector;
       attVector = new Vector();
       valVector = new Vector();
       unitVector = new Vector();

       int i;
       Element rootElement = doc.getDocumentElement();
       if(rootElement.hasChildNodes())
       {
          NodeList recordList = rootElement.getChildNodes();
          for(i=0;i<recordList.getLength();i++)
          {
             Node recordNode = recordList.item(i);
             if((recordNode.getNodeType() == Node.ELEMENT_NODE)&&(recordNode.getNodeName().compareToIgnoreCase("record") == 0))
             {
                  NamedNodeMap attMap = recordNode.getAttributes();

                  Node tmpNode = attMap.getNamedItem("attribute");
                  attVector.add(tmpNode.getNodeValue());

                  tmpNode = attMap.getNamedItem("value");
                  valVector.add(tmpNode.getNodeValue());

                  tmpNode = attMap.getNamedItem("unit");
                  unitVector.add(tmpNode.getNodeValue());
             }
          }
       }

       if((attVector.size() != valVector.size()) || (valVector.size() != unitVector.size()))
       {
          throw new RemoteException("an error occured in retrieving metadata.");
       }

       if(attVector.size() == 0)
         return;

       SRBAccount srbAccount = CreateSrbAccountByURI(srbUri);

       String[][] definableMetaDataValues = new String[attVector.size()][3];
       for(i=0;i<attVector.size();i++)
       {
           definableMetaDataValues[i][0] = (String)attVector.elementAt(i);
           definableMetaDataValues[i][1] = (String)valVector.elementAt(i);
           definableMetaDataValues[i][2] = (String)unitVector.elementAt(i);
       }

       int[] operators = new int[definableMetaDataValues.length];

       MetaDataTable metaDataTable = new MetaDataTable( operators, definableMetaDataValues);

       MetaDataRecordList[] rl = new MetaDataRecordList[1];
       rl[0] = new SRBMetaDataRecordList( SRBMetaDataSet.getField(
                       SRBMetaDataSet.DEFINABLE_METADATA_FOR_FILES ), metaDataTable );

       SRBFileSystem srbFileSystem = null;
       SRBFile srbFile = null;
       try
       {
           srbFileSystem = new SRBFileSystem( srbAccount );
           srbFile = new SRBFile( srbFileSystem, srbParentCollection, srbFilename);
           srbFile.modifyMetaData(rl[0]);
       }
       catch(Exception e)
       {
           srbFile = null;
           srbFileSystem = null;
           srbAccount = null;
           throw new RemoteException(e.getMessage());
       }

       srbFile = null;
       srbFileSystem = null;
       srbAccount = null;
   }

   void UploadSrbData(byte[] bytes, String srbUri) throws RemoteException
   {
       EcogridUtils.debugMessage("UploadSrbData()");

       SRBAccount srbAccount = CreateSrbAccountByURI(srbUri);
       if((srbDefaultResource == null) || (srbDefaultResource.length() == 0))
       {
          throw new RemoteException("No storage is defined via 'defaultResouce'.");
       }

       SRBFileSystem srbFileSystem = null;
       SRBFile srbFile = null;

       SRBRandomAccessFile srbRandomAccessFile= null;
       GeneralFileSystem fileSystem = null;

       int cnt = bytes.length/20000;
       int remain_cnt = (int)(bytes.length%20000);

       int i,j;
       try
       {
           byte[] tmpBytes = new byte[20000];

           srbFileSystem = new SRBFileSystem( srbAccount );
           srbFile = new SRBFile( srbFileSystem, srbParentCollection, srbFilename);
           srbRandomAccessFile = new SRBRandomAccessFile( srbFile, "rw");

           for(i=0;i<cnt;i++)
           {
              for(j=0;j<20000;j++)
                tmpBytes[j] = bytes[i*20000 + j];
              srbRandomAccessFile.write(tmpBytes, 0, 20000);
           }
           if(remain_cnt > 0)
           {
              for(j=0;j<remain_cnt;j++)
                tmpBytes[j] = bytes[cnt*20000+j];
              srbRandomAccessFile.write(tmpBytes, 0, remain_cnt);
           }
           srbRandomAccessFile.close();
       }
       catch ( Throwable e )
       {
            srbRandomAccessFile = null;
            srbFile = null;
            srbFileSystem = null;
            srbAccount = null;

            throw new RemoteException(e.getMessage());
       }

       srbRandomAccessFile = null;
       srbFile = null;
       srbFileSystem = null;
       srbAccount = null;
   }
}
