package org.ROADnet;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.DoubleToken;
import ptolemy.data.IntToken;
import ptolemy.data.StringToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

import com.brtt.antelope.Orb;
import com.brtt.antelope.OrbErrorException;
import com.brtt.antelope.OrbPacketChannel;
import com.brtt.antelope.OrbWaveformPacket;
import com.brtt.antelope.SourceName;

/** Ptolemy actor to send waveform data to an Antelope ORB 
 *  
 * @author Tobin Fricke, University of California
 * @version $Id: OrbWaveformSink.java,v 1.3 2005/11/01 20:39:05 ruland Exp $
 * @Pt.ProposedRating Red (tobin)
 */

public class OrbWaveformSink extends TypedAtomicActor {
    
    /**
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */

    public OrbWaveformSink(CompositeEntity container, String name)
	throws NameDuplicationException, IllegalActionException {
	super(container, name);

	input = new TypedIOPort(this, "input", true, false);
	times = new TypedIOPort(this, "times", true, false);
	
	input.setMultiport(false);
	input.setTypeEquals(BaseType.INT);

	times.setMultiport(false);
	times.setTypeEquals(BaseType.DOUBLE);

	orbname = new Parameter(this, "orbname");
	srcname = new Parameter(this, "srcname");
	nsamp   = new Parameter(this, "nsamp");
	samprate= new Parameter(this, "samprate");

	orbname.setTypeEquals(BaseType.STRING);
	srcname.setTypeEquals(BaseType.STRING);
    }

    /* The initialization will connect to the ORB */

    public void initialize() throws IllegalActionException {
	System.out.println("OrbWaveformSink: initialize");
	try {
	    super.initialize();
	    _orb = new Orb(StringToken.convert(orbname.getToken()).stringValue(),"w");
	    _orb.select(StringToken.convert(srcname.getToken()).stringValue());
	    _orb.after(0);
	} catch (Exception e) {
	    System.out.println("Couldn't connect to ORB! (" + e + ")");
	}
    }
    

    public boolean prefire() throws IllegalActionException {
	return (input.hasToken(0) && super.prefire());
    }


    public void fire() throws IllegalActionException {
	super.fire();
	try {
	
	    int nsamp = ((IntToken)(this.nsamp.getToken())).intValue();

	    /* We expect timestamps to arrive synchronously with samples.  We
	       only keep the timestamps that correspond to the beginning of a
	       packet, however. */

	    double sampleTime = ((DoubleToken)(times.get(0))).doubleValue();

	    /* Manufacture a new array for each packet, since OrbPacket doesn't 
	       (currently) make a copy. Not sure if this is necessary. */

	    if (samplesSoFar == 0) {
		samplesBuffer = new int[nsamp];
		packetTime = sampleTime;
	    }

	    samplesBuffer[samplesSoFar] = ((IntToken)(input.get(0))).intValue();
	    samplesSoFar ++;
	    
	    if (samplesSoFar == nsamp) {

		samplesSoFar = 0;

		String srcname = 
		    ((StringToken)(this.srcname.getToken())).stringValue();

		double samprate = 
		    ((DoubleToken)(this.samprate.getToken())).doubleValue();
		
		OrbWaveformPacket pkt = 
		    new OrbWaveformPacket(packetTime, 
					  0, new SourceName(srcname));

		double calib = 1.0;   // FixMe
		double calper = -1;
		String segtype = "c";
	
		OrbPacketChannel channel = 
		    new OrbPacketChannel(samplesBuffer, pkt.srcname, calib, 
					 calper, segtype, packetTime, 
					 samprate);
		
		pkt.addChannel(channel); 
//		pkt.stuffGEN();

		System.out.println("Pushing this channel:\"" + channel + "\"");
		System.out.println("In this packet: \"" + pkt + "\"");
		_orb.put(pkt.stuff());

	    } 

	} catch (java.io.IOException e) {
	    // hmm -- what is the best thing to do here?
	    System.out.println("OrbWaveformSink: fire() experienced IOexception: "+e);
	} catch (OrbErrorException e) {
	    System.out.println("OrbWaveformSink: fire() experienced OrbErrorException: "+e); 
	}

    }

    public Parameter orbname;
    public Parameter srcname;
    public Parameter nsamp;    // number of samples per packet
    public Parameter samprate;

    public TypedIOPort input;
    public TypedIOPort times;

    private Orb _orb = null;
    private int samplesBuffer[] = null;
    private int samplesSoFar = 0;
    private double packetTime = 0;
}
