/**
 *    '$RCSfile: SearchQueryGenerator.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2006/01/30 15:01:22 $'
 *   '$Revision: 1.13 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.ecogrid.quicksearch;

import java.util.Enumeration;
import java.util.Hashtable;

import javax.xml.transform.TransformerException;

import org.apache.axis.types.URI;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ecoinformatics.ecogrid.queryservice.query.QueryType;
import org.ecoinformatics.ecogrid.queryservice.util.EcogridQueryParser;
import org.ecoinformatics.seek.ecogrid.exception.InvalidEcogridQueryException;
import org.ecoinformatics.util.Config;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * This class will read kepler configure file and get the query part from it
 * base on search name space.
 * @author Jing Tao
 *  
 */

public class SearchQueryGenerator 
{
  private String       _queryId        = null;
  private Hashtable    _replacementMap = null;
  private QueryType    _query          = new QueryType();
  
  private final static String QUERYPATH   = "//ecogridService/queryList/query[@queryId='";
  private final static String CONDITION   = "condition";
  
  protected final static Log log;
  static {
    log = LogFactory.getLog( "org.ecoinformatics.seek.ecogrid.SearchQueryGenerator" );
  }
  /**
   * Constructor of SearchQueryGenerator
   * @param queryId   String  the queryId which will be found in config
   * @param replacementMap Hashtable  the hash table which contain the 
   *                                   key - be replaced value
   *                                   value - replacement
   * For example, if key is "#value#", and value is "soil". This means any
   * element and attribute in xml has value "#value#" will be replaced by soil
   */
  public SearchQueryGenerator(String    queryId, 
                              Hashtable replacementMap)
                                  throws  InvalidEcogridQueryException

  {
    _queryId        = queryId;
    _replacementMap = replacementMap;
    try
    {
      generateQuery();
    }
    catch(Exception e)
    {
      throw new InvalidEcogridQueryException(e.getMessage());
    }
  }//SearchQueryGenerator
  
  /**
   * Method to get query which generate by this class
   * @return QueryType
   */
  public QueryType getQuery()
  {
    return _query;
  }//getQuery
  
  /**
   * Recursively walks the tree looking for Condition values inorder to subsitute in the search value
   * @param aNode the parent node
   * @param aIsChildCond indicates whether the current parent node is a Condition node
   */
  private void mapInValue(Node aNode, boolean aIsChildCond, Hashtable aMap)
  {
      NodeList childList = aNode.getChildNodes();
      if ( childList == null)
      {
        return;
      }
      
      // go through every child element
      int length = childList.getLength();
      for (int i=0; i<length; i++)
      {
        Node kid = childList.item(i);
        if (kid.getNodeName().equals(CONDITION) || aIsChildCond)
        {
          String value = kid.getNodeValue();
          // replace the value by search value if this value in replacementMap
          if (value != null && aMap.containsKey(value))
          {
              log.debug("Replacing ["+value+"] with ["+aMap.get(value)+"]");
             kid.setNodeValue((String)aMap.get(value));
          } 
          else 
          {
            mapInValue(kid, true, aMap);
          }
        }
        else
        {
          mapInValue(kid, false, aMap);
        }
      }
  }
  
  /*
   * Method to read config file and generate a query(It will chose the first one
   * if it has more than one in configure file)
   */
  private void generateQuery() throws URI.MalformedURIException, 
                                      TransformerException,
                                      InvalidEcogridQueryException
  {
    String xpath   = QUERYPATH + _queryId+ "']";
    Node queryNode = null;
    NodeList list   = Config.getNodeListFromPath(xpath);
    if (list == null)
    {
      return;
    }
    
    // chose first query
    queryNode = list.item(0);
    NodeList queryChildrenList = queryNode.getChildNodes();
    if ( queryChildrenList == null)
    {
      return;
    }
    mapInValue(queryNode, false, _replacementMap);
    
    try {
        EcogridQueryParser queryParser = new EcogridQueryParser(queryNode);
        queryParser.parseXML();
        _query = queryParser.getEcogridQuery();
        //System.out.println(EcogridQueryTransformer.toXMLString(query));
        
        // Since the DOM that the query is created from is cached in memory we need to set back
        // the token that is searched for in order to insert the search string
        // otherwise we won't find the token next time. So just reverse the mapping hash and call it again.
        Hashtable reverseMap = new Hashtable();
        for (Enumeration e=_replacementMap.keys();e.hasMoreElements();)
        {
            String key = (String)e.nextElement();
            reverseMap.put(_replacementMap.get(key), key);
        }
        mapInValue(queryNode, false, reverseMap);

    } catch (Exception e)
    {
        log.error("Exception", e);
    }

  }  // generateQuery
   
}  // SearchQueryGenerator
