/**
 *    '$RCSfile: GrassExportActor.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2004/02/17 23:59:18 $'
 *   '$Revision: 1.1 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.gis.grass;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.io.IOException;

import ptolemy.data.StringToken;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.expr.Parameter;


/**
 * This actor exports a GRASS database object into an ARC file.
 */

public class GrassExportActor extends TypedAtomicActor
{
  //input ports
  public TypedIOPort dataType = new TypedIOPort(this, "dataType", true, false);
  public TypedIOPort outFileName = new TypedIOPort(this, "outFileName", true, false);
  public TypedIOPort inputObjectName = new TypedIOPort(this, "inputObjectName", true, false);
  public TypedIOPort trigger = new TypedIOPort(this, "trigger", false, true);

  public Parameter grass_gisbase =  new Parameter(this, "GRASS_GISBASE");
  public Parameter location_name = new Parameter(this, "LOCATION_NAME");
  public Parameter gisdbase = new Parameter(this, "GISDBASE");
  public Parameter gisrc = new Parameter(this, "GISRC");

  private String GRASS_GISBASE;
  private String LOCATION_NAME;
  private String GISDBASE;
  private String GISRC;

  /**
   * Grass Export Actor.
   */
  public GrassExportActor(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException
  {
    super(container, name);
    dataType.setTypeEquals(BaseType.STRING);
    outFileName.setTypeEquals(BaseType.STRING);
    inputObjectName.setTypeEquals(BaseType.STRING);
    trigger.setTypeEquals(BaseType.STRING);
  }

  /**
   *
   */
  public void initialize()
    throws IllegalActionException
  {
    if(grass_gisbase.getToken() != null)
    {
      GRASS_GISBASE = ((StringToken)grass_gisbase.getToken()).toString();
      //get rid of the quotes
      GRASS_GISBASE = GRASS_GISBASE.substring(1, GRASS_GISBASE.length() -1);
      System.out.println("GRASS_GISBASE: " + GRASS_GISBASE);
    }
    if(location_name.getToken() != null)
    {
      LOCATION_NAME = ((StringToken)location_name.getToken()).toString();
      //get rid of the quotes
      LOCATION_NAME = LOCATION_NAME.substring(1, LOCATION_NAME.length() -1);
      System.out.println("LOCATION_NAME: " + LOCATION_NAME);
    }
    if(gisdbase.getToken() != null)
    {
      GISDBASE = ((StringToken)gisdbase.getToken()).toString();
      //get rid of the quotes
      GISDBASE = GISDBASE.substring(1, GISDBASE.length() -1);
      System.out.println("GISDBASE: " + GISDBASE);
    }
    if(gisrc.getToken() != null)
    {
      GISRC = ((StringToken)gisrc.getToken()).toString();
      //get rid of the quotes
      GISRC = GISRC.substring(1, GISRC.length() -1);
      System.out.println("GISRC: " + GISRC);
    }

    if(GISRC == null ||
       GISDBASE == null ||
       LOCATION_NAME == null ||
       GRASS_GISBASE == null)
    {
      throw new IllegalActionException("The parameters GISRC, GISBASE, " +
        "LOCATION_NAME and GRASS_GISBASE must have valid values.  GISRC is " +
        "the path to your .grassrc file.  GISBASE is the path to your grass5" +
        "installation directory.  LOCATION_NAME is the name of the database " +
        "location within GRASS.  GRASS_GISBASE is the base directory for " +
        "your GRASS database.  Please provide these paths and re-execute.");
    }
  }

  /**
   *
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

  /**
   *
   */
  public void fire()
    throws IllegalActionException
  {
    System.out.println("firing GrassExportActor");
    super.fire();

    StringToken outFileToken = (StringToken)outFileName.get(0);
    String outFileStr = outFileToken.stringValue();

    StringToken inputObjectNameToken = (StringToken)inputObjectName.get(0);
    String  inputObjectNameStr = inputObjectNameToken.stringValue();

    StringToken dataTypeToken = (StringToken)dataType.get(0);
    String dataTypeStr = dataTypeToken.stringValue();

    String errorout = "";
    ///////////////////////////// IMPL CODE //////////////////////////////////

    try
    {
      final Process listener;
      String exportCommand = null;

      if ( dataTypeStr.equalsIgnoreCase("asc") )
      {
       exportCommand = "r.out.arc";
      }
      else if ( dataTypeStr.equalsIgnoreCase("shp") )
      {
       exportCommand = "r.out.shape";
      }
      else
      {
       exportCommand = "r.in.arc"; // default
      }

      String args[] = {this.GRASS_GISBASE + "/bin/" + exportCommand,
       "input=" + inputObjectNameStr, "output=" + outFileStr};

      String envvars[] = { "GISBASE=" + this.GRASS_GISBASE,
      "LOCATION_NAME=" + this.LOCATION_NAME,
      "GISDBASE=" + this.GISDBASE,
      "GISRC=" + this.GISRC};

      listener = Runtime.getRuntime().exec(args, envvars);

      new Thread(new Runnable() {
       public void run() {
         try {
           BufferedReader br_in = new BufferedReader(
             new InputStreamReader(listener.getInputStream()));
           BufferedReader br_err_in = new BufferedReader(
             new InputStreamReader(listener.getErrorStream()));
           String buff = null;
           while ((br_in != null && (buff = br_in.readLine()) != null))
           {
             System.out.println("Process out: " + buff);
             try {Thread.sleep(1); } catch(Exception e) {}
           }
           br_in.close();
           while ((br_err_in != null && (buff = br_err_in.readLine()) != null))
           {
             System.out.println("Process error out: " + buff);
             try {Thread.sleep(1); } catch(Exception e) {}
           }
           br_err_in.close();
         }
         catch (IOException ioe)
         {
           System.out.println("Exception caught printing result");
           ioe.printStackTrace();
         }
       }
      }).start();/*

      BufferedReader br_err_in = new BufferedReader(
       new InputStreamReader(listener.getErrorStream()));
      String buff = null;
      while ((br_err_in != null && (buff = br_err_in.readLine()) != null))
      {
        errorout += buff + "\n";
      }
      br_err_in.close();*/

      System.out.println("****" + errorout + "****");

      //thread to wait for grass process to terminate
      new Thread(new Runnable() {
       public void run() {
         try {
           System.out.println("*Thread waiting for Process to exit");
           listener.waitFor();
           System.out.println("*Thread detected Process exiting");
         } catch (InterruptedException ie) {
           System.out.println("InterruptedException caught whilst waiting "
                                       +"for Mozilla Process to exit: "+ie);
           ie.printStackTrace();
         }
       }
      }).start();
    }
    catch (Exception e)
    {
      e.printStackTrace();
      throw new IllegalActionException("Unexpected error: " + e.getMessage());
    }

    //////////////////////////////////////////////////////////////////////////

    /*if(errorout.indexOf("CREATING SUPPORT FILES FOR") != -1)
    {
      trigger.broadcast(new StringToken("SUCCESS"));
      System.out.println("Grass Import Action done");
    }
    else
    {
      throw new IllegalActionException("There was a problem running the " +
        "GRASS script: " + errorout);
    }*/
  }
}
