/**
 *    '$RCSfile: PortEditorDialog.java,v $'
 *
 *     '$Author: ruland $'
 *       '$Date: 2005/11/01 20:39:07 $'
 *   '$Revision: 1.3 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.ecoinformatics.seek.workflow.gui;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.border.Border;

import org.ecoinformatics.seek.workflow.ActorBuilder;
import org.ecoinformatics.seek.workflow.TypeObject;
import org.ecoinformatics.seek.workflow.TypedIOPortObject;
import org.ecoinformatics.seek.workflow.Types;

/**
 * This class builds a declaration statement for a TypedIOPort
 * (encapsulated in a TypedIOPortObject).  TypedIOPortObject builds
 * the constructor call of the form:
 * TypedIOPort(ComponentEntity container, java.lang.String name,
 *   boolean isInput, boolean isOutput)
 * Where container is set to "this", name is set to the name that the user
 * fills in in the form and isInput and isOutput are passed into
 * this classes constructor by the caller.  An example of the string
 * returned (via TypedIOPortObject.toString()) would be
 * TypedIOPort(this, "input1", true, false)
 * The caller can pass the TypedIOPortObject back to this class to be edited
 * and returned.
 */
public class PortEditorDialog extends JDialog
{
  //Layout constants
  private final static int PAD = 5;
  private final static int HORIZONTAL = 300;
  private final static int SPACE = 3;
  private final static int WIDE_SPACE = 10;
  private final static int COLUMNS = 25;
  private final static int ROWS = 4;

  private final static int INPUT = 0;
  private final static int OUTPUT = 1;

  // PRIVATE MEMBERS
  private int itemType;
  private JComboBox typeDimensionCombo;
  private JComboBox typeCombo;
  private JTextField nameField;
  private JButton cancelButton;
  private JButton saveButton;
  private boolean cancelled;
  private JFrame parent;
  private TypedIOPortObject port = null;
  private boolean isInput = false;
  private boolean isOutput = false;

  /**
   * Creates a new port if port is null, or uses the existing on if it isn't
   * @param isInput true if the port to be created is an input port
   * @param isOutput true if the port to be created is an output port
   * @param  parent  the parent frame for this dialog
   */
  public PortEditorDialog(JFrame parent, boolean isInput, boolean isOutput,
    TypedIOPortObject port)
  {
    super(parent);
    this.isInput = isInput;
    this.isOutput = isOutput;
    this.parent = parent;
    this.port = port;
    initializeFields();
    layoutFields();
  }

  /**
   * edit an existing port
   * @param parent the parent frame of this dialog
   * @param port the port to edit
   */
  public PortEditorDialog(JFrame parent, TypedIOPortObject port)
  {
    this(parent, port.getIsInput(), port.getIsOutput(), port);
  }

  /**
   * Create a new port
   */
  public PortEditorDialog(JFrame parent, boolean isInput, boolean isOutput)
  {
    this(parent, isInput, isOutput, null);
  }

  /**
   * return the newly created port.
   */
  public TypedIOPortObject getPort()
  {
    return port;
  }

  /**
   * Get the cancelled flag.
   *
   * @return    boolean true if the dialog was dismissed using the Cancel button
   */
  public boolean isCancelled()
  {
    return cancelled;
  }

  /**
   * Initialize the fields for the dialog to default values.
   */
  private void initializeFields()
  {
    Object[] dimensionTypes = {Types.SCALAR, Types.MATRIX};
    Object[] types = {Types.BOOLEAN, Types.COMPLEX, Types.DOUBLE, Types.INT,
      Types.LONG, Types.STRING, Types.OBJECT,
      Types.RECORD, Types.UNSIGNEDBYTE, Types.NUMERICAL, Types.GENERAL};

    typeDimensionCombo = new JComboBox(dimensionTypes);
    typeDimensionCombo.setEditable(false);
    typeCombo = new JComboBox(types);
    typeCombo.setEditable(false);
    nameField = new JTextField(COLUMNS);
  }

  /**
   * Create the graphical components of the dialog.
   */
  private void layoutFields()
  {
    setModal(true);
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    setTitle("Port Editor");
    cancelled = false;

    Container content = getContentPane();

    //content.setBackground(Color.white);
    content.setLayout(new BoxLayout(getContentPane(), BoxLayout.X_AXIS));
    content.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));

    JPanel editPane = new JPanel();

    editPane.setLayout(new BoxLayout(editPane, BoxLayout.Y_AXIS));
    //editPane.setBackground(Color.white);
    editPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    //type name
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));
    editPane.add(createPanel(nameField, "Name:"));

    // Type chooser
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));
    editPane.add(createPanel(typeCombo, "Type:"));

    // Type dimension chooser
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));
    editPane.add(createPanel(typeDimensionCombo, "Type dimension:"));

    if(port != null)
    {
      nameField.setText(port.getName());
      typeCombo.setSelectedItem(port.getType().getType());
      typeDimensionCombo.setSelectedItem(port.getType().getTypeDimension());
    }

    cancelButton = new JButton("Cancel");
    saveButton = new JButton("Save");
    editPane.add(Box.createRigidArea(new Dimension(PAD, WIDE_SPACE)));

    JPanel buttonPanel = new JPanel();

    buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
    buttonPanel.add(Box.createHorizontalGlue());
    buttonPanel.add(cancelButton);
    buttonPanel.add(Box.createRigidArea(new Dimension(PAD, PAD)));
    buttonPanel.add(saveButton);
    editPane.add(buttonPanel);

    editPane.add(Box.createRigidArea(new Dimension(PAD, PAD)));

    content.add(editPane);
    content.add(Box.createRigidArea(new Dimension(WIDE_SPACE, PAD)));

    // register a listener for ActionEvents
    ActionHandler myActionHandler = new ActionHandler();

    cancelButton.addActionListener(myActionHandler);
    saveButton.addActionListener(myActionHandler);

    pack();

    //set the position of the window
    int x = parent.getX();
    int y = parent.getY();
    int width = parent.getWidth();
    int height = parent.getHeight();
    setLocation(x + ((width / 2) - (this.getWidth() / 2)),
                y + ((height / 2) - (this.getHeight() / 2)));

    setVisible(true);
  }

  /**
   * Create a JPanel with a horizontal layout, placing the component
   * and its field horizontally next to one another for text fields, and
   * as a titled border for text areas.
   *
   * @param  comp   the component to be placed on the panel
   * @param  label  the label for the component
   * @return        Description of the Returned Value
   */
  private JPanel createPanel(JComponent comp, String label)
  {
    JPanel pane = new JPanel();

    pane.setLayout(new BoxLayout(pane, BoxLayout.X_AXIS));
    //pane.setBackground(Color.white);

    if(comp instanceof JTextArea)
    {
      JScrollPane scroll = new JScrollPane(comp);
      //scroll.setBackground(Color.white);
      Border border = BorderFactory.createTitledBorder(
          BorderFactory.createLineBorder(Color.black), label);

      scroll.setBorder(border);
      pane.add(scroll);
    }
    else if((comp instanceof JTextField) ||
        (comp instanceof JComboBox))
    {
      JLabel fieldLabel = new JLabel(label);

      pane.add(fieldLabel);
      pane.add(Box.createRigidArea(new Dimension(PAD, SPACE)));
      pane.add(comp);
    }

    return pane;
  }

  /**
   * Perform actions associated with the save button
   *
   * @param  event  Description of Parameter
   */
  private void saveButtonHandler(ActionEvent event)
  {
    String name = nameField.getText();
    if(name == null || name.trim().equals(""))
    {
      JOptionPane.showMessageDialog(null,
        "Please fill out the name of the port.",
        "Error", JOptionPane.ERROR_MESSAGE);
      return;
    }

    name = ActorBuilder.processName(name);

    TypeObject typeObj;
    try
    {
      typeObj = new TypeObject((String)typeCombo.getSelectedItem(),
        (String)typeDimensionCombo.getSelectedItem());
    }
    catch(Exception e)
    {
      JOptionPane.showMessageDialog(null,
        e.getMessage(),
        "Error", JOptionPane.ERROR_MESSAGE);
      return;
    }

    port = new TypedIOPortObject(name, typeObj, isInput,
      isOutput);

    setVisible(false);
  }

  /**
   * Perform actions associated with the cancel button
   *
   * @param  event  Description of Parameter
   */
  private void cancelButtonHandler(ActionEvent event)
  {
    cancelled = true;
    dispose();
  }


  /**
   * Listener used to detect button presses
   */
  private class ActionHandler implements ActionListener
  {
    /**
     *  Description of the Method
     *
     * @param  event  Description of Parameter
     */
    public void actionPerformed(ActionEvent event)
    {
      Object object = event.getSource();

      if(object == saveButton)
      {
        saveButtonHandler(event);
      }
      else if(object == cancelButton)
      {
        cancelButtonHandler(event);
      }
    }
  }
}

