/* A database connection actor...

@Copyright (c) 2002-2003 The Regents of the University of California.
All rights reserved.

Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the
above copyright notice and the following two paragraphs appear in all
copies of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

                                                PT_COPYRIGHT_VERSION 2
                                                COPYRIGHTENDKEY
*/

package org.geon;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;

import ptolemy.actor.IOPort;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.DBConnectionToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Entity;
import ptolemy.kernel.Port;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;
import util.DBUtil;

//////////////////////////////////////////////////////////////////////////
//// OpenDBConnection
/**
This actor opens a database connection using the database format, database
URL, username and password, and sends a reference to it.

@author Efrat Jaeger
@version $Id: OpenDBConnection.java,v 1.20 2006/04/04 17:23:49 altintas Exp $
@since Ptolemy II 3.0.2
*/
public class OpenDBConnection extends TypedAtomicActor {

 /** Construct an actor with the given container and name.
 *  @param container The container.
 *  @param name The name of this actor.
 *  @exception IllegalActionException If the actor cannot be contained
 *   by the proposed container.
 *  @exception NameDuplicationException If the container already has an
 *   actor with this name.
 */

public OpenDBConnection(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

    super(container, name);

    dbcon = new TypedIOPort(this, "dbcon", false, true);
    // Set the type constraint.
    dbcon.setTypeEquals(BaseType.DBCONNECTION);

//    catalog = new StringAttribute(this, "catalog");
//    catalog.setExpression("");
//    _catalog = _none;

    databaseFormat = new StringParameter(this, "database format");
    databaseFormat.setExpression("Oracle");
    databaseFormat.addChoice("Oracle");
    databaseFormat.addChoice("DB2");
    databaseFormat.addChoice("Local MS Access");
    databaseFormat.addChoice("Remote MS Access (using objectweb server)");
    databaseFormat.addChoice("MS SQL Server");
    databaseFormat.addChoice("PostgreSQL");
    databaseFormat.addChoice("MySQL");
    databaseFormat.addChoice("Sybase SQL Anywhere");
    _dbFormat = _ORCL;

//    driverName = new StringAttribute(this, "driverName");
    databaseURL = new StringAttribute(this, "databaseURL");
    username = new StringParameter(this, "username");
    password = new StringParameter(this, "password");

    _attachText("_iconDescription", "<svg>\n"
            + "<ellipse cx=\"0\" cy=\"-30\" "
            + "rx=\"20\" ry=\"10\"/>\n"
            + "<line x1=\"20\" y1=\"0\" "
            + "x2=\"20\" y2=\"-30\"/>\n"
            + "<line x1=\"-20\" y1=\"0\" "
            + "x2=\"-20\" y2=\"-30\"/>\n"
            + "<line x1=\"-20\" y1=\"0\" "
            + "x2=\"20\" y2=\"0\"/>\n"
            + "</svg>\n");

}

  ///////////////////////////////////////////////////////////////////
  ////                     ports and parameters                  ////

  /** A reference to a db connection
   */

  public TypedIOPort dbcon;

//  public StringAttribute catalog;
  public StringParameter databaseFormat;
//  public StringAttribute driverName;
  public StringAttribute databaseURL;
  public StringParameter username;
  public StringParameter password;

  public String strFileOrURL;

  ///////////////////////////////////////////////////////////////////
  ////                        public methods                     ////

  /** Callback for changes in attribute values.
   *  If the dbFormat has changed, points to the new driver path.
   *  At any change in the connection attributes, tries to connect to the 
   *  referenced database to get the schema and send it to connected actors.
   * 
   *  @param at The attribute that changed.
   *  @exception IllegalActionException If the offsets array is not
   *   nondecreasing and nonnegative.
   */
  public void attributeChanged(Attribute at) throws IllegalActionException {
  	      if (at == databaseFormat) {
              String dbFormat = databaseFormat.stringValue();
              _driverName = DBUtil.get(dbFormat.trim().toLowerCase());
              if (dbFormat.equals("Oracle")) {
                  _dbFormat = _ORCL;
              } else if (dbFormat.equals("DB2")) {
                  _dbFormat = _DB2;
              } else if (dbFormat.equals("Local MS Access")) {
                  _dbFormat = _LACCS;
              } else if (dbFormat.startsWith("Remote MS Access")) {
              	  _dbFormat = _RACCS;
              } else if (dbFormat.equals("MS SQL Server")) {
                  _dbFormat = _MSSQL;
              } else if (dbFormat.equals("PostgreSQL")) {
                  _dbFormat = _PGSQL;
              } else if (dbFormat.equals("MySQL")) {
              	_dbFormat = _MYSQL;
//              } else if (dbFormat.equals("Sybase SQL Anywhere")) {
              	// TODO: To be added..
  //                _dbFormat = _SYBASE;
              } else {
                  throw new IllegalActionException(this,
                          "No jdbc driver within the system for " + dbFormat);
              }
              if (!_driverName.equals(this._prevDriver)) {
              	_setDBURL();
              	_prevDriver = _driverName;
              	_getAndSendSchema();              
              }
          } else if (at == databaseURL) {
              _setDBURL();
              if (!_databaseURL.equals(_prevDBURL)) {
              	_prevDBURL = _databaseURL;
              	_getAndSendSchema();              
              }
          } else if (at == username) {
              _username = username.stringValue();
              if (!_username.equals(_prevUser)) {
              	_prevUser = _username;
              	_getAndSendSchema();              
              }              
          } else if (at == password) {
              _password = password.stringValue();
              if (!_password.equals(_prevPasswd)) {
              	_prevPasswd = _password;
              	_getAndSendSchema();              
              }              

          } else {
              super.attributeChanged(at);
          }
   }

  
  /**
   * When connecting the dbcon port, trigger the connectionsChanged of 
   * the connected actor.
   */
  public void connectionsChanged(Port port) {
      super.connectionsChanged(port);
      if (port == dbcon) {
      	List conPortsList = dbcon.connectedPortList();
      	Iterator conPorts = conPortsList.iterator();
      	while (conPorts.hasNext()) {
      		IOPort p = (IOPort) conPorts.next();
      		if (p.isInput()) {
      			Entity container = (Entity)p.getContainer();
      			container.connectionsChanged(p);
      		}
      	}
      }
  }

  
  /** Connect to a database
   * outputs a reference to the DB connection.
   */

    public void fire() throws IllegalActionException {
      try {

        //String _username = username.stringValue();
        //String _password = password.stringValue();

        //_setDBURL();

        Connection con = _connect("fire");
        dbcon.broadcast(new DBConnectionToken(con));

      }
      catch (Exception ex) {
        throw new IllegalActionException(this, ex, "fire exception DB connection");
      }
    }
    
    /** postfiring the actor.
     * 
     */
    public boolean postfire() {
      return false;
    }
    

    /** Returns the database schema. This function is called from the 
     *  dbcon port connected actors.
     */ 
    public String sendSchemaToConnected() {
    	if (_schema.equals("")) {
    		try {
    			_schema = _getSchema();
    		} catch (Exception ex) {}
    	}
    	return _schema;
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private methods                   ////

    /** Connecting to the database and returning a database connection reference.
     *  The context is either "fire" or "other", if unable to connect from a "fire" 
     *  context throws an exception (for connecting from other context it might be 
     * 	the case that not all the connection attributes have already been set).
     * 
     * @param context
     * @return
     * @throws IllegalActionException
     */
    private Connection _connect(String context) throws IllegalActionException {
        Connection con = null;
        try {
        Class.forName(_driverName).newInstance();

        con = DriverManager.getConnection(_databaseURL,
                                                   _username,
                                                   _password);
        } catch (Exception ex) {
            if (context.equals("fire"))
                throw new IllegalActionException(this, ex, "fire exception DB connection");
        }
        return con;
    }


    /** Called from attributeChanged. Once a connection attribute has been changed
     *  tries to connect to the database, get the schema and forward it to dbcon 
     *  port connected actors.
     *  
     * @throws IllegalActionException
     */ 
    private void _getAndSendSchema() throws IllegalActionException {
      	_getSchema();

      	// send schema to connected
      	if (!_schema.equals(""))
      		connectionsChanged(dbcon);
    }
    
    /** Gets the database schema in an XML format readable by the query builder.
     * 
     * @param con - the database connection object.
     */
    private void _getDBSchema(Connection con) {
    	StringBuffer schema = new StringBuffer();
    	schema.append("<schema>\n");
    	StringBuffer table = new StringBuffer();
    	String prevTable = "";
    	String prevSchema = "";
        int numTables = 0;
        int numFields = 0;
        
    	try {
      		DatabaseMetaData dmd = con.getMetaData();
      		ResultSet schemas = dmd.getSchemas();
      		
      		while (schemas.next()) {
      			String schemaName = schemas.getString(1);
      		
      			// Ignore system schemas.
      			if (_dbFormat == _ORCL) {
        			// schema will be the same as username.
        			if (!schemaName.toLowerCase().equals(_username)) 
        				continue;
        		} else if (schemaName.toLowerCase().startsWith("sys")) 
        	    	continue;

      			ResultSet rs = dmd.getColumns(null, schemaName,"%", "%");
      		
      		//	ResultSetMetaData md = rs.getMetaData();
      			while (rs.next()) {

      				//String schemaName = rs.getString(2);
	        	    String tableName = rs.getString(3);
	        	    String columnName = rs.getString(4);
	        	    String columnType = rs.getString(6);
	        	    
	        	    if (tableName.equals("")) continue;
	        	    if (!tableName.equals(prevTable) || !schemaName.equals(prevSchema)) {
	        	    	// new table, closing a previous one if exists
	        	        if (numFields > 0) {
	                		table.append("  </table>\n");
	            			numTables++;
	            			schema.append(table.toString());
	            		}
	
	            		table = new StringBuffer();
	            		table.append("  <table name=\""); 
	            		if (!schemaName.toLowerCase().equals("null")) {
	            		    table.append(schemaName + ".");
	            		}
	            		table.append(tableName + "\">\n");
	            		numFields = 0;
	        	        prevTable = tableName;
	        	        prevSchema = schemaName;
	        	    }
	    			if (columnName.equals("")) continue;
	    			else {
	    			    table.append("    <field name=\"" + columnName + "\" dataType=\"" + columnType + "\"/>\n");
	    				numFields++;
	    			}
	        	}
      		}
        	table.append("  </table>\n");
        	if (numFields > 0) {
        	    numTables++;
        	    schema.append(table.toString());
        	}
            schema.append("</schema>");
            if (numTables > 0) {
            	_schema = schema.toString();
            }
    	} catch (Exception ex) {
    	    _schema = "";
    	}
    }

    
    /** Connects to the database and get the schema.
     * 
     * @return database schema string
     * @throws IllegalActionException
     */
    private String _getSchema() throws IllegalActionException {
        Connection con =_connect("other");
    	if (con != null) {
    		_getDBSchema(con);
    		try {
    			con.close();
    		} catch (SQLException e) {
    		    con = null;
    		}
    		if (!_schema.equals("")) 
    			return _schema;
    	}
    	return "";
    }
    
    /** Set the absolute database URL depending on the database driver.
     * 
     * @throws IllegalActionException
     */
    private void _setDBURL() throws IllegalActionException {
        _databaseURL = databaseURL.getExpression();
        switch(_dbFormat) {
	        case _ORCL:
	        	if (!_databaseURL.trim().startsWith("jdbc:oracle:thin:@")) {
	        		if (_databaseURL.trim().startsWith("jdbc:oracle:")) {// a different driver type is spcified.
	        			int ind = _databaseURL.indexOf("@");
	        			if (ind > -1) {
	        				_databaseURL = "jdbc:oracle:thin:@" + _databaseURL.substring(ind);
	        	  	    } else throw new IllegalActionException(this,
	        	  	    		"Illegal database URL: " + _databaseURL);
	        		} else {
	        			_databaseURL = "jdbc:oracle:thin:@" + _databaseURL;
	        		}
	        	}
	            break;
	        case _DB2:
	        	if (!_databaseURL.trim().startsWith("jdbc:db2:")) {
	        	  	  _databaseURL = "jdbc:db2:" + _databaseURL;
	        	}
	            break;
	        case _LACCS:
	      	  	if (!_databaseURL.trim().startsWith("jdbc:odbc:")) {
	      	  		_databaseURL = "jdbc:odbc:" + _databaseURL;
	      	  	}
	      	  	break;
	        case _RACCS:
                //TODO: TAKE CARE OF HOW TO SPECIFY DRIVER FOR REMOTE ACCESS DB.
	        	break;
	        case _MSSQL:
	        	if (!_databaseURL.trim().startsWith("jdbc:microsoft:sqlserver:")) {
	        		_databaseURL = "jdbc:microsoft:sqlserver:" + _databaseURL;
	        	}
	        	_databaseURL = _databaseURL + ";User=" + _username + ";Password=" + _password;
	        	//_username = null; _password = null;
	            break;
	        case _PGSQL:
	        	if (!_databaseURL.trim().startsWith("jdbc:postgresql:")) {
	        	  	  _databaseURL = "jdbc:postgresql:" + _databaseURL;
	        	}
	            break;
	        case _MYSQL:
	        	if (!_databaseURL.trim().startsWith("jdbc:mysql:")) {
	        		_databaseURL = "jdbc:mysql:" + _databaseURL;
	        	}
	            break;
	        case _SYBASE:
	        	// TODO: To be added..
	            break;
	        default:
	            System.out.println(databaseFormat.getExpression() + " is not supported");
        }
    }

    ///////////////////////////////////////////////////////////////////
    ////                         private variables                 ////
    
    // An indicator for the db format.
    private int _dbFormat;
    private String _databaseURL = "";
    private String _driverName = "";
    private String _username = "";
    private String _password = "";
   
    
    // Saving previous values to track changes. 
    private String _prevDriver = "";
    private String _prevDBURL = "";
    private String _prevUser = ""; 
    private String _prevPasswd = "";
    
    // the database schema
    private String _schema = "";

    // Constants used for more efficient execution.
    private static final int _ORCL = 0;
    private static final int _DB2 = 1;
    private static final int _LACCS = 2;
    private static final int _RACCS = 3;
    private static final int _MSSQL = 4;
    private static final int _PGSQL = 5;
    private static final int _MYSQL = 6;
    private static final int _SYBASE = 7;

}
