/* A class that initializes Vergil for Kepler.

 Copyright (c) 2006 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

 PT_COPYRIGHT_VERSION_2
 COPYRIGHTENDKEY
 */
package org.kepler.gui;

import java.io.File;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ecoinformatics.util.Config;
import org.ecoinformatics.util.DBConnection;
import org.ecoinformatics.util.DBConnectionFactory;

import ptolemy.data.DBConnectionToken;
import ptolemy.data.expr.Constants;


//////////////////////////////////////////////////////////////////////////
//// KeplerInitializer

/**
 Initialize Kepler.  This example class does nothing, though applications
 could do things like add constants.

 <p>How this works is that
 configs/ptolemy/configs/kepler/configuration.xml
 sets a
 StringParameter called "_applicationInitializer" that names this
 class.  KeplerApplication reads the parameter and instantiates this class.

 @author Christopher Brooks
 @version $Id: KeplerInitializer.java,v 1.5 2006/02/13 15:09:05 ruland Exp $
 @since Ptolemy II 6.0
 @Pt.ProposedRating Red (cxh)
 @Pt.AcceptedRating Red (cxh)
 */
public class KeplerInitializer {

	private static Log log = LogFactory.getLog( "org.kepler.gui.KeplerInitialization" );
	private static boolean hasBeenInitialized = false;
	
	/** Perform any Kepler specific initialization. */
    public KeplerInitializer() throws Exception {
    	initializeSystem();

    }

	public static void initializeSystem() throws Exception {
		
		// Only run initialization once.
		if (hasBeenInitialized) {
			return;
		}
		
		hasBeenInitialized = true;
		
		// Add the dbconnection type.
        Constants.add("dbconnection", new DBConnectionToken());

        Config c = Config.getInstance();
		
        
        //
		// First get the entries named mkdir.
		// We will make directories for each of these entries under the UserDir.
		//
		List mkdirs = c.getList( "//startup/mkdir" );
		
		for( Iterator i = mkdirs.iterator(); i.hasNext(); ) {
			String dir = (String) i.next();
			if ( dir == null || dir.length() == 0 ) {
				continue;
			}
			dir = c.getUserDirPath( dir );
			File f = new File( dir );
			f.mkdirs();
			log.info( "Making directory " + dir );
		}
		
		DBConnection conn = DBConnectionFactory.getDBConnection();
		
		//
		// Get the tabletestsql entry.  This is the sql used to test if a
		// table already exists in the database.
		//
		String tabletestsql = Config.getValue("//startup/tabletestsql");
		PreparedStatement tabletest = null;
		if ( tabletestsql != null ) {
			tabletest = conn.prepareStatement( tabletestsql );
		}
		
		// We use pattern matching to extract the tablename from the ddl statement.
		// This is a Java 1.4 regex regular expression which
		// extracts the word after the string "table".  The match is case insensitive so
		// "table" will also match "TABLE".
		// The "(" ")" characters denote a grouping.  This will be returned as group 1.
		Pattern extractTable = Pattern.compile( "table\\s+(\\w*)", Pattern.CASE_INSENSITIVE );

		// Get the list of ddl statements defining the tables to create.
		List createtables = Config.getList( "//startup/createtable" );
		for( Iterator i = createtables.iterator(); i.hasNext(); ) {
			String ddl = (String) i.next();
			// Create our Matcher object for the ddl string.
			Matcher m = extractTable.matcher(ddl);
			// Materh.find() looks for any match of the pattern in the string.
			m.find();
			
			String tablename = m.group(1);
			if ( tabletest == null ) {
				log.error("unable to test for table: " + tablename );
				continue;
			}
			
			tabletest.setString(1,tablename);
			ResultSet rs = tabletest.executeQuery();
			if ( rs.next() ) {
				log.debug("Table " + tablename + " already exists");
				continue;
			}
			conn.executeSQLCommand(ddl);
			log.info("Table " + tablename + " created");
		}
		
		//
		// Hook to execute arbitrary sql statements on the internal database.
		List sqls = c.getList( "//startup/sql" );
		for( Iterator i = sqls.iterator(); i.hasNext(); ) {
			String statement = (String) i.next();
			log.info("Executing sql command " + statement);
			conn.executeSQLCommand( statement);
		}
        
        conn.close();
	}
	
}
