/**
 *  '$RCSfile: TabbedLibraryPane.java,v $'
 *  '$Author: brooke $'
 *  '$Date: 2006/02/13 19:48:46 $'
 *  '$Revision: 1.16 $'
 *
 *  For Details:
 *  http://kepler.ecoinformatics.org Copyright (c) 2004 The Regents of the
 *  University of California. All rights reserved. Permission is hereby granted,
 *  without written agreement and without license or royalty fees, to use, copy,
 *  modify, and distribute this software and its documentation for any purpose,
 *  provided that the above copyright notice and the following two paragraphs
 *  appear in all copies of this software. IN NO EVENT SHALL THE UNIVERSITY OF
 *  CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL,
 *  OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS
 *  DOCUMENTATION, EVEN IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY
 *  DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE
 *  SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.gui;


import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;

import org.ecoinformatics.seek.ecogrid.EcoGridQueryServicesController;
import org.ecoinformatics.seek.ecogrid.quicksearch.DatasetPanel;

import ptolemy.actor.gui.Configuration;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.NamedObj;
import ptolemy.util.MessageHandler;
import ptolemy.vergil.tree.EntityTreeModel;

import org.kepler.objectmanager.library.LibraryIndex;


/**
 *  A simple pane used to display the contents of the library tree using a
 *  tabbed pane to provide additional resources in the library.
 *
 *@author     Matt Jones
 *@created    February 17, 2005
 *@version    $Id: TabbedLibraryPane.java,v 1.16 2006/02/13 19:48:46 brooke Exp $
 *@since      Kepler 1.0
 */
public class TabbedLibraryPane extends LibraryPane {

/**
   *  Construct a new library pane for displaying the tree of actors and other
   *  resourses that can be dragged to the graph editor.
   *
   *@param  _libraryModel  the model containing the library to be displayed
   *@param  config         Description of the Parameter
   */
  public TabbedLibraryPane(EntityTreeModel _libraryModel, Configuration config) {
    super();

    try
    {
      //get the tree model from the LibraryIndex
      LibraryIndex libIndex = LibraryIndex.getInstance();
      _libraryModel = libIndex.createLibrary();
      _library = new AnnotatedPTree(_libraryModel, this);
      libIndex.addLibraryComponent(_library);
    }
    catch(IllegalActionException iae)
    {
      try {
        MessageHandler.warning("Could not set the EntityTreeModel from " +
          "the LibraryIndex: ", iae);
      } catch (Exception ce) {
        // Do nothing
      }
    }

    this.setLayout(new BorderLayout());

//    _library = new AnnotatedPTree(_libraryModel, this);

    LibraryActionHandler handler = new LibraryActionHandler();
    _library.addListener(handler);
    _library.setRootVisible(false);

    //create the library search results panel from the config
    LibrarySearchResultPaneFactory SRPfactory =
      (LibrarySearchResultPaneFactory)
      config.getAttribute("librarySearchResultPane");
    if (SRPfactory != null) {
      try {
        _libraryPanel =
          SRPfactory.createLibrarySearchResultPane(_library, null);
        if (_libraryPanel == null) {
          System.out.println("error: _libraryPanel is null.  " +
                             "This " +
                             "problem can be fixed by adding a librarySearcherFactory " +
                             "property in the configuration.xml file.");
        }
      } catch (IllegalActionException iae) {
        System.out.println("Error displaying search results: " + iae);
      }
    } else {
      System.out.println("error: librarySearchResultPane.factory is " +
                         "null.  This " +
                         "problem can be fixed by adding a librarySearcherFactory " +
                         "property in the configuration.xml file.");
    }

    //create the search GUI pane
    LibrarySearchGUIPaneFactory LSGfactory =
      (LibrarySearchGUIPaneFactory)
      config.getAttribute("librarySearchGUIPane");
    if (LSGfactory != null) {
      _searchPane =
        LSGfactory.createLibrarySearchGUIPane(searchButtonHandler);
      if (_searchPane == null) {
        System.out.println("error: _searchPane is null.  " +
                           "This " +
                           "problem can be fixed by adding a librarySearchGUIPane " +
                           "property in the configuration.xml file.");
      }
    } else {
      System.out.println("error: LibrarySearchGUIPane.factory is " +
                         "null.  This " +
                         "problem can be fixed by adding a librarySearchGUIPane " +
                         "property in the configuration.xml file.");
    }

    //create the librarySearcher from the configuration
    LibrarySearcherFactory LSfactory = (LibrarySearcherFactory)
                                       config.getAttribute(
      "librarySearcherFactory");
    if (LSfactory != null) {
      librarySearcher = LSfactory.createLibrarySearcher(_library, _searchPane);
      if (librarySearcher == null) {
        System.out.println("error: librarySearcher is null.  This " +
                           "problem can be fixed by adding a librarySearcherFactory " +
                           "property in the configuration.xml file.");
      }
    } else {
      System.out.println("error: librarySearcher.factory is null.  This " +
                         "problem can be fixed by adding a librarySearcherFactory " +
                         "property in the configuration.xml file.");
    }

    JPanel librarySearchPanel = new JPanel();

    librarySearchPanel.setLayout(topLayout);

    final int LIB_PANEL_WIDTH = _searchPane.getMinimumWidth();

    librarySearchPanel.setMinimumSize(new Dimension(LIB_PANEL_WIDTH, 500));
    librarySearchPanel.setPreferredSize(new Dimension(LIB_PANEL_WIDTH, 500));
    //create and add the search panel to the library

    _libraryPanel.setPreferredSize(new Dimension(LIB_PANEL_WIDTH, 400));
    librarySearchPanel.add(_searchPane, BorderLayout.NORTH);
    librarySearchPanel.add(_libraryPanel, BorderLayout.CENTER);

    searchPartPanel.setPreferredSize(new Dimension(LIB_PANEL_WIDTH, 100));
    searchPartPanel.setMinimumSize(new Dimension(LIB_PANEL_WIDTH, 75));

    _libraryTabPane = new JTabbedPane(JTabbedPane.TOP);

    _libraryTabPane.add(COMPONENTS_TAB_TEXT, librarySearchPanel);

    try {
      EcoGridQueryServicesController controller
        = EcoGridQueryServicesController.getInstance();
      DatasetPanel _datasetPanel = new DatasetPanel(controller);

      //_datasetPanel.setEcoGridServiceController(controller);

      _libraryTabPane.add(DATA_TAB_TEXT, _datasetPanel);
    } catch (Exception e) {
      try {
        MessageHandler.warning("Could not create data pane.", e);
      } catch (Exception ce) {
        // Do nothing
      }
    }

    this.add(_libraryTabPane, BorderLayout.CENTER);
  }


  /**
   *  A factory that creates the library panel for the editors.
   *
   *@author     berkley
   *@created    February 17, 2005
   */
  public static class Factory extends LibraryPaneFactory {
    /**
     *  Create an factory with the given name and container.
     *
     *@param  container                     The container.
     *@param  name                          The name of the entity.
     *@exception  IllegalActionException    If the container is incompatible
     *      with this attribute.
     *@exception  NameDuplicationException  If the name coincides with an
     *      attribute already in the container.
     */
    public Factory(NamedObj container, String name) throws
      IllegalActionException, NameDuplicationException {
      super(container, name);
    }


    /**
     *  Create a library pane that displays the given library of actors.
     *
     *@param  _libraryModel  the model containing the actor library
     *@param  config         Description of the Parameter
     *@return                A new LibraryPane that displays the library
     */
    public LibraryPane createLibraryPane(EntityTreeModel _libraryModel,
                                         Configuration config) {
      TabbedLibraryPane pane = new TabbedLibraryPane(_libraryModel, config);
      return pane;
    }
  }


  /**
   *  class to handle the search button
   *
   *@author     berkley
   *@created    February 17, 2005
   */
  public class SearchButtonActionHandler implements ActionListener {
    /**
     *  Description of the Method
     *
     *@param  e  Description of the Parameter
     */
    public void actionPerformed(ActionEvent e) {
      try {
        _libraryPanel.update(librarySearcher.search(_searchPane.getSearchTerm()));
      } catch (IllegalActionException iae) {
        //FIXME: handle this exception correctly
        System.out.println("Error searching: " + iae);
        iae.printStackTrace();
      }
    }
  }


  ///////////////////////////////////////////////////////////////////
  ////                         private variables                 ////
  ///////////////////////////////////////////////////////////////////

  // FIXME: Comments are needed on all these.
  // FIXME: Need to be in alphabetical order.

  private final String COMPONENTS_TAB_TEXT = "Components";
  //spaces needed as padding to make tab as wide as "Components" tab
  private final String DATA_TAB_TEXT = "  Data  ";

  private AnnotatedPTree _library;
  private LibrarySearchResultPane _libraryPanel;
  private JTabbedPane _libraryTabPane;
  private JPanel searchPartPanel = new JPanel();
  private LibrarySearcher librarySearcher;
  private BorderLayout topLayout = new BorderLayout();
  private LibrarySearchPane _searchPane;
  private SearchButtonActionHandler searchButtonHandler =
    new SearchButtonActionHandler();

  ///////////////////////////////////////////////////////////////////
  ////                         private classes                   ////
  ///////////////////////////////////////////////////////////////////

  /**
   * Listener used to changes from the NewFolderFrame
   */
  private class LibraryActionHandler implements ActionListener {
    /**
     *  Description of the Method
     *
     * @param  event  Description of Parameter
     */
    public void actionPerformed(ActionEvent event) {
      String command = event.getActionCommand();
      if (command.equals("new_folder_created") ||
          command.equals("new_actor_created")) {
        //_library.clearSelection(); //clear the current selections
        //LibrarySearchResultPane.collapseAll(_library);
        /*TreeAction action = (TreeAction)event.getSource();
                 TreePath path = action.getPath();
                 _library.addSelectionPath(path);
                 System.out.println("path expanded.");
                 //LibrarySearchResultPane.expandAll(_library);
                 _library.expandPath(path);*/
      }
    }
  }
}
