/**
 *    '$RCSfile: ActorCacheObject.java,v $'
 *
 *     '$Author: berkley $'
 *       '$Date: 2006/02/23 02:25:01 $'
 *   '$Revision: 1.10 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2003 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.objectmanager.cache;

import org.kepler.objectmanager.lsid.KeplerLSID;
import org.kepler.objectmanager.ActorMetadata;

import java.io.*;
import java.util.*;

/**
 * Class that represents an object in the ObjectCache.  This class should
 * be extended by each type of object that wants to control its own lifecycle
 * events and serialization events.
 */
public class ActorCacheObject extends CacheObject implements Externalizable
{
  private ActorMetadata actor;
  
  public ActorCacheObject()
  {
    super();
    actor = null;
  }
  
  /**
   * construct a new CacheObject
   */
  public ActorCacheObject(String name, KeplerLSID lsid)
  {
    super(name, lsid);
  }
  
  /**
   * create an actorCacheObject from a file
   */
  public ActorCacheObject(InputStream actorStream)
    throws CacheException
  {
    super();
    //write out the stream to an object
    try
    {
      actor = new ActorMetadata(actorStream, false);
      this.name = actor.getName();
      this.lsid = new KeplerLSID(actor.getId());
      actor.setChanged(true);
    }
    catch(Exception e)
    {
      throw new CacheException("Error creating ActorCacheObject: " + 
        e.getMessage());
    }
  }
  
  /**
   * this returns an ActorMetadata object.  It will need to be casted to be used
   */
  public Object getObject()
  {
    return getMetadata();
  }
  
  /**
   * serialize this class
   */
  public void writeExternal(ObjectOutput out) 
    throws IOException
  {
    byte[] b = actor.toString().getBytes();
    //System.out.println("serialized as " + actor.toString());
    out.write(b, 0, b.length);
    out.flush();
  }
  
  /**
   * deserialize this class
   */
  public void readExternal(ObjectInput in) 
    throws IOException, ClassNotFoundException
  {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    byte[] b = new byte[1024];
    int numread = in.read(b, 0, 1024);
    while(numread != -1)
    {
      bos.write(b, 0, numread);
      numread = in.read(b, 0, 1024);
    }
    bos.flush();
    
    try
    {
      this.actor = new ActorMetadata(new ByteArrayInputStream(bos.toByteArray()));
      this.lsid = new KeplerLSID(actor.getId());
    }
    catch(Exception e)
    {
      throw new IOException("Error creating lsid in ActorCacheObject: " + 
        e.getMessage());
    }
    this.name = actor.getName();
  }
  
  /**
   * return the ActorMetadata for this actor
   */
  public ActorMetadata getMetadata()
  {
    return actor;
  }
}
