/**
 *   '$RCSfile: LSIDTree.java,v $'
 *
 *   '$Author: berkley $'
 *   '$Date: 2006/03/28 21:49:42 $'
 *   '$Revision: 1.18 $'
 *
 *  For Details: http://kepler-project.org
 *  Copyright (c) 2003-2004 The Regents of the University of California.
 *  All rights reserved.
 *  Permission is hereby granted, without written agreement and without
 *  license or royalty fees, to use, copy, modify, and distribute this
 *  software and its documentation for any purpose, provided that the above
 *  copyright notice and the following two paragraphs appear in all copies
 *  of this software.
 *  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 *  FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 *  ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 *  THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 *  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 *  PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 *  CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 *  ENHANCEMENTS, OR MODIFICATIONS.
 *  PT_COPYRIGHT_VERSION_2
 *  COPYRIGHTENDKEY
 */
package org.kepler.objectmanager.lsid;

import java.io.*;
import java.util.*;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.ecoinformatics.util.DBConnection;
import org.ecoinformatics.util.DBConnectionFactory;
import org.ecoinformatics.util.Config;

import com.ibm.lsid.LSIDException;

/**
 * this  class stores and serializes a tree of lsids and allows
 * easy searching of the tree.  The tree is structured like:
 * authority
 *    |_ namespace
 *           |_ object
 *                 |_revision
 */
public class LSIDTree
{
  private Hashtable authTree;
  private static LSIDTree lsidTree = null;
  private DBConnection conn = null;
  private static String LSIDTABLENAME = "LSIDTable";
  private boolean useDiskFile = false;
  private File lsidFile;

  public LSIDTree()
  {
    authTree = new Hashtable();
    try {
    	conn = DBConnectionFactory.getDBConnection();
    }
    catch( Exception e ) {
        e.printStackTrace();
        throw new RuntimeException("Error obtaining database connection: " + 
          e.getMessage());
    }
  }

  /**
   * construct a new LSIDTree. Read the serialized version from disk if
   * the boolean is true.
   */
  public LSIDTree(boolean readFromCache)
    throws LSIDException
  {
    this();
    if(readFromCache)
    {
      readTree();
    }
  }
  
  /**
   * read the tree info from the specified file instead of the database
   */
  public LSIDTree(File lsidFile)
    throws LSIDException, FileNotFoundException, IOException
  {
    authTree = new Hashtable();
    useDiskFile = true;
    this.lsidFile = lsidFile;
    if(!lsidFile.exists())
    {
      lsidFile.createNewFile();
    }
    readTree(lsidFile);
  }

  /**
   * return the singleton instance of this class
   */
  public static synchronized LSIDTree getInstance()
  {
    if(lsidTree == null)
    {
      try
      {
        lsidTree = new LSIDTree(true);
      }
      catch(LSIDException lside)
      {
        lsidTree = new LSIDTree();
      }
    }
    return lsidTree;
  }
  
  /**
   * return an instance of the lsid tree built from a file.
   */
  public static synchronized LSIDTree getInstance(File lsidFile)
    throws Exception
  {
    if(lsidTree == null)
    {
      try
      {
        lsidTree = new LSIDTree(lsidFile);
      }
      catch(Exception lside)
      {
        throw new Exception("Error getting instance of the LSIDTree: " + 
          lside.getMessage());
      }
    }
    return lsidTree;
  }
  
  /**
   * adds the lsid to the database (or file) and the tree
   */
  public void addLSID(KeplerLSID lsid) 
    throws LSIDTreeException, FileNotFoundException, IOException
  {
    addLSID(lsid, true);
  }

  /**
   * add the specified lsid to the tree.  If it already exists in the tree
   * throw an Exception.  
   */
  public void addLSID(KeplerLSID lsid, boolean addToDB) 
    throws LSIDTreeException, FileNotFoundException, IOException
  {
    int lastObj = -1;
    int lastRev = -1;

    Hashtable nsTree = (Hashtable)authTree.get(lsid.getAuthority());
    if(nsTree == null)
    {
      Hashtable revHash = new Hashtable();
      revHash.put(lsid.getRevision(), new Integer(-1));
      Hashtable objHash = new Hashtable();
      objHash.put(lsid.getObject(), revHash);
      nsTree = new Hashtable();
      nsTree.put(lsid.getNamespace(), objHash);
      authTree.put(lsid.getAuthority(), nsTree);
      addLSIDToDB(lsid, addToDB);
    }
    else
    {
      Hashtable objTree = (Hashtable)nsTree.get(lsid.getNamespace());
      if(objTree == null)
      {
        objTree = new Hashtable();
        Hashtable revHash = new Hashtable();
        revHash.put(lsid.getRevision(), new Integer(-1));
        objTree = new Hashtable();
        objTree.put(lsid.getObject(), revHash);
        nsTree.put(lsid.getNamespace(), objTree);
        addLSIDToDB(lsid, addToDB);
      }
      else
      {
        Enumeration objEnum = objTree.keys();
        while(objEnum.hasMoreElements())
        {
          int i = new Integer((String)objEnum.nextElement()).intValue();
          if(i > lastObj)
          { //find the highest number object
            lastObj = i;
          }
        }

        Hashtable revTree = (Hashtable)objTree.get(lsid.getObject());
        if(revTree == null)
        {
          revTree = new Hashtable();
          revTree.put(lsid.getRevision(), new Integer(-1));
          objTree.put(lsid.getObject(), revTree);
          addLSIDToDB(lsid, addToDB);
        }
        else
        {
          Enumeration revEnum = revTree.keys();
          Vector revs = new Vector();
          while(revEnum.hasMoreElements())
          {
            int rev = new Integer((String)revEnum.nextElement()).intValue();
            if(rev > lastRev)
            {
              lastRev = rev;
            }
            revs.addElement(new Integer(rev));
          }

          if(!revs.contains(new Integer(lsid.getRevision())))
          {
            revTree.put(lsid.getRevision(), new Integer(-1));
            addLSIDToDB(lsid, addToDB);
          }
          else
          { //if the revision already exists, throw an exception
            throw new LSIDTreeException(lsid, lastObj, lastRev);
          }
        }
      }
    }
  }
  
  /**
   * returns true if the lsid is registered in the tree, false otherwise.
   */
  public boolean isRegistered(KeplerLSID lsid)
  {
    int lastObj = -1;
    int lastRev = -1;
    
    Hashtable nsTree = (Hashtable)authTree.get(lsid.getAuthority());
    if(nsTree == null)
    {
      return false;
    }
    else
    {
      Hashtable objTree = (Hashtable)nsTree.get(lsid.getNamespace());
      if(objTree == null)
      {
        return false;
      }
      else
      {
        Enumeration objEnum = objTree.keys();
        while(objEnum.hasMoreElements())
        {
          int i = new Integer((String)objEnum.nextElement()).intValue();
          if(i > lastObj)
          { //find the highest number object
            lastObj = i;
          }
        }

        Hashtable revTree = (Hashtable)objTree.get(lsid.getObject());
        if(revTree == null)
        {
          return false;
        }
        else
        {
          Enumeration revEnum = revTree.keys();
          Vector revs = new Vector();
          while(revEnum.hasMoreElements())
          {
            int rev = new Integer((String)revEnum.nextElement()).intValue();
            if(rev > lastRev)
            {
              lastRev = rev;
            }
            revs.addElement(new Integer(rev));
          }

          if(!revs.contains(new Integer(lsid.getRevision())))
          {
            return false;
          }
          else
          { //if the revision already exists, the lsid is registered
            return true;
          }
        }
      }
    }
  }
  
  /**
   * Remove the given lsid from the registry.
   * @param lsid the lsid to remove
   */
  public void removeLSID(KeplerLSID lsid)
  {
      try
      {
        String sql = "delete from " + LSIDTABLENAME + " where lsid = '" + lsid.toString() + "';";
        conn.executeSQLCommand(sql);
        conn.commit();
      }
      catch( SQLException e ) {
          return;
      }

      Hashtable nsTree = (Hashtable)authTree.get(lsid.getAuthority());
      if(nsTree == null)
      {
          return;
      }
      Hashtable objTree = (Hashtable)nsTree.get(lsid.getNamespace());
      if(objTree == null)
      {
          return;
      }
      Hashtable revTree = (Hashtable)objTree.get(lsid.getObject());
      if(revTree == null)
      {
          return;
      }
      revTree.remove( lsid.getRevision() );
      return;
  }
  
  /**
   * return, but do not register the next revision of the given lsid
   * @param lsid the lsid to find the next rev for.
   */
  public KeplerLSID findNextRevision(KeplerLSID lsid)
  {
    int lastRev = -1;

    Hashtable nsTree = (Hashtable)authTree.get(lsid.getAuthority());
    if(nsTree == null)
    {
      return lsid;
    }
    else
    {
      Hashtable objTree = (Hashtable)nsTree.get(lsid.getNamespace());
      if(objTree == null)
      {
        return lsid;
      }
      else
      {
        Hashtable revTree = (Hashtable)objTree.get(lsid.getObject());
        if(revTree == null)
        {
          return lsid;
        }
        else
        {
          Enumeration revEnum = revTree.keys();
          Vector revs = new Vector();
          while(revEnum.hasMoreElements())
          {
            int rev = new Integer((String)revEnum.nextElement()).intValue();
            if(rev > lastRev)
            {
              lastRev = rev;
            }
            revs.addElement(new Integer(rev));
          }

          try
          {
            return new KeplerLSID(lsid.getAuthority(), lsid.getNamespace(),
              new Integer(lsid.getObject()).intValue(), lastRev + 1);
          }
          catch(LSIDException lside)
          {
            throw new RuntimeException("Error finding last revision: " + 
              lside.getMessage());
          }
        }
      }
    }
  }
  
  /**
   * returns the lsid with the next possible revision number.  if you pass
   * in urn:lsid:localhost:actor:1:1, this should return
   * urn:lsid:localhost:actor:1:2 if 2 is the next number available to use.
   * if there is already a revision 2, it will return the next available
   * revision (i.e. 3).  if this lsid does not exist in the database,
   * the lsid will be registered and the same lsid will be returned to you.
   * Note that the new lsid WILL be registered in the database prohibiting
   * the same lsid from being registered again.
   * @param lsid the lsid to inc the revision
   
   */
  public KeplerLSID getNextRevision(KeplerLSID lsid)
  {
    return getNextRevision(lsid, true);
  }

  /**
   * returns the lsid with the next possible revision number.  if you pass
   * in urn:lsid:localhost:actor:1:1, this should return
   * urn:lsid:localhost:actor:1:2 if 2 is the next number available to use.
   * if there is already a revision 2, it will return the next available
   * revision (i.e. 3).  if this lsid does not exist in the database,
   * the lsid will be registered and the same lsid will be returned to you.
   * if register is false, the new lsid will not be registered in the database
   * @param lsid the lsid to inc the revision
   * @param register set to true if you want the new lsid registered, false
   * otherwise
   */
  public KeplerLSID getNextRevision(KeplerLSID lsid, boolean register)
  {
    try
    {
      if(register)
      {
        addLSID(lsid);
        return lsid;
      }
      else
        return findNextRevision(lsid);
      
    }
    catch(LSIDTreeException lsidte)
    {
      try
      {
        KeplerLSID newLsid = new KeplerLSID("urn:lsid:" + lsid.getAuthority() + ":" +
          lsid.getNamespace() + ":" + lsid.getObject() + ":" +
          (lsidte.lastRevision + 1));
        if(register)
        {
          addLSID(newLsid);
        }
        
        return newLsid;
      }
      catch(Exception e)
      { //basically ignore this.  it shouldn't happen
        e.printStackTrace();
        return null;
      }
    }
    catch(FileNotFoundException fnfe)
    {
      if(useDiskFile)
      {
        System.out.println("Error, could not find the lsid file: " + fnfe.getMessage());
      }
      return null;
    }
    catch(IOException ioe)
    {
      if(useDiskFile)
      {
        System.out.println("Error, could not open the lsid file: " + ioe.getMessage());
      }
      return null;
    }
  }

  /**
   * return the KeplerLSID with the next object number or
   * return the passed lsid itself if it has not been registered
   * and is already has the next object number in the series
   * if register is true, register the new lsid.
   * @param lsid
   */
  public KeplerLSID getNextObject(KeplerLSID lsid, boolean register)
    throws LSIDException
  {
    int nextObjNum = getNextObject(lsid.getAuthority(), lsid.getNamespace());

    if(nextObjNum > (new Integer(lsid.getObject())).intValue())
    {
      KeplerLSID newLsid = new KeplerLSID(lsid.getAuthority(), lsid.getNamespace(),
        nextObjNum, 1);
      if(register)
      {
        try
        {
          addLSID(newLsid);
        }
        catch(Exception e)
        {
          System.out.println("Error adding lsid: " + e.getMessage());
          e.printStackTrace();
        }
      }
      return newLsid;
    }

    if(register)
    {
      try
      {
        addLSID(lsid);
      }
      catch(Exception e)
      {
        System.out.println("Error adding lsid: " + e.getMessage());
        e.printStackTrace();
      }
    }
    
    return lsid;
  }
  
  /**
   * get the next object and register the lsid
   */
  public KeplerLSID getNextObject(KeplerLSID lsid)
    throws LSIDException
  {
    return getNextObject(lsid, true);
  }

  /**
   * return a string rep of the tree.
   */
  public String toString()
  {
    return authTree.toString();
  }
  
  /**
   * clears the tree and deletes the serial file.
   */
  public void clearTree()
    throws LSIDTreeException
  {
    authTree = new Hashtable();
    if(useDiskFile)
    {
      lsidFile.delete();
    }
    else
    {
      try
      {
        String sql = "delete from " + LSIDTABLENAME;
        conn.executeSQLCommand(sql);
        conn.commit();
      }
      catch(SQLException sqle)
      {
        throw new LSIDTreeException("Error clearing tree info from the database: " + 
          sqle.getMessage());
      }
    }
  }
  
  /**
   * read the tree from the database and create a tree object from that.
   */
  protected void readTree()
    throws LSIDException
  {
    try
    {
      String sql = "select lsid from " + LSIDTABLENAME;
      ResultSet rs = conn.executeSQLQuery(sql);
      while(rs.next())
      {
        String lsid = rs.getString("lsid");
        try
        {
          addLSID(new KeplerLSID(lsid), false);
        }
        catch(Exception e)
        {
          System.out.println("Error adding lsid: " + e.getMessage());
          e.printStackTrace();
        }
      }
    }
    catch(SQLException sqle)
    {
      throw new LSIDException("Error reading tree from the database: " + 
        sqle.getMessage());
    }
  }
  
  /**
   * read the tree from a file on disk and create a tree object from that.
   */
  protected void readTree(File lsidFile)
    throws LSIDException, FileNotFoundException, IOException
  {
    FileInputStream fis = new FileInputStream(lsidFile);
    StringBuffer sb = new StringBuffer();
    byte[] b = new byte[1024];
    int numread = fis.read(b, 0, 1024);
    while(numread != -1)
    { //read the file and put the contents in a stringbuffer
      sb.append(new String(b, 0, numread));
      numread = fis.read(b, 0, 1024);
    }
    StringTokenizer st = new StringTokenizer(sb.toString(), "\n");
    while(st.hasMoreTokens())
    { //tokenize the string and add it to the hash.
      String s = st.nextToken();
      addLSID(new KeplerLSID(s), false);
    }
  }

  /**
   * returns the next possible object number for an authority and namespace
   *
   * @param authority
   * @param namespace
   */
  private int getNextObject(String authority, String namespace)
    throws LSIDException
  {
    KeplerLSID lsid;
    Hashtable nsTree = (Hashtable)authTree.get(authority);
    if(nsTree != null)
    { //check if the authority exists
      Hashtable objTree = (Hashtable)nsTree.get(namespace);
      if(objTree != null)
      { //check if the namespace exists
        int highestObj = 0;
        Enumeration keys = objTree.keys();
        while(keys.hasMoreElements())
        {
          int key = (new Integer((String)keys.nextElement())).intValue();
          if(key > highestObj)
          {
            highestObj = key;
          }
        }

        return highestObj + 1;
      }
      else
      {
        return 1;
      }
    }
    else
    {
      return 1;
    }
  }
  
  /**
   * add an lsid to the DB
   */
  private void addLSIDToDB(KeplerLSID lsid, boolean add)
    throws LSIDTreeException, FileNotFoundException, IOException
  {
    if(add)
    {
      if(useDiskFile)
      { //write the lsid to a file
        FileOutputStream fos = new FileOutputStream(lsidFile, true);
        byte[] b = (lsid.toString() + "\n").getBytes();
        fos.write(b);
        fos.flush();
        fos.close();
      }
      else
      { //write the lsid to the db
        try
        {
          String sql = "insert into " + LSIDTABLENAME + " (lsid) " +
            "values ('" + lsid.toString() + "');";
          conn.executeSQLCommand(sql);
          conn.commit();
        }
        catch(Exception sqle)
        {
          throw new LSIDTreeException("Could not add LSID to the database.");
        }
      }
    }
  }
}
