/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.util.ArrayList;
import java.util.List;

import ptolemy.actor.Actor;
import ptolemy.actor.IOPort;
import ptolemy.actor.TypedIOPort;


/**
 * The importer for building a SchemaTree from a list of actors in Kepler/Ptolemy
 * Workflow System, so SCIA can work on.
 *
 *@author Guilian Wang
 */
public class ActorsImporter {
    private List _actors; // the actors to build SchemaTree from
    private boolean _isTarget; //target actors or not

    /**
     * The default constructor
     */
    public ActorsImporter(List actors, boolean isTarget) {
        _actors = actors;
        _isTarget = isTarget;
    }

    /**
     * Build an SchemaTree from _actors
     */
    public SchemaTree buildSchemaTree() {
        //create an empty schema tree
        SchemaTree tree = new SchemaTree();

        if (_isTarget) {
            tree.treeName = "targetTree";
        } else {
            tree.treeName = "sourceTree";
        }

        //to do: get description for the root node
        String description = null;
        SchemaNode root = new SchemaNode("Actors", "rootElem", "", description);

        //add the root node into the tree
        tree.rootNode = new TNode(root);
        tree.rootNode.tree = tree;

        for (int i = 0; i < _actors.size(); i++) {
            Actor actor = (Actor) _actors.get(i);

            if (SCIA.debug_on) {
                System.err.println("processing actor " + actor.getName());
            }

            //to do: get description for the root node
            description = null;

            //add the actor node into the tree
            SchemaNode node = new SchemaNode(actor.getName(), "parentElem", "",
                    description);
            TNode actorTNode = new TNode(node, tree.rootNode);

            List ports = new ArrayList();

            // get output ports if _actors are source actors
            if (!_isTarget) {
                ports = actor.outputPortList();
            }

            // get input ports if _actors are target actors
            if (_isTarget) {
                ports = actor.inputPortList();
            }

            //add each port as a subelement of the actor element
            for (int j = 0; j < ports.size(); j++) {
                IOPort port = (IOPort) ports.get(j);

                if (SCIA.debug_on) {
                    System.err.println("Port " + port.getName());
                }

                String portDataType = null;

                if (port instanceof TypedIOPort) {
                    portDataType = ((TypedIOPort) port).getType().toString();
                }

                node = new SchemaNode(port.getName(), "leafElem", portDataType,
                        description);

                TNode portTNode = new TNode(node, actorTNode);
            }
        }

        return tree;
    }
}
