/**
 * Copyright (c) 2005 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.kepler.scia;

import java.awt.BasicStroke;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.io.BufferedReader;
import java.io.StringReader;
import java.text.DecimalFormat;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;


/**
 * MyJSplitPane extends JSplitPane, to allow drawing lines between its two
 * component trees
 *
 * @author Guilian Wang and Vitaliy Zavesov
 */
class MyJSplitPane extends JSplitPane {
    SchemaTree leftTree;
    SchemaTree rightTree;
    JScrollPane leftSPane;
    JScrollPane rightSPane;
    Vector m_matchLines;
    Point focusPoint = null; // for multiple matchings

    // tree offsets
    int x1Offset;

    // tree offsets
    int y1Offset;

    // tree offsets
    int x2Offset;

    // tree offsets
    int y2Offset;
    public Vector highlightNodes;

    public MyJSplitPane(int orientation, JScrollPane spane1,
        JScrollPane spane2, SchemaTree tree1, SchemaTree tree2,
        Vector matchLines) {
        super(orientation, spane1, spane2);
        leftSPane = spane1;
        rightSPane = spane2;
        leftTree = tree1;
        rightTree = tree2;
        setDividerLocation(330);
        m_matchLines = matchLines;

        tree1.setCellRenderer(new MyTreeCellRenderer());
        tree2.setCellRenderer(new MyTreeCellRenderer());
    }

    /**
     * The over-written JSplitPane paintComponent method
     * which calls drawMapping
     *
     * @param g the graphics component
     */
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        drawMapping(g);
    }

    /**
     * Draws the mapping (match lines) on the trees
     *
     * @param g graphics component for drawing lines
     */
    public boolean drawMapping(Graphics g) {
        // the left tree offset in the MyJSplitPane
        x1Offset = (int) leftTree.getLocation().getX() +
            (int) leftSPane.getViewport().getLocation().getX() +
            (int) leftSPane.getLocation().getX();
        y1Offset = (int) leftTree.getLocation().getY() +
            (int) leftSPane.getViewport().getLocation().getY() +
            (int) leftSPane.getLocation().getY();

        // the right tree offset in the MyJSplitPane
        x2Offset = (int) rightTree.getLocation().getX() +
            (int) rightSPane.getViewport().getLocation().getX() +
            (int) rightSPane.getLocation().getX();
        y2Offset = (int) rightTree.getLocation().getY() +
            (int) rightSPane.getViewport().getLocation().getY() +
            (int) rightSPane.getLocation().getY();

        for (int i = 0; i < m_matchLines.size(); i++) {
            MatchLine mline = (MatchLine) m_matchLines.get(i);

	    Graphics2D g2d = (Graphics2D)g;

            if (mline.selected == true) {
                g2d.setColor(Color.red);
            } else {
                g2d.setColor(Color.blue);
            }

            // round the similarity value to two decimal digits
            DecimalFormat format = new DecimalFormat("#.##");
            String simStr = format.format(mline.sim);

	    if(mline.sameLines.size() != 0) {
		g2d.setStroke(new BasicStroke(2));
		if(!mline.topLine) {
		    simStr = new String();
		}
	    }

            if (mline.leftTNode == null) {
                drawLocalMultipleMatchings(mline.mMatchings, mline.rightTNode,
                    g2d, simStr);
            } else {
                drawSingleMatching(mline.leftTNode, mline.rightTNode, g2d,
                    simStr);
            }

            // set the focus point in the match line
            mline.focusPoint = focusPoint;
        }

        return true;
    }

    /**
     * Draws a line between two nodes in the left and right
     * tree for single matchings
     *
     * @param leftNode left node
     * @param rightNode right node
     * @param g2d graphics component for drawing lines
     * @param sim the similarity value displayed
     */
    public void drawSingleMatching(TNode leftNode, TNode rightNode, Graphics2D g2d,
        String sim) {
        TreePath leftTreePath = leftNode.getTreePath();
        TreePath rightTreePath = rightNode.getTreePath();

        int leftRow = leftTree.getRowForPath(leftTreePath);
        int rightRow = rightTree.getRowForPath(rightTreePath);

        if (leftTree.isVisible(leftTreePath)) {
            if ((rightTree.getRowBounds(rightRow) == null) ||
                    (leftTree.getRowBounds(leftRow) == null)) {
                return;
            }

            // start point
            int x2 = (int) rightTree.getRowBounds(rightRow).getX() + x2Offset;
            int y2 = (int) rightTree.getRowBounds(rightRow).getY() +
                ((int) rightTree.getRowBounds(rightRow).getHeight() / 2) +
                y2Offset;

            //end point
            int x1 = (int) leftTree.getRowBounds(leftRow).getX() +
                (int) leftTree.getRowBounds(leftRow).getWidth() + x1Offset;
            int y1 = (int) leftTree.getRowBounds(leftRow).getY() +
                ((int) leftTree.getRowBounds(leftRow).getHeight() / 2) +
                y1Offset;

            g2d.drawLine(x1, y1, x2, y2);

            // position of the similarity value
            int simX = x1 + ((x2 - x1) / 2);
            int simY = y1 + ((y2 - y1) / 2);



            //String simStr = Double.toString(simRnd);
            // draw similarity value
            if (leftTree.mainUI.showSimilarity) {
                g2d.drawString(sim, simX, simY-1);
            }
        }
	
	g2d.setStroke(new BasicStroke(1));
    }

    /**
     * Draws a line connecting multiple nodes for local multiple
     * matchings
     *
     * @param mMatchings a vector of left tree nodes which
     * make up the the multiple matching
     * @param rightNode right node
     * @param g2d graphics component for drawing lines
     * @param sim the similarity value displayed
     */
    public void drawLocalMultipleMatchings(Vector mMatchings, TNode rightNode,
        Graphics2D g2d, String sim) {
        // find the vector of left coordinates
        Vector cd = new Vector();
        mapCoordinates(mMatchings, cd);

        // find the center coordinates
        Point center;
        center = findCenterXY(cd);

        // find the right coordinates
        TreePath rightTreePath = rightNode.getTreePath();
        int rightRow = rightTree.getRowForPath(rightTreePath);

        if (rightTree.getRowBounds(rightRow) == null) {
            return;
        }

        int x2 = (int) rightTree.getRowBounds(rightRow).getX() + x2Offset;
        int y2 = (int) rightTree.getRowBounds(rightRow).getY() +
            ((int) rightTree.getRowBounds(rightRow).getHeight() / 2) +
            y2Offset;

        // find the x-coordinate where all the left lines come together
        int focusCoordinate = center.x + ((x2 - center.x) / 8);

        // Draw the lines from the left nodes to the center
        for (int i = 0; i < cd.size(); i++) {
            Point p = (Point) cd.get(i);
            g2d.drawLine(p.x, p.y, focusCoordinate, center.y);

            // set the focus point 
            focusPoint = new Point(focusCoordinate, center.y);
        }

        // Draw the line from the center to the right node
        g2d.drawLine(focusCoordinate, center.y, x2, y2);

        // position of the similarity value
        int simX = focusCoordinate + ((x2 - focusCoordinate) / 3);
        int simY = center.y + ((y2 - center.y) / 2);

        // draw similarity value
        if (leftTree.mainUI.showSimilarity) {
            g2d.drawString(sim, simX, simY-1);
        }

	g2d.setStroke(new BasicStroke(1));
    }

    /**
     * The method used to draw multiple matching lines which
     * collects the locations of the left tree nodes
     *
     * @param ml the vector of left tree nodes
     * @param cd the vector of the coordinates of the
     * left tree nodes
     */
    public void mapCoordinates(Vector ml, Vector cd) {
        for (int i = 0; i < ml.size(); i++) {
            TreePath leftTreePath = ((TNode) ml.get(i)).getTreePath();

            if (leftTree.isVisible(leftTreePath)) {
                int leftRow = leftTree.getRowForPath(leftTreePath);

                int x1 = (int) leftTree.getRowBounds(leftRow).getX() +
                    (int) leftTree.getRowBounds(leftRow).getWidth() + x1Offset;
                int y1 = (int) leftTree.getRowBounds(leftRow).getY() +
                    ((int) leftTree.getRowBounds(leftRow).getHeight() / 2) +
                    y1Offset;

                Point p = new Point(x1, y1);

                cd.add(p);
            }
        }
    }

    /**
     * Finds the center point for drawing multiple match lines
     *
     * @param cd the vector of the coordinates of the
     * left tree nodes
     */
    public Point findCenterXY(Vector cd) {
        int maxX = 0;
        int maxY = 0;
        int minX = 1000000;
        int minY = 1000000;

        for (int i = 0; i < cd.size(); i++) {
            Point p = (Point) cd.get(i);

            if (p.x > maxX) {
                maxX = p.x;
            }

            if (p.y > maxY) {
                maxY = p.y;
            }

            if (p.x < minX) {
                minX = p.x;
            }

            if (p.y < minY) {
                minY = p.y;
            }
        }

        Point u = new Point();

        u.x = minX + ((maxX - minX) / 2);
        u.y = minY + ((maxY - minY) / 2);

        return u;
    }

    /**
     * Highlight passed in nodes in the left and right trees
     * Used to prompt the user during the semi-automatic matching
     * process
     *
     * @param nodes the list of nodes to hightlight
     * @param g the graphics component used to highlight nodes
     */
    public void drawHighlight(Vector nodes, Graphics g) {
        Color c = new Color((float) 0.4, (float) 0.2, (float) 0.0);

        if ((nodes != null) && !nodes.isEmpty()) {
            TNode node = new TNode();

            TreePath[] selectionPaths = new TreePath[5];

            // now highlight left tree nodes
            for (int i = 0; i < nodes.size(); i++) {
                node = (TNode) nodes.get(i);

                if (node.tree.treeName.equals(leftTree.treeName)) {
                    MyTreeCellRenderer cellRenderer = (MyTreeCellRenderer) leftTree.getCellRenderer();

                    cellRenderer.setHighlightColor();
                    leftTree.setCellRenderer(cellRenderer);

                    selectionPaths[i] = node.getTreePath();
                }
            }

            leftTree.setSelectionPaths(selectionPaths);

            // now highlight right tree nodes
            for (int i = 0; i < nodes.size(); i++) {
                node = (TNode) nodes.get(i);

                if (node.tree.treeName.equals(rightTree.treeName)) {
                    MyTreeCellRenderer cellRenderer = (MyTreeCellRenderer) rightTree.getCellRenderer();

                    cellRenderer.setHighlightColor();

                    if (rightTree.getRowForPath(rightTree.getSelectionPath()) != rightTree.getRowForPath(
                                node.getTreePath())) {
                        rightTree.setSelectionPath(node.getTreePath());
                    }

                    for (int j = 0; j < m_matchLines.size(); j++) {
                        MatchLine mline = (MatchLine) m_matchLines.get(j);

                        if (mline.rightTNode == node) {
                            mline.selected = true;
                        }
                    }
                }
            }

            // select middle arrow
            rightTree.mainUI.arr.manualSelect("middle");
        }
    }

    /**
     * Un-highlights the nodes
     *
     */
    public void removeHighlight() {
        MyTreeCellRenderer cellRenderer = (MyTreeCellRenderer) leftTree.getCellRenderer();

        cellRenderer.unsetHighlightColor();

        cellRenderer = (MyTreeCellRenderer) rightTree.getCellRenderer();

        cellRenderer.unsetHighlightColor();
    }

    /**
     * This class is used to mark recursion nodes and the parents of
     * deleted sub-trees with different foreground colors
     *
     */
    class MyTreeCellRenderer extends JPanel implements TreeCellRenderer {
        protected boolean highlightOn = false;

        public MyTreeCellRenderer() {
            setLayout(new BorderLayout());
        }

        public void setHighlightColor() {
            highlightOn = true;
        }

        public void unsetHighlightColor() {
            highlightOn = false;
        }

        public Component getTreeCellRendererComponent(JTree tree, Object value,
            boolean sel, boolean expanded, boolean leaf, int row,
            boolean hasFocus) {
            MyTextArea contentArea = new MyTextArea();
            MyTextArea1 contentArea1 = new MyTextArea1();

            if (highlightOn) {
                contentArea.setHighlightColor();
                contentArea1.setHighlightColor();
            }

            setEnabled(tree.isEnabled());
            contentArea.setSelect(sel);
            contentArea1.setSelect(sel);

            TNode node = (TNode) value;

            Color color = Color.red;
            boolean recursion = false;

            if (node.reference != null) {
                contentArea.setForeground(color);
                contentArea1.setForeground(color);
                recursion = true;
            }

            color = Color.blue;

            boolean deletedSubtree = false;

            if ((node.deletedChildren != null) &&
                    (node.deletedChildren.size() > 0)) {
                contentArea.setForeground(color);
                contentArea1.setForeground(color);
                deletedSubtree = true;
            }

            String msg = new String();

            if (recursion) {
                msg += "(rec.) ";
            }

            if (deletedSubtree) {
                msg += "(sub.) ";
            }

            msg += node.data.getTag();

            String dt = node.data.dataType;
            String cardi = node.getCardinality();

            boolean showCardinality = (cardi != null) && (cardi.length() != 0);

            if ((tree != null) && (((SchemaTree) tree).mainUI != null)) {
                showCardinality = showCardinality &&
                    ((SchemaTree) tree).mainUI.showCardinality;
            }

            boolean showType = (dt != null) && (dt.length() != 0);

            if ((tree != null) && (((SchemaTree) tree).mainUI != null)) {
                showType = showType && ((SchemaTree) tree).mainUI.showType;
            }

            if (!node.data.nodeType.equals("compositor")) {
                if (showType || showCardinality) {
                    msg += " (";
                }

                if (showType) {
                    msg += dt;
                }

                if (showType && showCardinality) {
                    msg += " ";
                }

                if (showCardinality) {
                    msg += cardi;
                }

                if (showType || showCardinality) {
                    msg += ")";
                }
            }

            contentArea.setText(msg);

            String msg1;
            String desc = node.data.description;

            boolean showDescription = (desc != null) && (desc.length() != 0);

            if ((tree != null) && (((SchemaTree) tree).mainUI != null)) {
                showDescription = showDescription &&
                    ((SchemaTree) tree).mainUI.showDescription;
            }

            if (showDescription) {
                contentArea.setSingleArea(false);
            } else {
                contentArea.setSingleArea(true);
            }

            contentArea.setFocus(hasFocus);
            contentArea1.setFocus(hasFocus);

            removeAll();

            add(contentArea, BorderLayout.NORTH);

            if (!node.data.nodeType.equals("compositor")) {
                if (showDescription) {
                    msg1 = "(" + desc + ")";
                    contentArea1.setText(msg1);

                    add(contentArea1, BorderLayout.SOUTH);
                }
            }

            validate();

            SchemaTreeModel model = (SchemaTreeModel) (tree.getModel());
            model.nodeChanged((TreeNode) node);

            return this;
        }
    }

    class MyTextArea extends JTextArea {
        Color selectionColor = Color.white;
        boolean singleArea = true;

        MyTextArea() {
            setLineWrap(true);

            //setWrapStyleWord(true);
            setOpaque(true);
        }

        public void setForeground(Color c) {
            super.setForeground(c);
        }

        public void setHighlightColor() {
            selectionColor = Color.red;
        }

        public void unsetHighlightColor() {
            selectionColor = Color.white;
        }

        public void setSingleArea(boolean single) {
            singleArea = single;
        }

        public void setText(String str) {
            FontMetrics fm = getToolkit().getFontMetrics(getFont());
            BufferedReader br = new BufferedReader(new StringReader(str));

            String line = new String();
            int width = 0;
            int lines = 1;

            String finalString = new String();

            try {
                if ((line = br.readLine()) != null) {
                    width = SwingUtilities.computeStringWidth(fm,
                            line + "       ");
                    finalString = line + "\n";
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            int height = fm.getHeight() * lines;
            super.setPreferredSize(new Dimension(width, height));
            super.setText(finalString);
        }

        public void setTextForChildren(String str) {
            super.setText(str);
        }

        void setSelect(boolean isSelected) {
            Color bColor;
            Color fColor = Color.black;

            if (isSelected) {
                if (selectionColor == Color.white) {
                    bColor = UIManager.getColor("Tree.selectionBackground");
                } else {
                    bColor = selectionColor;
                    fColor = Color.white;
                }
            } else {
                bColor = UIManager.getColor("Tree.textBackground");
            }

            super.setBackground(bColor);
            super.setForeground(fColor);
        }

        void setFocus(boolean hasFocus) {
            if (hasFocus) {
                Color lineColor = UIManager.getColor(
                        "Tree.selectionBorderColor");

                if (singleArea) {
                    setBorder(BorderFactory.createLineBorder(lineColor));
                } else {
                    setBorder(BorderFactory.createMatteBorder(1, 1, 0, 1,
                            lineColor));
                }
            } else {
                setBorder(BorderFactory.createEmptyBorder(1, 1, 0, 1));
            }
        }
    }

    class MyTextArea1 extends MyTextArea {
        public void setText(String str) {
            Font newFont = getFont().deriveFont(Font.ITALIC,
                    getFont().getSize() - (float) 0.5);
            FontMetrics fm = getToolkit().getFontMetrics(newFont);
            super.setForeground(new Color((float) 0.0, (float) 0.5, (float) 0.0));

            BufferedReader br = new BufferedReader(new StringReader(str));

            String line = new String();
            int width = 0;
            int lines = 1;

            String finalString = new String();

            try {
                line = br.readLine();

                if (line != null) {
                    int maxWidth = SwingUtilities.computeStringWidth(fm, " ") * 80;

                    if (fm.stringWidth(line + "  ") > maxWidth) {
                        width = maxWidth;
                    } else {
                        width = fm.stringWidth(line + "  ");
                        finalString = "  " + line;
                    }

                    if (finalString.length() == 0) {
                        StringTokenizer st = new StringTokenizer(line);
                        String next = new String();
                        line = new String();

                        while (st.hasMoreTokens()) {
                            next = st.nextToken();

                            if (fm.stringWidth("  " + line + " " + next + "\n") < width) {
                                line += (" " + next);
                            } else {
                                finalString += (" " + line + "\n");
                                line = " " + next;
                                lines++;
                            }
                        }

                        finalString += (" " + line);
                    }
                }
            } catch (Exception ex) {
                ex.printStackTrace();
            }

            int height = fm.getHeight() * lines;
            super.setPreferredSize(new Dimension(width, height));

            super.setFont(newFont);
            super.setTextForChildren(finalString);
        }

        void setFocus(boolean hasFocus) {
            if (hasFocus) {
                Color lineColor = UIManager.getColor(
                        "Tree.selectionBorderColor");

                setBorder(BorderFactory.createMatteBorder(0, 1, 1, 1, lineColor));
            } else {
                setBorder(BorderFactory.createEmptyBorder(0, 1, 1, 1));
            }
        }
    }
}
