/**
 *    '$RCSfile: NamedOntClass.java,v $'
 *
 *     '$Author: bowers $'
 *       '$Date: 2005/06/28 00:36:52 $'
 *   '$Revision: 1.9 $'
 *
 *  For Details: http://kepler.ecoinformatics.org
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package org.kepler.sms;

/**
 * 
 */

import com.hp.hpl.jena.ontology.*; 
import java.util.*;

public class NamedOntClass implements Comparable { 

    private OntClass _ontClass;
    private String _name;

    public NamedOntClass(OntClass ontClass) {
	_ontClass = ontClass;
	String str = ontClass.getLabel(null);
	if(str != null)
	    _name = new String(str);
	else
	    _name = new String(ontClass.getLocalName());
    }

    public String toString() {
	return getName();
    }

    public OntClass ontClass() {
	return _ontClass;
    }

    public String getName() {
	return _name;
    } 

    public String getOntologyName() {
	String nspace = getNameSpace();
	OntologyCatalog _catalog = OntologyCatalog.instance();
	String oname = _catalog.getOntologyName(nspace);
	return oname;
    }

    public String getLocalName() {
	return _ontClass.getLocalName();
    }

    public String getNameSpace() {
	return _ontClass.getNameSpace();
    }


    /**
     * @param sorted Return sorted list if true.
     */
    public Iterator getNamedSubClasses(boolean sorted) {
	Vector result = new Vector();
	for(Iterator i = _ontClass.listSubClasses(true); i.hasNext();) {
	    OntClass subcls = (OntClass)i.next();
	    if(subcls.getLabel(null) != null || subcls.getLocalName() != null)
	    result.add(new NamedOntClass(subcls));
	}
	if(sorted)
	    Collections.sort(result);
	return result.iterator();
    }

    /**
     * @param sorted Return sorted list if true.
     */
    public Iterator getNamedSuperClasses(boolean sorted) {
	Vector result = new Vector();
	for(Iterator iter = _ontClass.listSuperClasses(true); iter.hasNext();) {
	    OntClass supcls = (OntClass)iter.next();
	    if(supcls.getLabel(null) != null || supcls.getLocalName() != null)
		result.add(new NamedOntClass(supcls));
	}
	if(sorted)
	    Collections.sort(result);
	return result.iterator();
    }

    /**
     * @param sorted Return sorted list if true.
     */
    public Iterator getNamedProperties(boolean sorted) {
	Vector result = new Vector();
	for(Iterator iter = _ontClass.listDeclaredProperties(true); iter.hasNext();) {
	    OntProperty prop = (OntProperty)iter.next();
	    result.add(new NamedOntProperty(prop));
	}
	if(sorted)
	    Collections.sort(result);
	return result.iterator();
    }


    /**
     * @return The (first) comment associated with the class
     */
    public String getComment() {
	return _ontClass.getComment(null);
    }



    /**
     * @return Answer true if the given class is a "direct" sub-class of this class.
     */
    public boolean isDirectSubClass(NamedOntClass cls) {
	for(Iterator iter = getNamedSubClasses(false); iter.hasNext();) {
	    NamedOntClass tmpCls = (NamedOntClass)iter.next();
	    if(tmpCls.equals(cls))
		return true;	    
	}
	return false;
    }

    /**
     * @return Answer true if the given class is a sub-class of this class.
     */
    public boolean isSubClass(NamedOntClass cls) {
	//return _ontClass.hasSubClass(cls.ontClass(), false);
	for(Iterator iter = getNamedSubClasses(false); iter.hasNext();) {
	    NamedOntClass tmpCls = (NamedOntClass)iter.next();
	    if(tmpCls.equals(cls))
		return true;
	    if(tmpCls.isSubClass(cls))
		return true;
	}
	return false;
    }


    /**
     * @return Answer true if the given class is a "direct" super-class of this class.
     */
    public boolean isDirectSuperClass(NamedOntClass cls) {
	//return _ontClass.hasSuperClass(cls.ontClass(), true);	
	for(Iterator iter = getNamedSuperClasses(false); iter.hasNext();) {
	    NamedOntClass tmpCls = (NamedOntClass)iter.next();
	    if(tmpCls.equals(cls))
		return true;
	}
	return false;    
    }


    /**
     * @return Answer true if the given class is a super-class of this class.
     */
    public boolean isSuperClass(NamedOntClass cls) {
	//return _ontClass.hasSuperClass(cls.ontClass(), false);
	for(Iterator iter = getNamedSuperClasses(false); iter.hasNext();) {
	    NamedOntClass tmpCls = (NamedOntClass)iter.next();
	    if(tmpCls.equals(cls))
		return true;
	    if(tmpCls.isSuperClass(cls))
		return true;
	}
	return false;
    }

    /**
     * @return Answer true if the given class is disjoint with this class.
     */
    public boolean isDisjointWith(NamedOntClass cls) {
	return _ontClass.isDisjointWith(cls.ontClass());
    }
    

    /**
     * @return Answer true if the given class is equivalent to this class.
     */
    public boolean isEquivalent(NamedOntClass cls) {
	return this.equals(cls) || _ontClass.hasEquivalentClass(cls.ontClass());
    }


    public boolean equals(Object obj) {
	if(!(obj instanceof NamedOntClass))
	    return false;
	NamedOntClass tmp = (NamedOntClass)obj;
	if(!tmp.ontClass().equals(this.ontClass()))
	    return false;
	return true;
    }


    public int compareTo(Object obj) {
	String str1 = toString();
	String str2 = obj.toString();
	return str1.compareTo(str2);
    }

}
