/* GriddlesExec actor subclased from stanard Exec from PtolemyII concepts.

 Copyright (c) 2005 The Regents of the University of California.
 All rights reserved.

  Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.

 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.

 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.

                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY

*/

package org.monash.griddles;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;

import ptolemy.actor.lib.Exec;
import ptolemy.data.ArrayToken;
import ptolemy.data.RecordToken;
import ptolemy.data.StringToken;
import ptolemy.data.expr.StringParameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.util.StringUtilities;





/**
  * GriddlesExec actor is a subclass of the original Ptolemy Exec actor which executes 
  * a command repeatedly as per the directions of the model director. The GriddLeS actor 
  * executes a command only one time independent of the model director specification. 
  * The output produced by stdout and stderr during command execution is stored in temporary 
  * files; at the end these files are read and streamed through respective ports. This could 
  * handle data of any extent and size  incontrary to the previous case where the data size 
  * limited to that of the java string buffer object. We noticed some of the models produce 
  * large volumes of data during execution. By using this we can eliminate failures which could 
  * occur due to limitation on the size of stderr and stdout.

  * This actor is designed to run jobs at the local location        
  * @author Jagan Kommineni, Monash University July 2005
  * @version $Id: GriddlesExec.java,v 1.3 2005/11/01 20:39:13 ruland Exp $
  **/


public class GriddlesExec extends Exec {
  public GriddlesExec(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
    super(container, name); 
    }

  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////
  //This method stops the execution at the end of first iteration  
  public boolean postfire() {
    return false;
    }
  
      /** Override the base class to stop waiting for input data.
     */
    public void stopFire() {
        // NOTE: This method used to be synchronized, as
        // was the fire() method, but this caused deadlocks.  EAL
        super.stopFire();
        _stopFireRequested = true;

        try {
            _terminateProcess();
        } catch (Exception ex) {
            throw new InternalErrorException(ex);
        }
    }
  /*
  This method creates a new string paprameter with a given name dynamically at run time, 
  it is primarely used by actors that create gridlets. 
  */  
  public StringParameter newStringParameter(String name) throws NameDuplicationException {
    try {
      _workspace.getWriteAccess();
      StringParameter param = new StringParameter(this, name);
      return param;
      } 
    catch (IllegalActionException ex) {
      // This exception should not occur, so we throw a runtime
      // exception.
      // throw new InternalErrorException(this, ex, null);
      } 
    finally {
      _workspace.doneWriting();
      }
    return null;
    } 
  /* This is the method that executes the command and returns stdout and stderr messages.      
  During execution these messages are stored in temporary files and at end of the execution 
  data from these files are streamed through appropriate ports and subsequently these temporary 
  files are removed.
   */  
  public synchronized void fire()  throws IllegalActionException{
          
    String line = null;
    _exec();
    if(input.numberOfSources() > 0 && input.hasToken(0)) {
      if((line = ((StringToken)input.get(0)).stringValue()) != null) {
        if(_debugging) {
          _debug("Exec: Input: '" + line + "'");
          }
        if(_inputBufferedWriter != null) {
          try{
            _inputBufferedWriter.write(line);
            _inputBufferedWriter.flush();
            }
          catch (IOException ex) {
            throw new IllegalActionException(this, ex, "Problem writing input '" + command + "'");
            }
          }
        }
      }
    try{
      // The next line waits for the subprocess to finish.
               
      int processReturnCode = _process.waitFor();
         
      if(processReturnCode != 0) {
        // We could have a parameter that would enable
        // or disable this.
        throw new IllegalActionException(this,
           "Executing command \""
           + ((StringToken)command.getToken()).stringValue()
           + "\" returned a non-zero return value of "
           + processReturnCode);
        }
      } 
    catch (InterruptedException interrupted) {
      throw new InternalErrorException(this, interrupted,
      "_process.waitFor() was interrupted");
      }        
    while(true){
      if(_errorGobbler.whetherReachedEndStream() && _outputGobbler.whetherReachedEndStream())
      break;
      try{
        Thread.sleep(1000);
        }
      catch(InterruptedException ie){}
      }
    try{
      char[] chars = new char[_errorGobbler.getBufferSize()];
      File errorFile = _errorGobbler.getErrOutFile();   
      BufferedReader br = new BufferedReader(new FileReader(errorFile));
      int length;
      while((length=br.read(chars, 0, _errorGobbler.getBufferSize()))!=-1) {        
        error.send(0, new StringToken(new String(chars).trim()));
        chars = new char[_errorGobbler.getBufferSize()];
        }        
      br.close();
      errorFile.delete(); 
      
      char[] outchars = new char[_outputGobbler.getBufferSize()];
      File outFile = _outputGobbler.getErrOutFile();
      BufferedReader  br1 = new BufferedReader(new FileReader(outFile));
      while((length=br1.read(outchars, 0, _outputGobbler.getBufferSize()))!=-1) {
        output.send(0, new StringToken((new String(outchars)).trim()));
        
        outchars = new char[_outputGobbler.getBufferSize()];
        }
      br1.close();
      outFile.delete(); 
      } 
    catch (IOException ex) {}
    }
    ///////////////////////////////////////////////////////////////////
    ////                     private methods                       ////
    // Execute a command, set _process to point to the subprocess
    // and set up _errorGobbler and _outputGobbler to read data.
    private void _exec() throws IllegalActionException{
      // This is a private method because fire() was getting too long.
      try {
        _stopFireRequested = false;
        if(_process != null) {
          // Note that we assume that _process is null upon entry
          // to this method, but we check again here just to be sure.
          _terminateProcess();
          }
        Runtime runtime = Runtime.getRuntime();
        command.update();

        // tokenizeForExec() handles substrings that start and end
        // with a double quote so the substring is considered to
        // be a single token and are returned as a single array
        // element.
        String [] commandArray = StringUtilities.tokenizeForExec(
                    ((StringToken)command.getToken()).stringValue());
        File directoryAsFile = directory.asFile();

        if(_debugging) {
          _debug("About to exec \""
          + ((StringToken)command.getToken()).stringValue()
          + "\""
          + "\n in \"" + directoryAsFile
          + "\"\n with environment:");
          }
        // Process the environment parameter.
        ArrayToken environmentTokens = (ArrayToken)environment.getToken();
        if(_debugging) {
          _debug("environmentTokens: " + environmentTokens);
          }
        String [] environmentArray = null;
        if(environmentTokens.length() >= 1) {
          environmentArray = new String[environmentTokens.length()];
          for(int i = 0; i < environmentTokens.length(); i++) {
            StringToken nameToken = (StringToken)
                   (((RecordToken) environmentTokens.getElement(i)).get("name"));
            StringToken valueToken = (StringToken)
                    (((RecordToken) environmentTokens.getElement(i)).get("value"));
            environmentArray[i] = nameToken.stringValue() + "=" + valueToken.stringValue();

            if(_debugging){
              _debug("  " + i + ". \"" + environmentArray[i] + "\"");
              }
            if( i == 0 && environmentTokens.length() == 1 && environmentArray[0].equals("=")) {
              if(_debugging) {
                 _debug("There is only one element, "
                 + "it is a string of length 0,\n so we "
                 + "pass Runtime.exec() an null "
                 + "environment so that we use\n "
                 + "the default environment");
                 }
               environmentArray = null;
               }
             }
           }
         _process = runtime.exec(commandArray, environmentArray, directoryAsFile);
         _outputGobbler = new _StreamReaderThread(_process.getInputStream(),
                             "ExecStdoutGobbler-" + _streamReaderThreadCount++);
         _errorGobbler = new _StreamReaderThread(_process.getErrorStream(),
                             "ExecStderrGobbler-" +  _streamReaderThreadCount++);
         _errorGobbler.start();
         _outputGobbler.start();   
         if(_streamReaderThreadCount > 1000) {
           // Avoid overflow in the thread count.
           _streamReaderThreadCount = 0;
           }
         OutputStreamWriter inputStreamWriter =
           new OutputStreamWriter(_process.getOutputStream());
           _inputBufferedWriter = new BufferedWriter(inputStreamWriter);
           } 
         catch (IOException ex) {
           throw new IllegalActionException(this, ex, "Problem setting up command '" + command + "'");
           }
         }
       // Terminate the process and close any associated streams.
       private void _terminateProcess() {
         if(_process != null) {
           _process.destroy();
           _process = null;
           }
         }
    ///////////////////////////////////////////////////////////////////
    ////                         inner classes                     ////

    // Private class that reads a stream in a thread and updates the
    // stringBuffer.
    private class _StreamReaderThread extends Thread {
      /** Create a _StreamReaderThread.
      *  @param inputStream The stream to read from.
      *  @param name The name of this StreamReaderThread,
      *  which is useful for debugging.
      *  @param actor The parent actor of this thread, which
      *  is used in error messages.
      */
      _StreamReaderThread(InputStream inputStream, String name) {
      _inputStream = inputStream;
      _inputStreamReader = new InputStreamReader(_inputStream);
      _stderroroutfile = new File(name); 
      }
    /** Read lines from the inputStream and append them to the
    *  stringBuffer.        */
    public void run() {
      _read();
      }
    // Read from the stream until we get to the end of the stream
    private void _read() {
      // We read the data as a char[] instead of using readline()
      // so that we can get strings that do not end in end of
      // line chars.
      char [] chars = new char[_bufferSize];
      int length=0; // Number of characters read.
      try {
        PrintWriter pwr=null;
        pwr= new PrintWriter(new BufferedWriter(new FileWriter(_stderroroutfile, false)));
              
        // Oddly, InputStreamReader.read() will return -1
        // if there is no data present, but the string can still
        // read.
                
        while (!_stopFireRequested  && !_stopRequested &&
                        ((length = _inputStreamReader.read(chars, 0, _bufferSize))!= -1)) {
                         
          if(_debugging) {
            // Note that ready might be false here since
            // we already read the data.
            System.out.println("_read(): Gobbler '" + getName()
                             + "' Ready: " + _inputStreamReader.ready()
                             + " Value: '"
                             + String.valueOf(chars, 0, length) + "'");
            }
          pwr.print(chars);
          pwr.flush();
          chars = new char[_bufferSize];
          }
        if(length == -1)
          {
          pwr.flush();
          pwr.close();
          _reachedEndStream = true;
          }                
        } 
      catch (Exception e) {}      
      }
      
    public boolean whetherReachedEndStream(){
      return _reachedEndStream;            
      }
    public File getErrOutFile(){
      return _stderroroutfile;       
      }    
    public int getBufferSize(){
      return _bufferSize;        
      }
    private File _stderroroutfile;    
    // Stream from which to read.
    private InputStream _inputStream;
    // Stream from which to read.
    private InputStreamReader _inputStreamReader;
    private boolean _reachedEndStream = false;
    private int _bufferSize = 80;
    }
        
  ///////////////////////////////////////////////////////////////////
  ////                         private variables                 ////
  // The subprocess gets its input from this BufferedWriter.
  private BufferedWriter _inputBufferedWriter;
  // StreamReader with which we read stderr.
  private _StreamReaderThread _errorGobbler;
  // StreamReader with which we read stdout.
  private _StreamReaderThread _outputGobbler;
  // The Process that we are running.
  private Process _process;
  // Indicator that stopFire() has been called.
  private boolean _stopFireRequested = false;
  // Instance count of output and error threads, used for debugging.
  // When the value is greater than 1000, we reset it to 0.
  private static int _streamReaderThreadCount = 0;
  private boolean _debugging = false;    
  }