/* Base class for actors that process collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

 package org.nddp;
  
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.nddp.CollectionHandler.TokenDisposition;
import org.nddp.exceptions.CollectionException;
import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.LoopTerminationToken;
import org.nddp.tokens.MetadataToken;
import org.nddp.util.Parameters;

import ptolemy.actor.IOPort;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.Port;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.NameDuplicationException;
 
/**
 * Base class for actors that process collections. The class simplifies the 
 * writing of such actors by providing the following services:
 * <ul>
 * <li>Provides a fire method that calls a handler method for each data token 
 * received.
 * <li>Provides implementations of the non-data handler methods that call 
 * corresponding handlers when a new start of a new collection arrives, when
 * the end of a collection arrives, when a metadata token arrives, etc.
 * <li>Creates collection managers for each incoming collection and passes 
 * references to the collection event handlers.
 * <li>Allows subclasses to specify whether to process and whether to forward
 * on the collections that are received by the actor.  The subclass is not
 * informed of arriving tokens and subcollections part of a collection that 
 * the subclass has chosen not to process.
 * <li>Provides a parameter for specifying types of collections and data that 
 * should be processed.  All other collections and data are passed through 
 * transparently.
 * </ul>
 * 
 *  @author Timothy M. McPhillips
 */

	public class AtomicCoactor extends TypedAtomicActor implements Coactor {


    /** 
    	 * Constructs an instance of CollectionActor.
   	 * 
   	 * @param container The run-time container of this actor.
 	 * @param name The name of this actor.
    	 */
	public AtomicCoactor(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {

    	    // call the superclass constructor
        	super(container, name);
            
        	collectionPath =  Parameters.stringParameter(this, 
    	        "collectionPath", "");
        	
     	// define the icon to represent the actor in Vergil
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"0\" y=\"0\" "
                + "width=\"60\" height=\"20\" "
                + "style=\"fill:white\"/>\n" +
                "</svg>\n");   
    }

	///////////////////////////////////////////////////////////////////
	////                         public variables                  ////

    public Parameter collectionPath;
	
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    /** 
     * Handles changes in parameter values.
     *  
     * @param attribute The parameter with a new value.
     */
  	public final void attributeChanged(Attribute attribute) 
		throws IllegalActionException {
            
        if (attribute == collectionPath) {
//        		_collectionPath = new CollectionPath(collectionPath);

        } else {
        	    
        	    if (attribute instanceof Parameter) {
	        	    _handleParameterChange((Parameter)attribute, 
	        	            ((Parameter)attribute).getToken());
        	    } else {
        	        super.attributeChanged(attribute);
        	    }
        	}   
  	}
  	
    /** 
     * The method called by the director when the actor is fired. The method
     * reads the next data token from the input port, passes the data token to
     * the handleData method, and then passes the data on via the current
     * collection manager if the collection has not been discarded and the 
     * handleData method did not return DO_NOT_SEND_DATA.
     * @throws IllegalActionException
     * @throws 
     */
    public void fire() throws IllegalActionException {
 
        // call superclass method
        super.fire();
        
        // return immediately if actor has no active input ports
       	if (_inputPort == null) {
       	    return;
       	}
        
        // read next data token from input port, passing a value
        // of false to the unwrapData parameter so that DataTokens
        // can be checked for proper nesting in collections
        Token token = _inputPort.get(false);
        
        if (token != null) {
            _collectionHandler.handleDataToken(token);
        }
    	}   

  	/** 
    	 * Initializes state of actor at start of workflow execution.
    	 */
	public void initialize() throws IllegalActionException {
 
	    // call superclass constructor
        	super.initialize();

        // validate and construct the collection path
        _collectionPath = new CollectionPath(collectionPath);

        // iterate over previously overriden parameters
        for (Iterator i = _overriddenParametersMap.keySet().iterator(); 
                i.hasNext(); ) {
            
            // change the parameter values back to their defaults
            Parameter parameter = (Parameter)i.next();
            Token value = parameter.getToken();
            _handleParameterChange(parameter, value);
        }
        
        _overriddenParametersMap.clear();

        _collectionHandler = new CollectionHandler(this, _collectionPath,
                _overridableParametersMap, _overriddenParametersMap);    
	}
   	
   	    
	/* (non-Javadoc)
     * @see org.nddp.CollectionActor#manageNewCollection(org.nddp.Collection, org.nddp.CollectionManager)
     */
	public CollectionManager manageNewCollection(
        	Collection collection, CollectionManager parentCollectionManager)
		throws IllegalActionException {

		return new NewCollectionManager(collection, 
		        parentCollectionManager, parentCollectionManager.outputPort());
	}	
	
	/* (non-Javadoc)
     * @see org.nddp.CollectionActor#manageNewCollection(org.nddp.Collection, org.nddp.CollectionManager, org.nddp.CollectionPort)
     */
	public CollectionManager manageNewCollection(
        Collection collection, CollectionManager parentCollectionManager, 
        CollectionPort outputPort) throws IllegalActionException {

		return new NewCollectionManager(
		        collection, parentCollectionManager, outputPort);
	}
	
	/** 
     * Adds a port to the actor at runtime.  This method is called by 
     * Vergil when the user interactively adds ports to actors from within 
     * Vergil and when an actor containing such a port is initialized from
     * its MOML description. This method should not be called under any 
     * other circumstances.
     * 
     * @param name The name of the port to create.
     * @return The new port.
     */ 
    	public Port newPort(String name) 
    		throws NameDuplicationException {
    	    
    		try {
    		    // get a write mutex for the workspace
            	_workspace.getWriteAccess();
            	
            	// construct the new port
            	CollectionIOPort port = CollectionIOPort.createInputPort(this, name);
            	
            	// return the port
            	return port;
            	
        	} catch (IllegalActionException ex) {
        	    
            	// translate to a runtime exception since this should never 
        	    // happen.
            	throw new InternalErrorException(this, ex, null);
            	
        	} finally {
        	    
        	    	// release the workspace mutex
            	_workspace.doneWriting();
        	}
    	}
    	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

	protected final void _enableParameterOverride(Parameter parameter)
		throws IllegalActionException {
        
        if (parameter == null) {
            throw new NullPointerException(
                "Null parameter not allowed.");
        }
        
	    if (parameter.getContainer() != this) {
	        throw new IllegalActionException(
                "Parameter '" + parameter.getName() + 
                "' is not contained by actor " + this.getName() + ".");
	    }
	    
        _overridableParametersMap.put(parameter.getName(), parameter);
	}

	protected final void _enableParameterOverride(Parameter[] parameters) 
		throws IllegalActionException {
	    
	    if (parameters == null) {
	        throw new NullPointerException(
                "Null parameter array not allowed.");
	    }

	    for (int i = 0; i < parameters.length; i++) {
	        _enableParameterOverride(parameters[i]);
	    }
	}

	public CollectionIOPort _getDefaultOutputPort() {
	    return null;
	}
	
	protected final CollectionIOPort _getOutputPortNamed(String portName) {
	    
	    if (portName == null) {
	        throw new NullPointerException("Null portName not allowed.");
	    }
	    
	    CollectionIOPort port = null;
	    
	    for (Iterator i = outputPortList().iterator(); i.hasNext(); ) {
	        
	        IOPort nextPort = (IOPort)i.next();
	        
	        if (nextPort.getName().equals(portName)) {
	            
	            port = (CollectionIOPort)nextPort;
	            break;
	        }
	    }
	    
	    return port;
	}

	public void _handleCollectionEnd(CollectionManager collectionManager) 
		throws IllegalActionException, CollectionException {
    }

	public CollectionHandler.ExceptionDisposition handleCollectionException(CollectionManager
        collectionManager, CollectionException exception) {

		return CollectionHandler.PROPAGATE_EXCEPTION;
	}
	
    /** 
    	 * Default collection-start handler for collection actors. This
    	 * implementation simply returns 
    	 * {@link #PROCESS_AND_FORWARD_COLLECTION}. Sub-classes may override
    	 * this method to do actor-specific tasks when a new collection
    	 * arrives.  Sub-classes should use the return value to specify 
    	 * what should be done with the collection.  Subclasses may return
    	 * one of four values:
    	 * <p>
    	 * <ul>
    	 * <li>{@link #PROCESS_AND_FORWARD_COLLECTION}
    	 * <li>{@link #PROCESS_AND_DISCARD_COLLECTION}
    	 * <li>{@link #IGNORE_AND_FORWARD_COLLECTION}
    	 * <li>{@link #IGNORE_AND_DISCARD_COLLECTION}
    	 * </ul>
    	 * The <code>handleData</code>, <code>handleMetadata</code>, and
    	 * <code>handleCollectionEnd</code> methods will be called 
    	 * for each data token, each metadata token, and the end delimiter, 
    	 * respectively, received by the actor if the return values indicates 
    	 * that the collection is processed (not ignored). The collection will
    	 * be passed through to succeeding actors in the workflow if the
    	 * collection is forwarded (not discarded).
    	 * 
    	 * @param collectionManager A reference to the 
    	 * {@link org.nddp.CollectionManager} object associated 
    	 * with the incoming collection.
    	 * @return A {@link CollectionDisposition} value of 
    	 * {@link #PROCESS_AND_FORWARD_COLLECTION}. 
    	 */
   	public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager 
        collectionManager) throws IllegalActionException, CollectionException {
   	    
   		return CollectionHandler.PROCESS_AND_FORWARD_COLLECTION;
   	}
   	
	/** 
    	 * Default data token handler for collection actors. This 
    	 * implementation simply returns  {@link #FORWARD_TOKEN}.  Sub-classes
    	 * may override this method in order to receive each data token as it
    	 * arrives.  Sub-classes should use the return value to specify what
    	 * should be done with the token.  Subclasses may return one of two 
    	 * values:
    	 * <p>
    	 * <ul>
    	 * <li>{@link #FORWARD_TOKEN}
    	 * <li>{@link #DISCARD_TOKEN}
    	 * </ul>
    	 * The data will be forwarded on to the succeeding actors in the
    	 * workflow if {@link #FORWARD_TOKEN} is returned and the return value
    	 * from {@link #handleCollectionStart} was 
    	 * {@link #PROCESS_AND_FORWARD_COLLECTION}.  Otherwise, the data token
    	 * is discarded.
    	 * 
    	 * @param collectionManager A reference to the 
    	 * {@link org.nddp.CollectionManager} object associated 
    	 * with the collection containing the data token.
    	 * @param token The data token.
    	 * @return A {@link CollectionDisposition} value of 
    	 * {@link #PROCESS_AND_FORWARD_COLLECTION}. 
	 * @throws CollectionException
	 * @throws 
    	 */
   	public CollectionHandler.TokenDisposition _handleData(CollectionManager collectionManager, 
        Token token) throws IllegalActionException, CollectionException {
 		
   	        return _defaultDataDisposition;
  	}
   	
   	public CollectionHandler.TokenDisposition _handleDomainObject(CollectionManager 
        collectionManager, DomainObject object) throws IllegalActionException,
        CollectionException {
   	    
   	    return _defaultDataDisposition;
   	}
   	
	public CollectionHandler.TokenDisposition _handleExceptionToken(CollectionManager
        collectionManager, ExceptionToken token) throws IllegalActionException,
        CollectionException  {
	    
	    return CollectionHandler.FORWARD_TOKEN;
    }
    
    public void handleLoopTermination(LoopTerminationToken token) throws IllegalActionException {
            
            throw new IllegalActionException("Actor " + this + 
                    " does not support loop termination tokens");
    }
    
	public CollectionHandler.TokenDisposition _handleMetadata(CollectionManager collectionManager, 
        MetadataToken token) throws IllegalActionException, 
        CollectionException {
  	    
  	    return _defaultMetadataDisposition;
    }

  	public void _handleParameterChange(Parameter parameter, Token value) 
		throws IllegalActionException {
  	    
		super.attributeChanged(parameter);
    }

	protected final CollectionPort _inputPort() {
	    return _inputPort;
	}
    
    protected final void _setDefaultDataDisposition(TokenDisposition disposition) {
        _defaultDataDisposition = disposition;
    }

    public TokenDisposition defaultDataDisposition() {
        return _defaultDataDisposition;
    }
    
    protected final void _setDefaultMetadataDisposition(TokenDisposition disposition) {
        _defaultMetadataDisposition = disposition;
    }

    /** 
    	 * Sets the input port used for receiving collections.  This method is
    	 * called by the {@link org.nddp.coactors.CollectionSource} class.  
    	 * Subclasses of this class need not call this function.
   	 * 
 	 * @param port The name of the port to use for receiving collections.
    	 */
    protected final void _setInputPort(CollectionPort port) {
	    
	    // set the input port for receiving collections to the passed value
		_inputPort = port;
	}
    
    public CollectionHandler collectionHandler() {
        return _collectionHandler;
    }
    
    TestProbe createTestProbe() {
        return new TestProbe();
    }

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
    protected CollectionHandler _collectionHandler;
    
    private CollectionPath _collectionPath;
    
    private TokenDisposition _defaultDataDisposition = CollectionHandler.FORWARD_TOKEN;

    private TokenDisposition _defaultMetadataDisposition = CollectionHandler.FORWARD_TOKEN;
    
    private CollectionPort _inputPort;

    private final Map _overridableParametersMap = new HashMap();
    
    private final Map _overriddenParametersMap = new HashMap();       
    
    private static final long serialVersionUID = 1L;
    
    class TestProbe {
        
        public Map overridableParametersMap() {
            return _overridableParametersMap;
        }
    }
}
