/* Class representing the type of a data object.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;

import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.WeightVector;
import org.nddp.proteins.ProteinAtom;
import org.nddp.proteins.ProteinSequence;

import ptolemy.data.DoubleToken;
import ptolemy.data.IntToken;
import ptolemy.data.ScalarToken;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.util.IllegalActionException;

public class DataTypes {

    private DataTypes() {}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

	public static double getDoubleProperty(DomainObject object, String property) 
		throws IllegalActionException {

	    Object returnedObject = getProperty(object, property);
		
		if (returnedObject instanceof Integer) {
		    
		    return ((Integer)returnedObject).doubleValue();
		    
		} else if (returnedObject instanceof Double) {
		    
		    return ((Double)returnedObject).doubleValue();
		    
		} else {
		    
		    return Double.parseDouble(returnedObject.toString());
		}
	}
	
	public static String getStringProperty(DomainObject object, String property)
		throws IllegalActionException {
	    
	    Object returnedObject = getProperty(object, property);
	    
		return returnedObject.toString();
	}
		
	public final static boolean isDefined(String typeName) {
	    
	    // explicitly disallow null typeName
	    if (typeName == null) {
		    throw new NullPointerException(
            		"Null typeName not allowed.");
		}
	    
	    // check for empty string specifically because it is defined in the
	    // _typeMap table but has a special meaning
	    if (typeName.equals("")) {
		    return false;
		}
        
 	    return typeMap.containsKey(typeName);
 	}
 	
	public final static Class valueOfParameter(Parameter 
        dataTypeParameter) throws IllegalActionException {

		if (dataTypeParameter == null) {
		    throw new NullPointerException(
            		"Null dataTypeParameter not allowed.");
		}

		return valueOfString(((StringToken)
		        (dataTypeParameter.getToken())).stringValue());
	}

	public final static Class valueOfString(String typeName) 
		throws IllegalActionException  {
		
	    if (typeName == null) {
	    		throw new NullPointerException(
    		        "Null typeName not allowed.");
		}
	    
		Class typeClass = (Class)typeMap.get(typeName);

		if (typeClass == null) {
	    		throw new IllegalActionException(
    		        "No data type "+ typeName);
		}
		
		return typeClass;
    }

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 
	
	protected static void clear() {
	    typeMap.clear();
	}
	
	protected static Object getProperty(DomainObject object, String property)
		throws IllegalActionException {
	    
		Class objectClass = object.getClass();
		Object returnedObject = null;
		Method method = null;
		
		try {
			method = objectClass.getMethod(property, null);
		}
		catch(Exception exception) {
		    throw new IllegalActionException( 
	            "Unable to access " + property + "() method for " + object.getClass());
		}
		
		try {
		    returnedObject = method.invoke(object,null);
		    
		} catch(Exception exception) {
		    throw new IllegalActionException( 
	            "Error invoking " + property + "() method on " + object.getClass());
		}
		
		if (returnedObject == null) {
		    throw new NullPointerException(
		    		"Null returned for property " + property + " for " + object.getClass());
		}
		
		return returnedObject;
	}

	public final static void register(String typeName, Class typeClass) {
	    
		assert typeName != null: "Null typeName not allowed.";
		assert typeClass != null: "Null typeClass not allowed.";
		assert typeClass != NoMatch.class: "Class NoMatch not allowed";
		
		typeMap.put(typeName, typeClass);
	}
 	
 	protected static void registerDefaultDataTypes() {

	    // map empty type name to Object class
    	    register("", Object.class);
        register("Object", Object.class);
        register("DomainObject", DomainObject.class);
    	    
    	    // add Ptolemy built-in token types
        register("Token", Token.class);
    	    register("Double", DoubleToken.class);
    	    register("Integer", IntToken.class);
    	    register("Scalar", ScalarToken.class);
    	    register("String", StringToken.class);
    	    
    	    // add domain object types
//	    register("Tree", Tree.class);
//	    register("CharacterMatrix", CharacterMatrix.class);
//	    register("DistanceMatrix", CharacterMatrix.class);
//	    register("WeightVector", WeightVector.class);
//	    register("ProteinAtom", ProteinAtom.class);
//    	   	register("ProteinSequence", ProteinSequence.class);
	}

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private final static Map typeMap = new HashMap(); 
	
	///////////////////////////////////////////////////////////////////
	////                    public inner classes                   ////

	public final static class NoMatch {}

	///////////////////////////////////////////////////////////////////
	////                   static initializers                     ////
 

	static { 
	    registerDefaultDataTypes();
	}
}
