/* CollectionManager for managing collections received by an actor.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.nddp.tokens.ExceptionToken;
import org.nddp.tokens.MetadataToken;

import ptolemy.data.Token;
import ptolemy.data.expr.Parameter;
import ptolemy.kernel.util.IllegalActionException;

/** 
  @author Timothy M. McPhillips
*/ 

class IncomingCollectionManager extends CollectionManager {

	IncomingCollectionManager(Collection collection, CollectionManager 
        	parentCollectionManager, CollectionPort outputPort) 
		throws IllegalActionException {
		
		super(collection, parentCollectionManager, outputPort);
	}		
		
	///////////////////////////////////////////////////////////////////
	//// public methods for CollectionManager.NewCollectionManager //// 

	public void closeCollection() throws IllegalActionException {
	    
		    assert ! _discarded: "Collection has been discarded";
	    _sendCollection();
	}
	
	public boolean discarded() {
	    
		return _discarded;
	}

	public boolean ignored() {
	    
		return _ignored;
	}
		
	///////////////////////////////////////////////////////////////////
	////      package-protected methods for NewCollectionManager   //// 

	void _announceFirstData() throws IllegalActionException {
	    
	    assert ! _firstDataReceived: 
	        "method should be called at most once per instance";
	        
		_firstDataReceived = true;
	    _sendOpeningDelimiterAndMetadata();
	}

	void _discard() {

	    	assert ! _openingDelimiterSent:
	        "Cannot discard collection after opening delimiter" + 
	        " has been sent";

	    _discarded = true;
	}

	boolean _firstDataReceived() {
	    
		return _firstDataReceived;
	}

	void _ignore() {
	    
	    _ignored = true;
	}
	
	Token _oldParameterValue(Parameter parameter) {
	    
	    return (Token)_oldParameterValues.get(parameter);
	}
	
	HashMap _oldParameterValues() {
	    
	    return _oldParameterValues;
	}
	
	void _overrideParameter(Parameter parameter, Token oldValue) {
	    
	    _oldParameterValues.put(parameter, oldValue);
	}
	
	void _passData(Token token) throws IllegalActionException {
		
		assert ! _collectionClosed: "collection is closed";			
		
		// send the data token
		_sendToken(token);
	}
	
	void _sendCollection() throws IllegalActionException {
	
	    	// only do the following if closing delimter has 
	    // not been sent yet
		if (!_collectionClosed) {
		
			// send the opening delimiter if it has not been sent
			if (!_openingDelimiterSent) {
				
				_sendOpeningDelimiterAndMetadata();
			}
			
			// send any new data tokens
			_sendNewDataTokens();
			
			// tell children collection managers to send close delimiters
			_sendChildCollections();	
			
			// send any posterior metadata
			_flushMetadata();
			
			// send any old exception tokens
			_sendInputExceptions();
			
			// send any new exception tokens
			_sendNewExceptions();
			
			// transmit closing delimiter
			_sendClosingDelimiterToken();
		}
	}

	void _storeInputExceptionToken(ExceptionToken token) {

	    // check preconditions
		assert ! _collectionClosed: "collection is closed";			
	    
		// add exception token to list
	    _inputExceptionList.add(token);
	    
	    _registerException();
	}

	void _storeInputMetadata(MetadataToken token) {
	    
    	    // check preconditions
		assert ! _collectionClosed: "collection is closed";			

		// store the metadata token contents in the collection 
		// metadata table
		_metadataTable.put(token.key(), token.value());

		// add metadata token to list of tokens to be sent if and when 
		// the collection is forwarded
		_inputMetadataTokenMap.put(token.key(), token);
	}		
	
	///////////////////////////////////////////////////////////////////
	/// private methods for CollectionManager.NewCollectionManager //// 

	private void _flushMetadata() throws IllegalActionException {
		
	    // override input metadata with keys matching new metadata
		    	for (Iterator i = _newMetadataTokenMap.keySet().iterator(); 
		    			i.hasNext(); ) {
		    	    
		    	    String key = (String)i.next();
		    	    
		    	   	if (_inputMetadataTokenMap.get(key) != null) {
		    	   	    
		    	   	    Token newValue = 
		    	   	        (Token)_newMetadataTokenMap.get(key);
		    	   	    _inputMetadataTokenMap.put(key, newValue);
		    	   	    i.remove();
		    	   	}
	    	}
	    
		// send the input metadata
		for (Iterator i = _inputMetadataTokenMap.keySet().iterator();
				i.hasNext(); ) {
			String key = (String)i.next();
			_sendToken((Token)(_inputMetadataTokenMap.get(key)));
		}
		
		_inputMetadataTokenMap.clear();
		
		// send the new metadata token if metadata defined
		if (_sendNewMetadataTokens()) {
			_clearNewMetadataList();
		}	
	}		
	
	private void _sendInputExceptions() throws IllegalActionException {
	    
	    for (Iterator i = _inputExceptionList.iterator(); i.hasNext(); ) {
			_sendToken((Token)(i.next()));		        
	    }
	}
	
	private void _sendOpeningDelimiterAndMetadata() 
		throws IllegalActionException {

		// send the opening delimiter only if it has not yet been sent 
	    // for the managed collectoin
		if (! _openingDelimiterSent) {
				
			if (parentCollectionManager() != null) {
				((IncomingCollectionManager)parentCollectionManager()).
					_sendOpeningDelimiterAndMetadata();
			}
			
			// send the opening delimiter
			_sendOpeningDelimiterToken();
			
			// remember that the opening delimiter has been sent
			_openingDelimiterSent = true;
		}
		
		// flush any metadata that has been queued up for the collection 
		// if requested by caller
		_flushMetadata();
	}

	///////////////////////////////////////////////////////////////////
	// private variables for CollectionManager.NewCollectionManager /// 

	private boolean _discarded = false;

	private boolean _firstDataReceived = false;

	private boolean _ignored = false;

	private final Map _inputMetadataTokenMap = new HashMap();
	
	private final HashMap _oldParameterValues = new HashMap();
	
	private boolean _openingDelimiterSent = false;
}