/* TODO One line description of class.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylip;

import java.io.IOException;
import java.util.Map;

import org.nddp.actors.EnvironmentActor;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.phylip.PhylipFiles;
import org.nddp.util.Parameters;
import org.nddp.util.Port;
import org.nddp.util.ProcessEnvironment;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class WritePhylipCharacterMatrix extends EnvironmentActor {

    public WritePhylipCharacterMatrix(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an input port for receiving an array of phylogenetic trees
        characterMatrix = new TypedIOPort(this, "characterMatrix", true, false);
        characterMatrix.setTypeEquals(BaseType.OBJECT);
        
        // create an output port for sending a map of taxon abbreviations
        taxonAbbreviations = new TypedIOPort(this, "taxonAbbreviations", false, true);
        taxonAbbreviations.setTypeEquals(BaseType.OBJECT);

        // create a parameter for optionally specifying the name of the trees file
        fileName = 
            Parameters.stringParameter(this, "fileName", "infile");
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort characterMatrix;
    public TypedIOPort taxonAbbreviations;
    public Parameter fileName;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public final void attributeChanged(Attribute attribute) 
        throws IllegalActionException {
        
        if (attribute == fileName) {
        
            _fileName = Parameters.stringValue(fileName);
    
        } else {
            super.attributeChanged(attribute);
        }
    }        
    public void fire() throws IllegalActionException {
 
        // call superclass method
        super.fire();
        
        // get a reference to the process environment
        ProcessEnvironment processEnvironment = receiveEnvironment();
        
        // extract the reference to the process environment
        CharacterMatrix matrix = (CharacterMatrix)Port.getObject(characterMatrix, 0, true);
        
        // Write list of trees to an input file for Phylip
        Map taxonAbbreviationsMap;
        try {
            taxonAbbreviationsMap = PhylipFiles.writeCharacterMatrixFile(
                matrix, processEnvironment.createInputFile(_fileName));
            
        } catch (IOException ex) {
            throw new IllegalActionException("Error writing infile");
        } catch (ExternalEnvironmentException e) {
            throw new IllegalActionException("Error writing infile");
        }

        // forward the process environment
        taxonAbbreviations.broadcast(new ObjectToken(taxonAbbreviationsMap));
        
        // forward the process environment
        sendEnvironment();
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                    private variables                      ////

    private String _fileName;
    private static final long serialVersionUID = 1L;
}
