/* Actor that parses Trees block of a Nexus-formatted string.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.actors.phylogeny;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.nddp.exceptions.ParseException;
import org.nddp.phylogeny.Tree;
import org.nddp.util.Port;

import ptolemy.actor.TypedIOPort;
import ptolemy.data.ArrayToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.Token;
import ptolemy.data.type.ArrayType;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.actor type="phylogenetics"
 *
 *  @author Timothy M. McPhillips
 */

public class ParseNexusTreesBlock extends NexusParsingActor {

    public ParseNexusTreesBlock(CompositeEntity container, String name)
    throws NameDuplicationException, IllegalActionException  {
        
        super(container, name);

        // create an output port for sending the character matrix
        treesPort = new TypedIOPort(this, "trees", false, true);
        treesPort.setTypeEquals(new ArrayType(BaseType.OBJECT));
        
        // precompile the pattern for finding the named Nexus block
        setBlockName("TREES");
        }

    ///////////////////////////////////////////////////////////////////
    ////                         public variables                  ////

    public TypedIOPort treesPort;
    
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    public void fire() throws IllegalActionException {
        
        super.fire();
        
        if (! actorEnabled()) {
            return;
        }

        // read the nexus file content string from the input port       
        String nexusFileString = Port.getString(nexusFilePort, 0, true);

        // extract the characters block
        String treesBlockString = extractNexusBlock(nexusFileString);

        List treeList = new LinkedList();
        
        // parse the characters block if it was found
        if (treesBlockString != null) {
            
            NexusCommand translateCommand = NexusCommand.extractNexusCommand("TRANSLATE", treesBlockString);
            
            Map abbreviationsMap = new HashMap();
            
            for (Iterator i = translateCommand.argumentIterator(); i.hasNext(); ) {
            
                abbreviationsMap.put(i.next(), i.next());
            }

            List treeCommandList = NexusCommand.extractNexusCommands("TREE", treesBlockString);

            for (Iterator treeCommandIterator = treeCommandList.iterator(); treeCommandIterator.hasNext(); ) {
                
                NexusCommand treeCommand = (NexusCommand)treeCommandIterator.next();
                
                Iterator propertyIterator = treeCommand.propertyIterator();
                
                if (propertyIterator.hasNext()) {
                    String allCapsTreeName = (String)propertyIterator.next();
                    String treeName = treeCommand.originalPropertyKey(allCapsTreeName);
                    String newickString = treeCommand.getProperty(allCapsTreeName);
                    
                    Tree tree;
                    
                    try {
                        tree = new Tree(newickString,  abbreviationsMap);
                    } catch (ParseException e) {
                        throw new IllegalActionException("Error parsing newick string.");
                    }
                    
                    tree.setName(treeName);
                    treeList.add(tree);
                }
            }
            
            // create an array for the trees
            Token[] tokenArray = new Token[treeList.size()];

            // iterate over trees in the list
            for (int i = 0; i < tokenArray.length; i++) {
                tokenArray[i] = new ObjectToken(treeList.get(i));
            }
            
            ArrayToken arrayToken = new ArrayToken(tokenArray);
            
            // send the arrayToken containing the trees
            treesPort.broadcast(arrayToken);
        }
    }
 
    private static final long serialVersionUID = 1L;
}
