/* Actor that reads a text file and outputs contents as one string token.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */

package org.nddp.coactors;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import java.io.BufferedReader;
import java.io.IOException;

import org.nddp.Collection;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.TextFileCollection;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.FileReadException;
import org.nddp.util.FileOrUrl;

/**
 * @nddp.coactor type="io"
 */
public class FileReader extends CollectionTransformer {
 
    public FileReader(CompositeEntity container, String name)
        throws IllegalActionException, NameDuplicationException {
        
        super(container, name);

        _attachText("_iconDescription", "<svg>\n"
                + "<rect x=\"-25\" y=\"-20\" "
                + "width=\"50\" height=\"40\" "
                + "style=\"fill:white\"/>\n"
                + "<polygon points=\"-15,-10 -12,-10 -8,-14 -1,-14 3,-10"
                + " 15,-10 15,10, -15,10\" "
                + "style=\"fill:red\"/>\n"
                + "</svg>\n");
    }
   
	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 

    public CollectionHandler.TokenDisposition _handleData(CollectionManager 
        	parentCollectionManager, Token token) 
    		throws IllegalActionException, CollectionException {
    	
		// read the next file name from the input port       
		String fileName = ((StringToken)token).stringValue();
		
		// open the file
		FileOrUrl fileOrUrl;
		BufferedReader reader;
		
		try {
		    
	        	fileOrUrl = new FileOrUrl(fileName);
	         reader = fileOrUrl.open();
	         
		} catch (NameDuplicationException exception) {
		    
        		throw new FileReadException(
        			"Name duplication while opening file " + fileName + 
        			", " + exception.getMessage()); 
        		
        	} catch (IllegalActionException exception) {
        	    
        		throw new FileReadException(
    		        "Illegal action while opening file " + fileName + 
    		        ", " + exception.getMessage());             
        }

		// TODO handle int overflow on conversion from long
		StringBuffer fileContents = new StringBuffer((int) fileOrUrl.length());
		
		try {
			// read file line by line, adding each line to the file collection 
		    // as a data token
			String line;
			while ( (line = reader.readLine()) != null) {
				fileContents.append(line + "\n");
			}
	
			// close the file
			reader.close();
		
		} catch (IOException exception) {
		    
            throw new FileReadException(
            		"IOException while reading file " + fileName + 
            		", " + exception.getMessage());
       	}    

		// create a new collection to hold the file contents
		Collection collection = 
		    new TextFileCollection(fileOrUrl.unqualifiedFileName());
		
        	CollectionManager collectionManager = 
        		manageNewCollection(
        		        collection, parentCollectionManager, output);
        	
		collectionManager.addMetadata(
		        "FilePath", new StringToken(fileOrUrl.name()));
	    
	    collectionManager.addDataToken(
	            new StringToken(fileContents.toString()));	
		
        // do not forward on the data token containing the file name
        return  CollectionHandler.DISCARD_TOKEN;
    }
}
