/* Actor that transforms text file collections into Nexus collections.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.coactors.phylogeny;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.nddp.Collection;
import org.nddp.AtomicCoactor;
import org.nddp.CollectionHandler;
import org.nddp.CollectionManager;
import org.nddp.coactors.CollectionTransformer;
import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.ParseException;
import org.nddp.phylogeny.NexusBlock;
import org.nddp.phylogeny.NexusBlockHandler;
import org.nddp.phylogeny.NexusCollection;
import org.nddp.util.Parameters;

import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

/**
 * @nddp.coactor type="phylogenetics"
 */
public final class NexusFileParser extends CollectionTransformer {

	public NexusFileParser(CompositeEntity container, String name)
        throws NameDuplicationException, IllegalActionException  {
	
	    super(container, name);
	    
	    Parameters.fix(collectionPath, "TextFile/String");
	}

	///////////////////////////////////////////////////////////////////
	////                      protected methods                    //// 
	
    public CollectionHandler.CollectionDisposition _handleCollectionStart(CollectionManager
        collectionManager) throws IllegalActionException {
    
       	    return  CollectionHandler.PROCESS_AND_DISCARD_COLLECTION;     	    
    }

    public CollectionHandler.TokenDisposition _handleData(CollectionManager
        fileCollectionManager, Token token) throws IllegalActionException, 
        CollectionException {
	    	    
	    // get Nexus file contents from token
		String nexusFileString = ((StringToken)token).stringValue();
		
		// make sure this is a NEXUS file
	    if (! nexusFileString.substring(0,6).toUpperCase().equals("#NEXUS")) {
	        throw new ParseException("Not a Nexus file.");
		}

	    // create a Nexus collection in the parent of the text file collection
	    Collection nexusCollection = 
	        new NexusCollection(fileCollectionManager.collection().name());
	    CollectionManager nexusCollectionManager = 
	        manageNewCollection(nexusCollection, 
	                fileCollectionManager.parentCollectionManager());
	    
	    nexusCollectionManager.addMetadata("FilePath", 
	            fileCollectionManager.metadataValue("FilePath"));
	    
	    // remove comment blocks from Nexus file
	    nexusFileString = 
	        NEXUS_COMMENT_PATTERN.matcher(nexusFileString).replaceAll("");
	    
	    // remove whitespace around = signs
	    nexusFileString = 
	        NEXUS_EQUALS_PATTERN.matcher(nexusFileString).replaceAll("=");

	    // parse file for Nexus blocks, calling the block handler for each
	    	Matcher nexusBlockMatcher = 
	    	    NEXUS_BLOCK_PATTERN.matcher(nexusFileString);
	    	
		while (nexusBlockMatcher.find()) {
		    NexusBlockHandler.handleNexusBlock(
		            new NexusBlock(nexusBlockMatcher.group()), 
		            		nexusCollectionManager);
	    }
		
		// discard the String version of the Nexus file
		return  CollectionHandler.DISCARD_TOKEN;
    }

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private static final Pattern NEXUS_COMMENT_PATTERN = 
	    Pattern.compile("\\[.*?\\]", Pattern.DOTALL);
	
    private static final Pattern NEXUS_EQUALS_PATTERN = 
        Pattern.compile("\\s*=\\s*", Pattern.DOTALL);
    
	private static final Pattern NEXUS_BLOCK_PATTERN = 
	    Pattern.compile("BEGIN.*?END\\s*?;", 
	            Pattern.DOTALL | Pattern.CASE_INSENSITIVE);
}