/* Base class for Nexus block handlers.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.nddp.CollectionManager;
import org.nddp.exceptions.IndexOutOfBoundsException;
import org.nddp.exceptions.ParseException;

import ptolemy.kernel.util.IllegalActionException;

public abstract class NexusBlockHandler {
    
	public NexusBlockHandler(NexusBlock block, CollectionManager 
        nexusFileCollectionManager) {
	    
	    super();
	    
	    _block = block;
	    _nexusFileCollectionManager = nexusFileCollectionManager;
	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

	public static void handleNexusBlock(NexusBlock block, CollectionManager 
        nexusFileCollectionManager) throws ParseException, 
        IllegalActionException, IndexOutOfBoundsException {
	    
		Class blockHandlerClass = 
		    (Class) _nexusBlockHandlerMap.get(block.type());
		
		if (blockHandlerClass != null) {
		    
    			Constructor blockHandlerConstructor;
	    		
			try {
				blockHandlerConstructor = 
				    blockHandlerClass.getConstructors()[0];
				
			} catch (SecurityException e) {
				throw new ParseException(
				        "Security exception accesing constructor for " + 
				        blockHandlerClass);
			}
			
			NexusBlockHandler blockHandler = null;
				
			try {
			    
			    blockHandler = 
			        (NexusBlockHandler)blockHandlerConstructor.newInstance(
			                new Object[] {block, nexusFileCollectionManager});
			    
			} catch (InstantiationException e) {
			    
			    throw new ParseException(
			            "Error instantiating Nexus block handler: " + 
			            blockHandlerClass);
			    
			} catch (IllegalAccessException e) {
			    
			    throw new ParseException(
			            "Illegal access instantiating Nexus block handler: " + 
			            blockHandlerClass);
			    
			} catch (IllegalArgumentException e) {
			    
			    throw new ParseException(
			            "Illegal argument for constructor for: " + 
			            blockHandlerClass);
			    
			} catch (InvocationTargetException e) {
			    
			    throw new ParseException(
			            "Illegal invocation target: " + 
			            blockHandlerClass);
			}
  
			blockHandler._handleBlock();
	    }	
	}
	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
	
	protected final void _dispatchCommandHandlers() 
		throws ParseException {
	    
		for (Iterator commandIterator = _block.commandIterator(); 
				commandIterator.hasNext(); ) {
		    
	        NexusCommand command = (NexusCommand) commandIterator.next();
		    NexusCommandHandler handler = 
		        (NexusCommandHandler)_nexusCommandHandlerMap.get(
		                command.name());
		    
			if ( handler != null ) {
			    handler.handle(command);
			}
		}
	}
	
	protected void _handleBlock() throws ParseException, 
		IllegalActionException, IndexOutOfBoundsException {
		
	    _dispatchCommandHandlers();    
	}	
	
	protected static List _parseCharacterGroup(String characterGroup) {
        
        // create a list to store the indices
        List indexList = new LinkedList();
        
        // break the index string into tokens
        StringTokenizer tokenizer = new StringTokenizer(characterGroup);
        
        // iterate over the tokens
        while (tokenizer.hasMoreTokens()) {
            
            // get the next token
            String token = tokenizer.nextToken();
            
            // find position of the dash character if present
            int dashPosition = token.indexOf('-');
            
            // if no dash then token represents just one index
            if (dashPosition == -1) {
                
                // parse the character index and add it to the index list
                indexList.add(new Integer(token.trim()));
            
	        // otherwise the token represents a range of indices
            } else {
            
                // parse the index starting the range
                int startIndex = 
                    Integer.parseInt(token.substring(0, dashPosition).trim());
                
                // parse the index ending the range
                int endIndex = 
                    Integer.parseInt(token.substring(dashPosition + 1).trim());
                
                // iterate over the index range
                for (int i = startIndex; i <= endIndex; i++) {
                    
                    // add each index in the range to the index list
                    indexList.add(new Integer(i));
                }
            }
        }
        
        // return the index list
        return indexList;
    }

	protected static Map _parseCharacterPropertyList(String valueString) {
    
        // create a map to store properties in
        	Map propertyMap = new HashMap();
        
    		// remove whitespace around dashes
    		valueString = 
    		    NEXUS_DASH_PATTERN.matcher(valueString).replaceAll("-");
    
    		// find the property assignment clauses
        	Matcher propertyMatcher = 
        	    NEXUS_CHARACTER_PROPERTY_PATTERN.matcher(valueString);
    
        	// iterate over property assignment clauses
        	while (propertyMatcher.find()) {
            
            	// extract the property value 
        		String propertyString = propertyMatcher.group(1).trim();
        
        		// extract the character indices 
        		List indexList = 
        		    NexusBlockHandler._parseCharacterGroup(
        		            propertyMatcher.group(2));
    
        		// iterator over the character indices in the list
        		for (Iterator i = indexList.iterator(); i.hasNext(); ) {
            
            		// extract the index
            		Integer index = (Integer) i.next();
            
            		// map the index to the property value
            		propertyMap.put(index, propertyString);
        		}
        	}
    
    		return propertyMap;
    }
	
	///////////////////////////////////////////////////////////////////
	////                         protected variables               ////

    	protected final NexusBlock _block;
	protected final Map _nexusCommandHandlerMap = new HashMap();
	protected final CollectionManager _nexusFileCollectionManager;

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////
	
	private static final Map _nexusBlockHandlerMap = new HashMap();
	
    	private static final Pattern NEXUS_CHARACTER_PROPERTY_PATTERN = 
    	    Pattern.compile("(.+?):(.+?)[,;]");

    	private static final Pattern NEXUS_DASH_PATTERN = 
    	    Pattern.compile("\\s*-\\s*", Pattern.DOTALL);
    	

	///////////////////////////////////////////////////////////////////
	////                      static initializers                  ////
    	
    	static {
	    _nexusBlockHandlerMap.put("ASSUMPTIONS", 
	            NexusAssumptionsBlockHandler.class);

	    _nexusBlockHandlerMap.put("CHARACTERS", 
	            NexusCharactersBlockHandler.class);

	    _nexusBlockHandlerMap.put("DATA", 
	            NexusDataBlockHandler.class);
	}
}