/* Class representing a vector of weights for taxonomic characters.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 package org.nddp.phylogeny;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;

import org.nddp.DomainObjectBase;
import org.nddp.util.Xml;


public class WeightVector extends DomainObjectBase {

    public WeightVector(int weightCount) {
        
        super();

        _weightCount = weightCount;
        _weight = new int[weightCount];
        Arrays.fill(_weight, 1);
    }
    
    public WeightVector(WeightVector weightVector) {
        _weightCount = weightVector._weightCount;
        _weight = (int[])weightVector._weight.clone();
    }
    
    public WeightVector(Map weightMap) {

        _weightCount = weightMap.size();
        _weight = new int[_weightCount];
        
        // iterate over the keys (character indices) in the property map
        for (Iterator keyIterator = weightMap.keySet().iterator(); 
        		keyIterator.hasNext(); ) {
	        
	        	// extract the index from the next key
	        	Integer index = (Integer) keyIterator.next();
	        
	        	// extract the weight from the map
	        	String weightString = (String) weightMap.get(index);
	        
	        	// assign the weight in the weight set
	        	_weight[index.intValue() - 1] = Integer.parseInt(weightString);
		}
    }

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 
	
    public int get(int weightIndex) {
        return _weight[weightIndex];
    }

    public void set(int weightIndex, int value) {
        assertNotWriteLocked();
        _weight[weightIndex] = value;
    }

    public int size() {
        return _weightCount;
    }
    
    public String toString() {
		
        StringBuffer buffer = new StringBuffer("[");
		
		for (int i = 0; i < _weightCount; i++) {
		    buffer.append(_weight[i]);
		}
		
		buffer.append("]");
		
		return buffer.toString();
    }
    
    public String xmlAttributeString() {
		
        StringBuffer buffer = new StringBuffer();
		
		for (int i = 0; i < _weightCount; i++) {
		    buffer.append(_weight[i]);
		}

		return Xml.attribute("weights", buffer.toString());
    }
    
    public String xmlContentString(Xml.Indentation indentation) {
        return null;
    }
    
	///////////////////////////////////////////////////////////////////
	////                         protected variables               ////
    
    protected int[] _weight;
    protected int _weightCount;
}
