/* Base class for Phylip tree inference runners.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny.phylip;

import java.io.IOException;

import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.DistanceMatrix;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunner;

public class PhylipDnadistRunner extends ProcessRunner {

    public PhylipDnadistRunner() {
        
        super();
   	}
    
	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

	public Object run() throws CollectionException, ExternalEnvironmentException {
		
	    ProcessEnvironment environment = _createProcessEnvironment("DNADIST");
	    
	    _writeInputFiles(environment);
	    
	    _startProgram(environment, 
	    		System.getProperty("PHYLIP_EXE_DIR") + "/dnadist");
	    
	    _writeStandardInput();

	    _waitForProgramCompletion();
	    
	    DistanceMatrix distanceMatrix = (DistanceMatrix) _parseOutputFiles(environment);
	    
	    _destroyEnvironment(environment);
		
	    return distanceMatrix;
	}

	public void setCharacterMatrix(CharacterMatrix characterMatrix) {
	    _characterMatrix = characterMatrix;
	}
	
	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////
        
    protected void _writeInputFiles(ProcessEnvironment environment)
    		throws CollectionException {
        
	    try {
            PhylipFiles.writeCharacterMatrixFile(
                _characterMatrix, environment.createInputFile("infile"));
        } catch (IOException e) {
			throw new ExternalApplicationException(
		        "Error writing to input file.");
        }
    }
    
    protected void _writeStandardInput() 
    		throws ExternalApplicationException {
        
		try {
		    
			_writeToProcess("L");

			// Accept configuration and begin execution
			_writeToProcess("Y");
		    	
		} catch (IOException e) {
			throw new ExternalApplicationException(
		        "Error writing to DNADIST process");
        	}		    
    }
    
    private Object _parseOutputFiles(ProcessEnvironment environment)
    		throws CollectionException {

	    // Read the output file
	    String outfile = environment.readFile("outfile");
        
        // Parse thefile and return the distance matrix
	    return PhylipFiles.parseDistanceMatrixFileString(outfile);
    }

    ///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private CharacterMatrix _characterMatrix;
}
