/* Base class for Phylip tree inference runners.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.phylogeny.phylip;

import java.io.IOException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.nddp.exceptions.CollectionException;
import org.nddp.exceptions.ExternalApplicationException;
import org.nddp.exceptions.ExternalEnvironmentException;
import org.nddp.exceptions.InconsistentDataException;
import org.nddp.phylogeny.CharacterMatrix;
import org.nddp.phylogeny.Tree;
import org.nddp.phylogeny.WeightVector;
import org.nddp.util.ProcessEnvironment;
import org.nddp.util.ProcessRunner;

public abstract class PhylipTreeInferenceRunner extends ProcessRunner {

    public PhylipTreeInferenceRunner() {
        
        super();
   	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

	public void ignoreAutopomorphies(boolean ignore) {
	    
	    	if (ignore) {
	    	    _characterMatrix.zeroWeightAutapomorphies();
	    	}
	}

	public void rootAtNode(String name) {
	    _rootAtNode = name;
	}
    	
  	public void setCharacterMatrix(CharacterMatrix matrix) 
    		throws InconsistentDataException {
     
	   _characterMatrix = new CharacterMatrix(matrix);
    }
    
	public void setOutgroup(int index) {
	    _outgroupIndex = index;
	}

	public void setWeights(WeightVector weights) 
    		throws InconsistentDataException {
     
	    	if (weights != null) {
	        	_characterMatrix.applyWeightSet(weights);
	    	}
    }

	///////////////////////////////////////////////////////////////////
	////                         protected methods                 ////

   protected List _createTrees(List treeBufferList, String treeIdRoot) 
   		throws InconsistentDataException {
        
		List treeList = new LinkedList();
		
		for (Iterator i = treeBufferList.iterator(); i.hasNext(); ) {
		    
		    Tree tree = (Tree)i.next();

		    if (! _rootAtNode.equals("")) {
			    tree.reroot(_rootAtNode);
		    }
		    
		    tree.setName(treeIdRoot + _treeCounter.next());

		    treeList.add(tree);
	    }
		
		return treeList;
	}
   
    protected int _findTable(String output, int index, String tableTitle, 
            int tableOffset) {
        
	    index = output.indexOf(tableTitle, index);
	 
	    for (int i = 0; i < tableOffset; i++) {
		    index = output.indexOf('\n', index) + 1;
	    }
	    
	    return index;
    }
   	
    protected List _parseOutputFiles(ProcessEnvironment environment) 
    		throws CollectionException {
        
		String outtree;

	    // Read the output tree file
	    outtree = environment.readFile("outtree");

		// Parse the tree file
		List treeBufferList = PhylipFiles.parseTreeFileString(outtree,
	        _taxonAbbreviations);
		
		// return the list of trees
		return treeBufferList;
    }
    
    protected int _parseStateTable(String output, Tree tree, 
        int index) throws InconsistentDataException {

	    int treeNodeCount = tree.nodeCount();

	    String[] parentNameArray = new String[treeNodeCount];
	    String[] childNameArray = new String[treeNodeCount];
	    String[] statesArray = new String[treeNodeCount];

	    for (int i = 0; i < tree.nodeCount(); i++ ) {
	        
	        int lineEndIndex = output.indexOf("\n", index);
	        
	        parentNameArray[i] = 
	            output.substring(index, index + 4).trim();
	        
	        childNameArray[i] = 
	            output.substring(index + 7, index + 17).trim();
	        
	        statesArray[i] = 
	            output.substring(index + 27, lineEndIndex);
	        
	        index = lineEndIndex + 1;
	    }
	    
		tree.setParentNames(childNameArray, parentNameArray);
    
		return index;
    }
    

   	protected List _run(String programName, String programPath) 
   		throws CollectionException {

	    _validateParameters();
	    
	    ProcessEnvironment environment = 
	        _createProcessEnvironment(programName);
	    
	    _writeInputFiles(environment);
	    
	    _startProgram(environment, programPath);
	    
	    _writeStandardInput();

	    _waitForProgramCompletion();
		
	    List treeBufferList = _parseOutputFiles(environment);
		
	    List treeList = _createTrees(treeBufferList, programName + "_Tree_");
		
		_destroyEnvironment(environment);
	    
	    return treeList;
	}   	
    
    protected void _validateParameters() { 
        
		if (_outgroupIndex > _characterMatrix.taxonCount()) {
		    throw new ArrayIndexOutOfBoundsException(
	            "Outgroup index " + _outgroupIndex + 
	            " out of range.");
		}
    }
    
    protected void _writeInputFiles(ProcessEnvironment environment) 
    		throws ExternalEnvironmentException {
        
	    	try {
			_taxonAbbreviations = PhylipFiles.writeCharacterMatrixFile(
		        _characterMatrix, 
	    			environment.createInputFile("infile"));
			
	    	} catch (IOException ex) {
	        	throw new ExternalEnvironmentException("Error writing infile");
	    	}

    	    try {
		    PhylipFiles.writeWeightsFile(
	            _characterMatrix, 
	            environment.createInputFile("weights"));
	    } catch (IOException ex) {
	        throw new ExternalEnvironmentException(
	                "Error writing weights file");
	    }
    }
    
    protected void _writeStandardInput() 
    		throws ExternalApplicationException {
        
		try {
			// enable use of weights
            	_writeToProcess("w");			

            	// optionally specify taxon to use as outgroup
            	if (_outgroupIndex > 0) {
	            	_writeToProcess("O");
	            	_writeToProcess(String.valueOf(_outgroupIndex));
            	}
 
		    	// accept configuration and begin execution
			_writeToProcess("Y");
		    	
		} catch (IOException e) {
			throw new ExternalApplicationException(
		        "Error writing to process");
        	}		    
    }

    ///////////////////////////////////////////////////////////////////
	////                         protected variables               ////

	protected int _outgroupIndex = 0;

	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

	private CharacterMatrix _characterMatrix;
	private String _rootAtNode;	
	private Map _taxonAbbreviations;
	private TreeCounter _treeCounter = TreeCounter.getInstance();
	
	///////////////////////////////////////////////////////////////////
	////                      private inner classes                ////

	private static class TreeCounter {
	    
	    private TreeCounter() {
	    }
	    
	    ///////////////////////////////////////////////////////////////////
		////             public methods for TreeCounter                ////
	    
	    public static TreeCounter getInstance() {
	        return _instance;
	    }
	    
	    public synchronized int next() {
	        return ++count;
	    }

	    ///////////////////////////////////////////////////////////////////
		////          private variables for TreeCounter                ////
	    
	    private int count = 0;
	    private static TreeCounter _instance = new TreeCounter();
	}
}

