/* Class representing external system process.
 *
 * Copyright (c) 2005 Natural Diversity Discovery Project.
 * All rights reserved.
 * 
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 * 
 * IN NO EVENT SHALL THE NATURAL DIVERSITY DISCOVERY PROJECT BE LIABLE 
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR 
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND 
 * ITS DOCUMENTATION, EVEN IF THE NATURAL DIVERSITY DISCOVERY PROJECT
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE NATURAL DIVERSITY DISCOVERY PROJECT SPECIFICALLY DISCLAIMS ANY 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE NATURAL 
 * DIVERSITY DISCOVERY PROJECT HAS NO OBLIGATION TO PROVIDE MAINTENANCE, 
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

   @ProposedRating Red (tmcphillips@naturaldiversity.org)
   @AcceptedRating Red (tmcphillips@naturaldiversity.org) 
 */
 
package org.nddp.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;


public class ExternalProcess {

    public ExternalProcess(String command, Writer outputStreamHandler, 
        	Writer errorStreamHandler, String[] variables, File directory) 
		throws IOException {

        // start the external application as a separate process
        _process = 
            Runtime.getRuntime().exec(command, variables, directory);

        // create a standard input stream for sending commands to the 
        // external application
        _stdInput = new BufferedWriter(new 
            OutputStreamWriter(_process.getOutputStream()));

        // create a standard output stream for reading output from the
        // application
        _stdOutput = new BufferedReader(new 
            InputStreamReader(_process.getInputStream()));

        // create a standard error stream for reading errors from the 
        // application
        _stdError = new BufferedReader(new 
            InputStreamReader(_process.getErrorStream()));
        
        _outputStreamThread = 
            new AsyncStreamRouter(_stdOutput, outputStreamHandler);
        _outputStreamThread.start();
        
        _errorStreamThread = 
            new AsyncStreamRouter(_stdError, errorStreamHandler);
        _errorStreamThread.start();
	}

	///////////////////////////////////////////////////////////////////
	////                         public methods                    //// 

    public final void flushStandardInput() throws IOException {
        _stdInput.flush();
    	}
    
    public int waitFor() {
        
        int returnValue = 0;
        boolean waitSucceeded = false;
        
        while (! waitSucceeded) {
	        try {
		        returnValue = _process.waitFor();
		        _outputStreamThread.join();
		        _errorStreamThread.join();
		        waitSucceeded = true;
	        } catch(InterruptedException e) {}
        }
        
        return returnValue;
    }
        
    public final void write(String text) throws IOException {   
        _stdInput.write(text);
    }

    public final void writeLine(int value) throws IOException {   
        _stdInput.write(String.valueOf(value) + '\n');
    }

    public final void writeLine(String text) throws IOException {   
        _stdInput.write(text + '\n');
    }
    
	///////////////////////////////////////////////////////////////////
	////                         private variables                 ////

    private final Thread _errorStreamThread;
    private final Thread _outputStreamThread;
    private final java.lang.Process _process;
    private final BufferedReader _stdError;    
    private final BufferedWriter _stdInput;
    private final BufferedReader _stdOutput;
}