/* ParameterizedGlobusJob actor gets the inouts and the program to run
   from the user, creates an RSL string from the inputs, submits it as
   a Globus job, and outputs the results.

 Copyright (c) 2000-2003 The Regents of the University of California.
 All rights reserved.
 Permission is hereby granted, without written agreement and without
 license or royalty fees, to use, copy, modify, and distribute this
 software and its documentation for any purpose, provided that the above
 copyright notice and the following two paragraphs appear in all copies
 of this software.
 IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
 THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
 SUCH DAMAGE.
 THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
 CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
 ENHANCEMENTS, OR MODIFICATIONS.
                                        PT_COPYRIGHT_VERSION_2
                                        COPYRIGHTENDKEY
 */

package org.sdm.spa;

import org.nmiworkflow.GridJob;

import ptolemy.actor.TypedIOPort;
import ptolemy.actor.lib.Source;
import ptolemy.data.StringToken;
import ptolemy.data.Token;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.StringAttribute;

//////////////////////////////////////////////////////////////////////////
//// ParameterizedGlobusJob
/**
   This actor takes a Globus RSL and hostname and submits a job to that
   host. The output is gathered and sent to the output port.
   <p>
   The <i>rsl</i> parameter is the Globus RSL that defines the job to be run.
   The <i>host</i> parameter is the hostname where the job will be executed.
   I should add more parameters so the user does not need to specify a full
   rsl. Rsl is annoying.

   @author Ilkay Altintas (Build on top of Steve Mock's GlobusJob.)
   @version $Id: ParameterizedGlobusJob.java,v 1.3 2006/04/04 22:01:13 altintas Exp $
 */


public class ParameterizedGlobusJob extends Source {

  public ParameterizedGlobusJob(CompositeEntity container, String name)
          throws IllegalActionException, NameDuplicationException {
      super(container, name);


      certificate = new TypedIOPort(this, "certificate", true, false);
      certificate.setTypeEquals(BaseType.STRING);

      arguments = new TypedIOPort(this, "arguments", true, false);
      arguments.setMultiport(true);
      arguments.setTypeEquals(BaseType.GENERAL);

      output.setTypeEquals(BaseType.STRING);

      executable = new StringAttribute(this, "Executable path");

      host = new StringAttribute(this, "GlobusHost");

      _attachText("_iconDescription", "<svg>\n"
              + "<rect x=\"-25\" y=\"-20\" "
              + "width=\"50\" height=\"40\" "
              + "style=\"fill:white\"/>\n"
              + "<polygon points=\"-15,-10 -12,-10 -8,-14 -1,-14 3,-10"
              + " 15,-10 15,10, -15,10\" "
              + "style=\"fill:red\"/>\n"
              + "</svg>\n");
  }

  ///////////////////////////////////////////////////////////////////
  ////                     ports and parameters                  ////

  /**
   * The Globus certificate generated by the GlobusProxy actor.
   */
  public TypedIOPort certificate;

  /**
   * Arguments for the job to execute that goes into the rsl string
   */
  public TypedIOPort arguments;

  /**
   * The rsl string defining the job to be run
   */
  public StringAttribute executable;

  /**
   * The host name to run the job on
   */
  public StringAttribute host;


  ///////////////////////////////////////////////////////////////////
  ////                         public methods                    ////

  /** If the specified attribute is <i>fileOrURL</i> and there is an
   *  open file being read, then close that file and open the new one;
   *  if the attribute is <i>numberOfLinesToSkip</i> and its value is
   *  negative, then throw an exception.  In the case of <i>fileOrURL</i>,
   *  do nothing if the file name is the same as the previous value of
   *  this attribute.
   *  @param attribute The attribute that has changed.
   *  @exception IllegalActionException If the specified attribute
   *   is <i>fileOrURL</i> and the file cannot be opened, or the previously
   *   opened file cannot be closed; or if the attribute is
   *   <i>numberOfLinesToSkip</i> and its value is negative.
   */
  public void attributeChanged(Attribute attribute)
          throws IllegalActionException {

      super.attributeChanged(attribute);
  }

  /** Output the data lines into an array.
   *  @exception IllegalActionException If there's no director.
   */

  public void fire() throws IllegalActionException {
      super.fire();

      //get the proxy string fed into the input port
      Token token = certificate.get(0);

      String hostString = host.getExpression();
      _debug("Hostname = " + hostString);
      String exeString  = executable.getExpression();
      _debug("executable = " + exeString);
      String argString = "(arguments=";
      // Get all the arguments in a row.
      // FIXME: The user has to connect the arguments in order right now.
      //        I don't know how to fix this? Ask!
      int i = 0;
      int width = arguments.getWidth();
      for (i = 0; i < width-1; i++) {
        if (arguments.hasToken(i)) {
          Token tokenArg = arguments.get(i);
          String value = tokenArg.toString();
          _debug("arguments(i) = " + value);
          value = value.substring(1, value.length()-1);
          argString += value + " ";
        }
      }
      if (arguments.hasToken(i)) {
        Token tokenArg = arguments.get(i);
          String value = tokenArg.toString();
          _debug("arguments(i) = " + value);
          value = value.substring(1, value.length()-1);
          argString += value;
      }
      argString += ")";

      String rslString = "&(executable=" + exeString + ")" + argString;
      _debug("RSL = " + rslString);

      _proxy = token.toString();
      _job = new GridJob(hostString, false);
      _result = _job.GlobusRun(rslString, _proxy);
      _result += _job.getOut();
      output.broadcast(new StringToken(_result));
  }

  /** Post fire the actor. Return false to indicated that the
  * process has finished. If it returns true, the process will
  * continue indefinitely.
  */

  public boolean postfire() {
      return false;
  }

  /** Open the file or URL and read the first line, and use the
   *  first line to set the type of the output.
   *  @exception IllegalActionException If the file or URL cannot be
   *  opened, or if the first line cannot be read.
   */
  public void preinitialize() throws IllegalActionException {
      super.preinitialize();
  }

  ///////////////////////////////////////////////////////////////////
  ////                         protected members                 ////


  ///////////////////////////////////////////////////////////////////
  ////                         private methods                   ////


  ///////////////////////////////////////////////////////////////////
  ////                         private members                   ////

  private GridJob _job;
  private String _proxy;

  /** Result string Variable. */
  private String _result = new String("");

}
