/* A manager to carry out smart reruns

Copyright (c) 1998-2004 The Regents of the University of California.
All rights reserved.
Permission is hereby granted, without written agreement and without
license or royalty fees, to use, copy, modify, and distribute this
software and its documentation for any purpose, provided that the above
copyright notice and the following two paragraphs appear in all copies
of this software.

IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.

THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES,
ENHANCEMENTS, OR MODIFICATIONS.

PT_COPYRIGHT_VERSION_2
COPYRIGHTENDKEY

*/

package ptolemy.actor;

import ptolemy.actor.ModelDependencyGraph;
import ptolemy.actor.StreamActor;
import ptolemy.actor.gui.Configuration;
import ptolemy.actor.gui.Effigy;
import ptolemy.actor.gui.ModelDirectory;
import ptolemy.actor.gui.PtolemyEffigy;
import ptolemy.actor.parameters.ParameterPort;
import ptolemy.data.Token;
import ptolemy.graph.DirectedGraph;
import ptolemy.graph.Element;
import ptolemy.graph.Edge;
import ptolemy.graph.Node;
import ptolemy.gui.ComponentDialog;
import ptolemy.kernel.ComponentEntity;
import ptolemy.kernel.ComponentRelation;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.AbstractSettableAttribute;
import ptolemy.kernel.util.Attribute;
import ptolemy.kernel.util.ChangeListener;
import ptolemy.kernel.util.ChangeRequest;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.InternalErrorException;
import ptolemy.kernel.util.KernelException;
import ptolemy.kernel.util.Location;
import ptolemy.kernel.util.NameDuplicationException;
import ptolemy.kernel.util.Nameable;
import ptolemy.kernel.util.NamedObj;
import ptolemy.kernel.util.Settable;
import ptolemy.kernel.util.ValueListener;
import ptolemy.kernel.util.Workspace;
import ptolemy.moml.MoMLChangeRequest;

import java.io.PrintStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

import java.net.URL;
import java.net.MalformedURLException;

import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.HashMap;
import java.util.HashSet;

//////////////////////////////////////////////////////////////////////////
//// SmartRerunManager

/**
   A cache manager to help carry out smart reruns.
   Note still experimental... doesnt work with composite actors yet.
   If you have to stream inside a composite actor the behavior is unknown.
   Note: uses same cache manager algorithm as the Vistrails system 
   
   You are now able to select actors in your workflow that cannot be rerun.
   The actors you select will be rerun again even if their parameters and 
   inputs have not changed.  This is because we do not save the outputs
   of the actors you select.  This feature is  useful if an actor produces some
   nondeterministic result and thus must be rerun everytime.  Another case where 
   this is useful is when an actor creates a data product, such as a file, that may or
   may not exist when the workflow is rerun.  If you unselect and actor as rerunable
   it will then save the results in the cache again but only use them when that actor 
   is not selected.

   @author Oscar Barney
   @version $Id: SmartRerunManager.java,v 1.1 2006/01/19 00:39:11 barney Exp $
   
*///TODO probably want to extend Attribute but for now that breaks the remove
//thing in mananger.
public class  SmartRerunManager extends AbstractSettableAttribute 
            implements ExecutionListener, ChangeListener, TokenSentListener {

    /** Construct a SmartRerunManager in the default 
    *  workspace with an empty string as its name. The listener 
     *  is added to the list of objects in the workspace. Increment the 
     *  version number of the workspace.
     */
    public SmartRerunManager() 
            throws IllegalActionException, NameDuplicationException{
        super();
        _addIcon();
        _init();
    }
     
    /** Construct a SmartRerunManager in the workspace 
    *  with an empty name.  The listener is added to the list of objects 
     *  in the workspace.  Increment the version number of the workspace.
     *
     *  @param workspace The workspace for this object.
     */
    public SmartRerunManager(Workspace workspace) 
            throws IllegalActionException, NameDuplicationException{
        super(workspace);
        _addIcon();
        _init();
    }
     
    /** Construct a SmartRerunManager in the given 
    *  container with the given name.  The container argument must 
     *  not be null, or a NullPointerException will be thrown.
     *  If the name argument is null, then the name is set to the
     *  empty string. Increment the version number of the workspace.
     *   
     *  @param container Container of the director.
     *  @param name Name of this listener.
     *  @exception IllegalActionException If the listener is not compatible
     *   with the specified container.  May be thrown in a derived class.
     *  @exception NameDuplicationException If the container is not a
     *   CompositeActor and the name collides with an entity in the container.
     */   
    public SmartRerunManager(CompositeEntity container, String name)
            throws IllegalActionException, NameDuplicationException {
        super(container, name);
        _addIcon();
        _init();
    }
    ///////////////////////////////////////////////////////////////////
    ////                         public methods                    ////
    
    //The following 6 functions are used to implement AbstractSettableAttribute.
    //This class is implemented because of the function validate which seems
    //to be the only time that the manager lets attributes know that it is
    //about to run the model.  Without using this function changes to the 
    //manager would be needed to let the ProvenanceExecutionListener know
    //that the model is about to fire and that it should register with the
    //appropriate listeners.  NOTE: Implementing AbstractSettableAttribute
    //is not the cleanest solution because the ProvenanceExecutionListener
    //really is not "settable" but it does allow us to minimaze the changes 
    //to Ptolemy II.
    public void addValueListener(ValueListener listener) {
        //do nothing
        //System.out.println("addValList Called");
    }

    public String getExpression() {
        //System.out.println("getExp Called");
        return "not really a settable attribute";
    }

    public Settable.Visibility getVisibility() {
        //System.out.println("getVis Called");
        return Settable.NONE;
    }

    public void removeValueListener(ValueListener Listener) {
        //do nothing
        //System.out.println("remValList Called");
    }

    public void setVisibility(Settable.Visibility visibility) {
        //do nothing
        //System.out.println("setVis Called");
    }

    //This function is called when the manager is about to run the model
    //and allows us to register with the manager before the run happens.
    //We would do this in the constructor but it seems that when the object
    //is created there is no manager yet.
    public void validate() {
        _graph = ModelDependencyGraph.generateDependencyGraph
                ((CompositeActor)_container);    
    
        CompositeActor topLevel = (CompositeActor)getContainer();
        Manager manager = topLevel.getManager();
        
        if(!_registered){
            manager.addExecutionListener(this);
            manager.addChangeListener(this);

            Collection nodes = _graph.nodes();
            Iterator portIterator = nodes.iterator();
            while(portIterator.hasNext()) {
                IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
                if(port.isOutput()) {
                    port.addTokenSentListener(this);
                }
            }
            _registered = true;
        }
    }
    
    public void externalFileEvent
            (String file, String action, Actor actor, String location){      
    }
    public void executionError(Manager manager, Throwable throwable) {
    }
    
    /** Report that the current execution has finished and the 
     *  wrapup sequence has completed normally. The number of 
     *  successfully completed iterations can be obtained by calling 
     *  getIterationCount() on the manager.
     *  @param manager The manager controlling the execution.
     */        
    public void executionFinished(Manager manager) {
        executionFinished();
    }
    
    /** Report that the manager has changed state.
     *  To access the new state, use the getState() method of Manager.
     *
     *  @param manager The manager controlling the execution.
     *  @see Manager#getState()
     */
    public void managerStateChanged(Manager manager){
	    Manager.State state = manager.getState();
        //System.out.println("STATE: "+state);
        if (state == Manager.PREINITIALIZING) {
            CompositeActor topLevel = (CompositeActor)getContainer();
            //if the container of this is null then it has been removed and we should unregister.
            if(topLevel == null){
                //manager.removeExecutionListener(this);
                //dont do this anymore because of concurent modification exception. Let it unregister itself
                manager.removeChangeListener(this);
                _registered = false;
                //Unregister all ports because this listener has been removed.
                Collection nodes = _graph.nodes();
                Iterator portIterator = nodes.iterator();
                while(portIterator.hasNext()) {
                    IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
                    if(port.isOutput()) {
                        port.removeTokenSentListener(this);
                    }
                }
            }
        }
    }
    
    
    /** This function is the implementation of a TokenSentListener.  IOPort
     *  will call this function when a token is sent so we can store it.
     *  
     *  @param event The event representing the token(s) sent.
     */
    public void tokenSentEvent(TokenSentEvent event) {
        IOPort port = event.getPort();
        int channel = event.getChannel();
        Token token = event.getToken();
        Token[] tokenArray = event.getTokenArray();
        int vectorLength = event.getVectorLength();

        if(vectorLength != -1) {
            if(channel != -1) {
                _tokenArraySent(channel, tokenArray, vectorLength, port);
            } else {
                //find channels because it is a broadcast
                Receiver[][] farReceivers;
                farReceivers = port.getRemoteReceivers();
                for (int i = 0; i < farReceivers.length; i++) {
                    _tokenArraySent(i, tokenArray, vectorLength, port);
                }
            }
        } else {
            //only one token was sent
            if(channel != -1) {
                _tokenSent(channel, token, port);
            } else {
                //find channels because it is a broadcast
                Receiver[][] farReceivers;
                farReceivers = port.getRemoteReceivers();
                for (int i = 0; i < farReceivers.length; i++) {
                    _tokenSent(i, token, port);
                }
            }
        }
    }
    

    
    /** React to a change request has been successfully executed.
     *  This method is called after a change request
     *  has been executed successfully.
     *  @param change The change that has been executed, or null if
     *   the change was not done via a ChangeRequest.
     */
    public void changeExecuted(ChangeRequest change) {
		//update graph for every graph change not EVERY change
        if(change.getSource().toString().indexOf("ActorGraph") != -1){
			_graph = ModelDependencyGraph.generateDependencyGraph
				((CompositeActor)_container);
        }
    }
    
    public void changeFailed(ChangeRequest change, Exception exception) {
		System.out.println("changeFailed()");
    }
    
    /** Saves tokens from the run and if it was a smart rerun opens a
    *  new window with the workflow as it existed before we deleted things
    *  and put the streamActor in.  
    */
    public void executionFinished(){
		
        //TODO store tokens with key should be done somewhere else 
        //(maybe when manager IDLE event happens)
        //if manager iterates more than once because we will be overwriting the
        //tokens for each previous iteration
        _storeTokensWithKey();
        
        if(_model != null){  //which means that we are ending a smart rerun
                Nameable configuration = 
                        Configuration.findEffigy(toplevel()).toplevel();
                try{
                    URL url = _model.toURL();
                    if (configuration instanceof Configuration) {
                        ((Configuration) configuration).openModel(
                                url, url, url.toExternalForm());
                    
                        ModelDirectory dir = ((Configuration) configuration).getDirectory();
                        Effigy effigy = dir.getEffigy(url.toExternalForm());
                        if(effigy != null){
                            NamedObj model = ((PtolemyEffigy)effigy).getModel();
                            List list =  model.attributeList(SmartRerunManager.class);
                            Iterator itr = list.iterator();
                            if(itr.hasNext()){
                                Attribute listener = (Attribute)itr.next(); 
                                //copy over the stored tokens
                                ((SmartRerunManager)listener)._streamRepository = _streamRepository;
                                //copy over the list of actors and weather they are cacheable
                                ((SmartRerunManager)listener)._cacheableMap = _cacheableMap;
                           
                                //TODO still need to call ProvenanceExecutionListener.setCacheData if we
                                //want it to work after the smart rerun?  yes we do
                                
                                //TODO figure out how to deactivate the old window or
                                //remove the smart rerun deal and make it work just doing
                                //the same thing over and over
                                
                            }else{
                                //should never happen
                                System.out.println("there is not a provenance listener in the new model!");
                            }
                            _model = null;
                        }else{
                            System.out.println("Could not find the effigy");
                        }
                    }else {
                        throw new InternalErrorException(
                                "Expected top-level to be a Configuration: "
                                + configuration.getFullName());
                    }
                } catch (Exception ex){
                    System.out.println(ex + " when trying to read temp file");
                }
            }
    
    }
    
    /** Start an execution of a smart rerun. 
     */
    public void startSmartRun(Manager manager) throws IllegalActionException {
        //needed or we get errors about unresolved types
        try {
            _workspace.getWriteAccess();
            try {
                TypedCompositeActor.resolveTypes((TypedCompositeActor) _container);
            } catch (TypeConflictException ex) {
                //FIXME change this to whatever it should be
                System.out.println("CRAP TYPES CONFLICT: " + ex);
            } 
        } finally {
            _workspace.doneWriting();
        }
        
        _graph = ModelDependencyGraph.generateDependencyGraph
            ((CompositeActor)_container);

        Collection nodes = _graph.nodes();
        Iterator portIterator = nodes.iterator();
        while(portIterator.hasNext()) {
            IOPort port = (IOPort)((Element)portIterator.next()).getWeight();
            String name = port.getContainer().getFullName();
            if(port.isOutput()) {
                if(!_cacheableMap.containsKey(name)){
                    _cacheableMap.put(name, new Boolean(false));
                }
            }
        }
            
        NameTrueFalseQuery panel = new NameTrueFalseQuery(_cacheableMap, "Not Rerunable");
        
        ComponentDialog dialog = new ComponentDialog(null,
                        "Pick actors that cannot be rerun:", panel);

        //next we put all nonCacheable actors in this set for later use
        _nonCacheableActors = new HashSet();
        //also we will save the boolean values for next time the menu is displayed
        Iterator actorIterator = _cacheableMap.keySet().iterator();
        while(actorIterator.hasNext()){
            String name = (String)actorIterator.next();
            Boolean temp = new Boolean(panel.getBooleanValue(name));
            _cacheableMap.put(name, temp);
            if(temp.booleanValue() == true){
                _nonCacheableActors.add(name);
            }    
        }
        
        //TODO if they press cancel do something intelligent
        if (!(dialog.buttonPressed().equals("OK"))) {
            // Restore original parameter values?
            panel.restore();
        }
        
        //now check For a Smart Rerun
        _checkForSmartRerun(manager);        
        
    }
        
    ///////////////////////////////////////////////////////////////////
    ////                         private variables                 ////
    
     //TODO move to protected variabels list
    /** key is the "unique id" generated for each subnetwork
    *  value is the tokens hashmap that was generated for
    *  that workflow run. 
     */
    protected HashMap _streamRepository = new HashMap();

    private NamedObj _container;

    //current graph
    private DirectedGraph _graph;

    //Holds the MoML of the orginal model while we modify things for 
    //the smart rerun.
    private File _model;

    //A flag to tell the provenance execution listener if it is 
    //registered with the manager or not.
    private boolean _registered;
        
    //graph before stream actor was added and things were changed
    private DirectedGraph _savedGraph;  
    
    //key is actor, value is full name before its container was set to null
    private HashMap _savedNames;  
    
    //the actors that are not cacheable
    private Set _nonCacheableActors = new HashSet();
    
    // key is a concatenation of the strings of the from TypedIOPort and
    //  the to TypedIOPort. value is the list of Token lists. 
    private HashMap _tokens = new HashMap();
    
    //This map will keep track of the actors and their status as cacheble
    //or not.  The key is the actor name and the value is ture or false.
    private HashMap _cacheableMap = new HashMap();
        
       
    ///////////////////////////////////////////////////////////////////
    ////                     private methods                  //////
    
    /**  Add an XML graphic as a hint to UIs for rendering the director.
     */ 
    private void _addIcon() {
        _attachText("_iconDescription", "<svg>\n" +
                "<rect x=\"-50\" y=\"-15\" "
                + "width=\"100\" height=\"30\" "
                + "style=\"fill:red\"/>\n" +
                "</svg>\n");
    }
    
    /** Check to see if any of the subnetworks in this graph
    *  have their results stored in the cache by calling _returnStreamEdges().
     *  This function is really what gets the whole smart re-run things started.
     */
    public void _checkForSmartRerun(Manager manager) 
            throws IllegalActionException {
        HashSet actorsToRerun = new HashSet();
        HashMap streamEdges = _returnStreamEdges(actorsToRerun);
         //System.out.println("StreamEdges: \n" + streamEdges);
         //System.out.println("ActorsToRerun: \n" + actorsToRerun);
        
        //if there are edges to stream on 
        if(!streamEdges.isEmpty()){      
            try{
                //create a temp file that will have the MOML you wanted to run which
                //we will reload when the stream actor is done so user can continue working
                _model = File.createTempFile("temp",".xml");
                _model.deleteOnExit();
                PrintStream stream = new PrintStream(new FileOutputStream(_model));
                //TODO assumes this is in the top level container
                stream.println(_container.exportMoML(_container.getName()));
                _savedGraph = _graph;
                _savedNames = _getActorNames(_container);
            } catch (Exception ex){
                    System.out.println(ex + " when trying to create temp file");
            }        
            //puts the stream actors in the work space etc.
            _initializeStreaming(streamEdges, actorsToRerun);
            manager.startRun();    
        } else if(actorsToRerun.isEmpty()) {
            System.out.println("You have done all this before!");
        } else {
            System.out.println("You have not tried any of this yet!");
            manager.startRun();
        }
    }
    
    //TODO do more testing with cycles
    //TODO make sure all this stuff works when a port is and input AND output
    /** Recursively finds the edges we need to stream on from the sink.
    *  We keep track of cycle nodes and sinks we have already been to so that we 
    *  dont do wasted work.  The cycle nodes are important because we cant stream
    *  to a node that is in a cycle because the behavior would be undefined.  Thus we
    *  back out of cycles and stream from a node not involved in the cycle.
    *  
    *  @param sink  Where to start the search.
    *  @param cycleNodes  All the nodes that are in a cycle in the graph.
    *  @param duplicates Keeps track of sinks we have already been to.
    *  @param actorsToRerun  Used to pass the set of actors to rerun back to the 
    *                                           caller.
    */
    private HashMap _findStreamEdgesFromSink
            (Node sink, Collection cycleNodes, HashSet duplicates, HashSet actorsToRerun){

        HashMap returnMap = new HashMap();
        //if we have seen this before return empty list.  This will happen if sink is in a loop
        if(duplicates.contains(sink)){
            return returnMap;
        } else{
            duplicates.add(sink);
        }
        
        Node pred;
        Collection predecessors = _graph.predecessors(sink);
        Iterator itr = predecessors.iterator(); 
        while(itr.hasNext()){
            pred = (Node)itr.next();
            if(!((IOPort)((Element)pred).getWeight()).isOutput()){
                returnMap.putAll(_findStreamEdgesFromSink(pred, cycleNodes, duplicates, actorsToRerun));
            } else {
                UniqueId id = _getUniqueId(pred);
                //if there is a stream for this unique id and pred and sink arnt in a loop
                //System.out.println("ID: " + id);
                if (id.isEmpty()) {
                    actorsToRerun.add(((IOPort)((Element)pred).getWeight()).getContainer());
                    returnMap.putAll(_findStreamEdgesFromSink(pred, cycleNodes, duplicates, actorsToRerun));
                } else if(!_streamRepository.containsKey(id) || 
                        (cycleNodes.contains(pred) && cycleNodes.contains(sink))){
                    //pred's actor has to be rerun because its result isnt in the cache
                    actorsToRerun.add(((IOPort)((Element)pred).getWeight()).getContainer());
                    returnMap.putAll(_findStreamEdgesFromSink(pred, cycleNodes, duplicates, actorsToRerun));
                } else {
                    //add the edge from this actor to the one that we were previously at
                    Collection edge = _graph.successorEdges(pred,sink);
                    Iterator edgeItr = edge.iterator();
                    if(edgeItr.hasNext()){
                        //associate this edge with the streams that were sent over it
                        returnMap.put(edgeItr.next(), 
                        ((HashMap)_streamRepository.get(id)).get(
                        ((Nameable)((Element)pred).getWeight()).getFullName()
                        + ((Nameable)((Element)sink).getWeight()).getFullName()));

                    }
                }
            }
        }  
        return returnMap;
    }
    
    //TODO maybe do this along with something else to be more efficient
    /** Recursively gets all the names of the actors in the model.
     *  This list is used when we go to delete actors that dont need to be
     *  rerun.  we can subtrace the actors to be rerun from this list to get the
     *  ones we need to delete.
     *
     *  @param container  The container to get names from
     */
    private HashMap _getActorNames(NamedObj container){
        HashMap returnMap = new HashMap();
        //TODO make sure they give you the top level composite actor
        List embeddedActors = ((CompositeActor)container).deepEntityList();
        Iterator embeddedActorsIterator = embeddedActors.iterator();
        while (embeddedActorsIterator.hasNext()) {
            NamedObj embeddedActor = (NamedObj)embeddedActorsIterator.next();
            //Added to add all stuff in composite actors
            if(embeddedActor instanceof CompositeActor 
                        && ((CompositeEntity)embeddedActor).isOpaque()) {
               returnMap.putAll(_getActorNames(embeddedActor));
            } else {
                returnMap.put(embeddedActor, embeddedActor.getFullName());
            }
                
        }
        return returnMap;
    }
    
    /** For all sinks and output ports in the graph make a uniqueId.
     *   If we are saving results of a smart rerun then we do not want to
     *   return unique ids of subgraphs that were not rerun.  _savedGraph
     *   will help us keep track of what to save. 
     */
    private HashSet _getAllUniqueIds(){
        //list of unique ids to be returned
        HashSet returnSet = new HashSet();  
        //will check to see we dont enter actors twice
        HashSet duplicates = new HashSet();
        
        if(_savedGraph != null){
            //switch the graphs so that the uniqueIds are generated using the old graph
            //and we dont make ids for subgraphs that were not rerun.
            DirectedGraph temp = _graph;
            _graph = _savedGraph;
            _savedGraph = temp;
        }

        //get the sinks for this run
        Collection sinks = _graph.sinkNodes();
        Iterator sinkItr = sinks.iterator();
        while(sinkItr.hasNext()){
            Node sink = (Node)sinkItr.next();
            if(_savedGraph == null || 
                        _savedGraph.containsNodeWeight(sink.getWeight())){
                //add the subgraph ending with the sinks
                UniqueId temp = _getUniqueId(sink);
                //unless the UniqueID is for a non rerunable actor
                if(!temp.isEmpty()){
                    returnSet.add(temp);
                }
                returnSet.addAll(_getSubgraphUniqueIds(sink, duplicates));
            }
        }
        _savedGraph = null;
        return returnSet;
    }

    /** Return list of all NEEDED unique ids (where sink is an output port)
    *  that are in the subgraph before this sink.
     *
     *  @param sink  Where to start looking for unique ids
     *  @param duplicates  A set of nodes we have already been to
     */
    private HashSet _getSubgraphUniqueIds(Node sink, HashSet duplicates){
        HashSet returnSet = new HashSet();
        //if we have seen this before return empty set
        if(duplicates.contains(sink) ||  (_savedGraph != null && 
                        !_savedGraph.containsNodeWeight(sink.getWeight()))){
            return returnSet;
        } else{
            duplicates.add(sink);
        }
        
        //add unique ids for all the predecessors to sink.
        Node pred;
        Collection predecessors = _graph.predecessors(sink);
        Iterator itr = predecessors.iterator(); 
        while(itr.hasNext()){
            pred = (Node)itr.next();
            returnSet.addAll(_getSubgraphUniqueIds(pred, duplicates));
        }
        //add a unique id for all output ports
        if(((IOPort)((Element)sink).getWeight()).isOutput()){
            UniqueId temp = _getUniqueId(sink);
            //unless it is an id associated with an actor that can not be rerun
            if(!temp.isEmpty()){
                returnSet.add(temp);
            }
        }
        return returnSet;
    }
    
    
    //TODO make sure that the container name has nothing to do with the
    //ID... this is so that if you rename you file the cache will still work
    //may also help you out when you want to reusue stuff that was 
    //duplicated in another composite actor doing basically the same thing
    
    //TODO we can simplify the selection of attributes by sorting on class settable
    //like actor/gui/configurer does.
    
    /** Gets UniqueId for the subgraph that leads up to the sink given.
     *
     *  @param sink  Get id up to this sink.
     */
    private UniqueId _getUniqueId(Node sink){
         //strings that helpu us build the hashcode for the uniqueId
        String paramValue = "";
        String paramName = "";
        String containerName = "";
        String codeString = "";
        //collect info in these hashtables for the uniqueId object
        HashMap actorsAndParams = new HashMap();
        //associate parameters with values
        HashMap paramsAndValues;
        HashSet edgeSet = new HashSet();
        //will check to see we dont enter actors twice
        HashSet duplicates = new HashSet();  
        //get all the nodes that are behind the sink
        Collection subgraphNodes = _graph.backwardReachableNodes(sink);
        //add the sink to the subgraph
        subgraphNodes.add(sink);
        Iterator nodes = subgraphNodes.iterator();
        while(nodes.hasNext()){
            IOPort port = (IOPort)((Element)nodes.next()).getWeight();
            NamedObj container = port.getContainer();
            
            //if any of the actors that come before this one are not cacheable make a bogus id
            if(_nonCacheableActors.contains(container.getFullName()) ){
                codeString = "\n\n";
                actorsAndParams.clear();
                edgeSet.clear();
                return new UniqueId(actorsAndParams, edgeSet, codeString.hashCode());
            }
            
            if(!duplicates.contains(container)){
                duplicates.add(container);  //so we dont add this actor's state to the uniqueid again
                paramsAndValues = new HashMap();
                if(_savedNames == null){
                    containerName = container.getFullName();
                } else {
                    //get the name of the object before its container was set to null
                    containerName = (String)_savedNames.get(container);
                }
                codeString += containerName;
                
                //TODO only add this stuff if that actor depends on the parameters
                //right now we do the safe thing and rerun everything.
                //if the toplevel has parameters then we should add them
                if(container.getContainer() == container.toplevel()){
                    List result = container.toplevel().attributeList(AbstractSettableAttribute.class);
                    Iterator params = result.iterator();
                    while(params.hasNext()) {
                        NamedObj param = (NamedObj)params.next();
                        paramName = param.getName();
                        String value = ((AbstractSettableAttribute)param).getExpression();
                        if(value == null){
                            paramValue = "null value";
                        } else{
                            paramValue = value.toString();
                        }
                        //TODO make sure this is the right thing to exclude
                        if (paramName.lastIndexOf("_windowProperties") == -1 &&
                            paramName.lastIndexOf("_doc") == -1 &&
                            paramName.lastIndexOf("_vergil") == -1 &&
                            paramName.lastIndexOf("_createdBy") == -1 &&
                            paramName.lastIndexOf("SmartRerunManager") == -1 &&
                            paramName.lastIndexOf("ProvenanceExecutionListener") == -1) {
                            paramsAndValues.put(paramName, paramValue);
                            codeString += "\t" + paramName + "_" + paramValue;
                        }
                    }
                }
                
                List result = container.attributeList(AbstractSettableAttribute.class);
                Iterator params = result.iterator();
                while(params.hasNext()) {
                    NamedObj param = (NamedObj)params.next();
                    paramName = param.getName();
                    String value = ((AbstractSettableAttribute)param).getExpression();
                    if(value == null){
                        paramValue = "null value";
                    } else{
                        paramValue = value.toString();
                    }
                    //TODO make sure this is the right thing to exclude
                    if (paramName.lastIndexOf("_windowProperties") == -1 &&
                         paramName.lastIndexOf("_doc") == -1 &&
                         paramName.lastIndexOf("firingsPerIteration") == -1) {
                        paramsAndValues.put(paramName, paramValue);
                        codeString += "\t" + paramName + "_" + paramValue;
                        //NOTE we have to make sure that a param name  a:: and value 6
                        //doesnt get mixed up with name a: and value :6 if the delimiter
                        //was :.  both would look like a:::6.  Thus we use unique id object
                    }
                }
                actorsAndParams.put(containerName, paramsAndValues);
                codeString += "\n";
            }
        }
        //now we get the edges
        Collection edges = _graph.edges();
        Iterator edgeItr = edges.iterator();
        while(edgeItr.hasNext()){
            Edge e = (Edge) edgeItr.next();
            if(subgraphNodes.contains(e.sink())){
                String temp = containerName + "." + 
                         ((NamedObj)e.source().getWeight()).getName() 
                         + containerName + "." + 
                         ((NamedObj)e.sink().getWeight()).getName();
                codeString += "\n" + temp;
                edgeSet.add(temp);
            }
        }
        //System.out.println("CODE for sink: " + sink + "\n" + codeString + "\n");
        
        return new UniqueId(actorsAndParams, edgeSet, codeString.hashCode());
    }
    
    private void _init(){
		_container = getContainer();
        _model = null;
        
        _registered = false;
        
        _graph = null;
        _savedGraph = null;
        _savedNames = null;
    }
    
    /** Puts the stream actor on the workspace and deletes the actors we wont
	 *  be rerunning.  Also removes some relations.
     *
     *  @param streamEdges This map has the edges we are going to stream on and their
     *                                        associated tokens.
     *  @param actorsToRerun  This is the set of actors that we will be rerunning.
     */
    private void _initializeStreaming(HashMap streamEdges, HashSet actorsToRerun) {

		//create a map of StreamActors where each StreamActor is associated with
		//the CompositeActor it is inside of.
		HashMap streamActors = new HashMap();
		
        // StreamActor streamActor = null;
//         if (_container != null) {
//             try {//TODO when we start putting the stream actor int containers were
//             //it goes the _container var may not be needed anymore
//                 streamActor = new StreamActor((CompositeEntity)_container);
//             } catch (Exception e) {
//                 System.out.println("Error while creating StreamActor: \n" + e);
//             }
//         }
//         streamActor.initializeStreamActor();

        for (Iterator changedEdgesEntry = streamEdges.entrySet().iterator(); 
             changedEdgesEntry.hasNext(); ) {
            Map.Entry edgeTokenMap = (Map.Entry)changedEdgesEntry.next();
            Edge changedEdge = (Edge) edgeTokenMap.getKey();
            TypedIOPort source = (TypedIOPort) changedEdge.source().getWeight();
            TypedIOPort sink = (TypedIOPort) changedEdge.sink().getWeight();

            //double check we got the kinds of ports we were expecting.
            if (source instanceof TypedIOPort) { /*this is what we want */ }
            else if (source instanceof ParameterPort)
                System.err.println("\nError!\n!!!!!Looks like you need to handles parameter ports!\n");
            else if (source instanceof IOPort) {
                System.err.println("Source is an IOPort. Here it is: " +source);
            }
            else {
                System.err.println("Error. I have no idea what source is: "+source);
            }

			//get CompositeActor this port/actor is inside of
			//XXXX not sure if we want source or sink actor...
			CompositeActor compositeActor = 
				(CompositeActor)source.getContainer().getContainer();
			
			StreamActor streamActor = 
				(StreamActor) streamActors.get(compositeActor);
			if (streamActor == null) {
				try {
					streamActor = new StreamActor(compositeActor);
                } catch (IllegalActionException e) {
                    System.out.println("Error creating StreamActor. " + e);
                }
                catch (NameDuplicationException e) {
                    System.out.println("Error creating StreamActor. " + e);
                }

				streamActor.initializeStreamActor();
				streamActors.put(compositeActor, streamActor);
			}

            streamActor.addConnection(source, sink, 
                                      (HashMap) edgeTokenMap.getValue());
        }

        //get list of all the actors to remove.
        ArrayList actorsToRemove = new ArrayList(_savedNames.keySet());
        actorsToRemove.removeAll(actorsToRerun);
        //System.out.println("actors to remove:  \t" + actorsToRemove + "\n");

        for (Iterator actors = actorsToRemove.iterator(); actors.hasNext(); ) {
            ComponentEntity actor = (ComponentEntity) actors.next();
            try { 
                //System.out.println(" Removing actor " + actor);

				CompositeActor compositeActor = 
					(CompositeActor)actor.getContainer();
				StreamActor streamActor = 
					(StreamActor) streamActors.get(compositeActor);

				actor.setContainer(null); 

				if (streamActor == null) {
					streamActors.put(compositeActor, null);
					continue;
				}

                //make StreamActor take the location of the actor being removed.
                Location actorlocation = (Location)actor.getAttribute(
                   "_location");
                Location streamActorlocation = (Location)streamActor.getAttribute(
                   "_location");
                if (streamActorlocation == null) {
                    streamActorlocation = new Location(streamActor, "_location");
                    streamActorlocation.propagateExistence();
                }
                streamActorlocation.setLocation(actorlocation.getLocation());
            } catch (Exception e) {
                System.err.println("An error occured while removing the following actor: " + 
                                   actor);
                System.err.println("Error was: " + e);
            }
        }

		//remove composite actors if they are empty
		for (Iterator compositeActors = streamActors.keySet().iterator();
			 compositeActors.hasNext(); )
		{
			CompositeActor compositeActor = (CompositeActor) compositeActors.next();
			if (compositeActor.entityList().isEmpty()){
				try { 
					compositeActor.setContainer(null);
				} catch (Exception e) {
					System.err.println("An error occured while removing the " +
									   "following composite actor: " + compositeActor);
					System.err.println("Error was: " + e);
				}
            } else {
				//System.out.println("here are the children of " + compositeActor +
					//			   " : " + compositeActor.entityList());
           }                        
		}
		

        //get list of all the relations to remove and remove them.
        //TODO make sure _container is the one we really want here
        for (Iterator relations = 
                ((CompositeEntity)_container).relationList().iterator(); 
                relations.hasNext(); ) {
            ComponentRelation relation = (ComponentRelation)relations.next();
            if (relation.linkedPortList().isEmpty()) {
                //System.out.println("Removing relation: " + relation.getName());
                try {
                    relation.setContainer(null);
                } catch (IllegalActionException e) {
                    System.out.println("Error removing relation: " + e);
                }
                catch (NameDuplicationException e) {
                    System.out.println("Error removing relation: " + e);
                }
            }
        }

		//make an empty change request so that these updates are performed.
		MoMLChangeRequest request = new MoMLChangeRequest(
                    this,
					this,
                    "<doc></doc>"); //needed to put something here. 
		getContainer().requestChange(request);

        //NOTE: TODO we used to update the graph here but i dont think its needed
    }
    
    /** Returns the edges of the graph that we want to stream on
     *  and the tokens that go on that edge.  actorsToRerun will be
      *  the set of actors that we need to rerun.
      *
      *  @param actorsToRerun  Used to return the set of actors to rerun.
      *
      */
    private HashMap _returnStreamEdges(HashSet actorsToRerun){
        //TODO make sure loops thing works right
        
        HashMap returnMap = new HashMap();    
        //get the nodes that are in loops and we cannot stream to
		Collection cycleNodes = _graph.cycleNodeCollection();
        //keep track of duplicates so we dont do more work 
        //than we have to or get stuck in loops;
        HashSet duplicates = new HashSet();
        //get the sinks for this run
        Collection sinks = _graph.sinkNodes();

        Iterator sinkItr = sinks.iterator();
        while(sinkItr.hasNext()){
            Node sink = (Node)sinkItr.next();
            //if this branch of the graph is not done
            if(!_streamRepository.containsKey(_getUniqueId(sink))){
                actorsToRerun.add(((IOPort)((Element)sink).getWeight()).getContainer());
                returnMap.putAll(_findStreamEdgesFromSink(sink, cycleNodes, duplicates, actorsToRerun));
            } else {
               // System.out.println("FOUND a whole branch that has been done before!!");
            }
        }

        return returnMap;
    }
    
    
    /** Get all of the keys and store tokens the cache associated with
     *  the keys.
     */
    private void _storeTokensWithKey(){

        //first we need to generate the keys
        HashSet uniqueIds =  _getAllUniqueIds();
		Iterator itr = uniqueIds.iterator();
        while(itr.hasNext()){
            _streamRepository.put((UniqueId)itr.next(), _tokens);
        }
 		
        //make tokens be a new HashMap so for the next run it is empty and 
        //thus faster to work with
        _tokens = new HashMap();
    }
    
    public void _tokenArraySent(int channelIndex, Token[] tokens, 
            int vectorLength, IOPort fromPort){
        //just call tokenSent allot!
        for(int i = 0; i < vectorLength; i++){
            _tokenSent(channelIndex, tokens[i], fromPort);
        } 
    }
    
    /** Report that a token was sent from fromPort on the
    *  channel indicated.
     *
     *  @param channelIndex The channel the token was sent on
     *  @param token The token that was sent
     *  @param fromPort The token was sent from this port
     */
    public void _tokenSent(int channelIndex, Token token, 
                                         IOPort fromPort) {
        _trySaveToken(channelIndex, token, fromPort);
    }
    
    /** This function will try and save the token given to it in a temporary
    *  cache so that it can be later saved and used with later runs.  
     *
     *  @param channelIndex  The channel the token was sent on.
     *  @param token  The token we are going to save.
     *  @param fromPort  The source that created this token.
     *  @param toPort  Where the token was sent.
     */
    private void _trySaveToken(int channelIndex, Token token, IOPort fromPort) {
        Receiver[][] farReceivers = fromPort.getRemoteReceivers();
        TypedIOPort toPort = null;
		if (farReceivers.length == 0) {
// 			System.out.println("when trying to getRemoteReceivers from port: " +
// 							   fromPort + ", we got no receivers.");
			return;
		} else if (false) {
            //if not cacheableActors.containes(fromPort.getContainer())
            //TODO we probably will not want to save things that are created by 
            //the non cacheable actor.
            return;
        }
                    
        for (int j = 0; j < farReceivers[channelIndex].length; j++) {
            toPort = (TypedIOPort)farReceivers[channelIndex][j].getContainer();
        
            //if this is a smart rerun then dont save tokens from
            //the stream actor because we already have them
            if(_savedGraph == null ||
                _savedGraph.containsNodeWeight(fromPort)){
                        
                //add token to a list of tokens sent on this port and channel      
                String key = ((Nameable)fromPort).getFullName()
                    + ((Nameable)toPort).getFullName();
                //NOTE that this does not gauentee that different streams get back
                //on the right channels but it does guarentee that there will be the same 
                //number of ingoing channels.
                HashMap streamMap = (HashMap) _tokens.get(key);
                if(streamMap == null){
                    streamMap = new HashMap();
                }
                List tokenList = (List) streamMap.get(new Integer(channelIndex));
                if (tokenList == null){
                    tokenList = new ArrayList();
                }
                tokenList.add(token);
                streamMap.put(new Integer(channelIndex), tokenList);
                _tokens.put(key, streamMap);
            }
        }
    
    }
    
    ///////////////////////////////////////////////////////////////////
    ////                        inner classes                      ////
    
    
    //TODO make sure that the order things get connected doesnt
   // give you a different unique id... try and do some kinda sort 
   // like vistrails does.
    /** UniqueID class.  The basic idea is that we need some way to tell
     *  one subnetwork from another.
     */
    public class UniqueId {
        private UniqueId(HashMap actorsAndParams, 
                HashSet edges, int hashCode){
            _actorAndParams = actorsAndParams;
            _edges = edges;
            _hash = hashCode;
        }
        
        public boolean equals(Object obj){
            if(obj instanceof UniqueId == false){
                return false;
            }
            UniqueId id = (UniqueId) obj;
            return _actorAndParams.equals(id._actorAndParams) 
                    && _edges.equals(id._edges);
        }

        public int hashCode(){
            return _hash;
        }
        
        /** Return true if this is a bogus ID */
        public boolean isEmpty(){
            if(_actorAndParams.isEmpty() && _edges.isEmpty()){
                return true;
            } else {
                return false;
            }
        }
        
        /** Return a string description of this class. */
        public String toString(){
            return "\n" + _actorAndParams.toString() + "\n" + _edges.toString();
        }

        //key is actor value is parameter Hashmap
        private HashMap _actorAndParams;
        //key is edgename
        private HashSet _edges;
        //precomputed has code
        private int _hash;
     }
}