/** Image Crop is an image manipulation actor for cropping the desired
 * image into a smaller image.
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package util;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.*;
import ptolemy.data.DoubleToken;
import ptolemy.data.AWTImageToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.actor.parameters.PortParameter;
 
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.lang.Float;

//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;
import java.awt.image.renderable.ParameterBlock;


/**
 * ImageCrop can be used to crop a given image into the specified
 * width,height at the specificied co-oridinates on an original image.
 *
 @author Nandita Mangal
 @version $Id: ImageCrop.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */

public class ImageCrop extends TypedAtomicActor
{

    ///////////////////////////////////////////////////////////////////////
    ////                   Parameters and Ports                       ////

    
    /*
     * Specifiy the croppedImage width and height
     */
    public StringParameter cropImageWidth;
    public StringParameter cropImageHeight;
  
    
    /*
     * The input image to be cropped
     */
    public TypedIOPort imgSrc;

    /*
     * Output cropped Image
     */
    public TypedIOPort outputImage;
  

    /*
     * Input X co-ordinate for cropping (Upper Left corner)
     */
    public PortParameter xCo_ordinate;

 
     /*
     * Input Y co-ordinate for cropping (Upper Left corner)
     */
    public PortParameter yCo_ordinate;
  


  /**
   * constructor for ImageCrop with the given container and name.
   *
   *@param  container                     The container.
    *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageCrop(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
	 imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);

         outputImage =new TypedIOPort(this,"Output Image",false,true);
    	 outputImage.setTypeEquals(BaseType.OBJECT);
         outputImage.setMultiport(false);

            
         cropImageWidth = new StringParameter(this,"Crop Image Width:");
	 cropImageHeight = new StringParameter (this,"Crop Image Height:");
	 cropImageWidth.addChoice("100");
	 cropImageHeight.addChoice("100");


         xCo_ordinate = new PortParameter(this, "X Co-ordinate"); 
         xCo_ordinate.setExpression("100"); 
	         
         yCo_ordinate = new PortParameter(this, "Y Co-ordinate");  
	 yCo_ordinate.setExpression("100");
        
  }
    ////////////////////////////////////////////////////////////////////////
    ////                 public methods                                ////

 
  /**
   * The source image is retrieved from the input port of the actor
   * and a JAI PlanarImage is created from the input image.
   * After calculating, the cropped image's specifications such 
   * as width,height,Co-ordinates the image is cropped with the JAI.create
   * operator.
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();

     /* PlanarImages in JAI are not inherited from the Java Image class and hence
    *  are stored in normal ObjectToken and transferred through actor ports. However
    * normal Java AWT images are stored in AWTImageTokens.
    */

    Object inputToken = imgSrc.get(0);
    PlanarImage input =null;

    if(inputToken instanceof AWTImageToken)  //then we know its an AWT Image (*not PlanarImage)
    {
    
    	Image src = (Image)((AWTImageToken)inputToken).getValue();         
        input= JAI.create("awtimage",src);	       

    }
    else if(inputToken instanceof ObjectToken) //then its a PlanarImage probably.	
    {
	if (  ((ObjectToken)inputToken).getValue() instanceof PlanarImage)
	 {
	      input=(PlanarImage)((ObjectToken)inputToken).getValue(); 
				  
         } 

    }

    
	
     // Parse the parameters.
    xCo_ordinate.update();
    yCo_ordinate.update();  

    Token t_x = xCo_ordinate.getToken();
    Token t_y = yCo_ordinate.getToken();
    String x_str="";
    String y_str="";   

    if(t_x instanceof IntToken)
          x_str= new String ( ((IntToken)t_x).intValue() + "");
    else if(t_x instanceof DoubleToken)
	    x_str= new String ( ((DoubleToken)t_x).intValue() + "");

    if(t_y instanceof IntToken)
         y_str= new String ( ((IntToken)t_y).intValue() + "");
    else if (t_y instanceof DoubleToken)
	   y_str= new String ( ((DoubleToken)t_y).intValue() + "");

  
    float x  = Float.parseFloat(x_str);
    float y =  Float.parseFloat(y_str);
   

    float width = Float.parseFloat(cropImageWidth.stringValue());
    float height = Float.parseFloat(cropImageHeight.stringValue());
  
    // Create a ParameterBlock with parameters.
    ParameterBlock pb = new ParameterBlock();
    pb.addSource(input);
    pb.add(x);
    pb.add(y);
    pb.add(width);
    pb.add(height);
    // Create the output cropped image 
    PlanarImage output = JAI.create("crop",pb,null);

    // translate the image by (-x,-y)
    pb = new ParameterBlock();
    pb.addSource(output);
    pb.add(-x);
    pb.add(-y);

    // Create the output translated image                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             
    output = JAI.create("translate",pb,null);

    outputImage.broadcast(new ObjectToken(output));


 
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

   


}
