/** Image Scaler is an image manipulation actor for scaling a given
 * image.
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */


package util;
import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.DoubleToken;
import ptolemy.data.AWTImageToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.actor.parameters.PortParameter;
import ptolemy.data.*;
 
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.lang.Float;

//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.PlanarImage;
import java.awt.image.renderable.ParameterBlock;
import javax.media.jai.InterpolationNearest;


/**
 * ImageScale can be used to scale a given image by the 
 * specified scale.
 *
 @author Nandita Mangal
 @version $Id: ImageScaler.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */



public class ImageScaler extends TypedAtomicActor
{

    //////////////////////////////////////////////////////////
    ////               Parameters and Ports              ////

    /*
     * The extent of scaling desired.
     */
   public PortParameter scalePercentage;
    
    /*
     * The input image to be scaled
     */
   public TypedIOPort imgSrc;
   
    /*
     * Output scaled Image
     */
   public TypedIOPort imgDest;
 
   
  

  /**
   * constructor for ImageScaler with given container & name
   *
   *@param  container                     The container.
   *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageScaler(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
	 imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);

         imgDest = new TypedIOPort(this, "Dest Image",false,true);
	 imgDest.setTypeEquals(BaseType.OBJECT);
         imgDest.setMultiport(false);

         scalePercentage= new PortParameter(this,"Scale Percent");
	 int valueScale=50;
	 scalePercentage.setExpression("10");
	 for(int i=0;i<20;i++)
	 {
			scalePercentage.addChoice(new String(valueScale + ""));
			valueScale +=50;  
	 }

  }

  /**
   * Retrieve the input image from the port and convert it to a JAI PlanarImage
   * After calculation of scale value, use JAI operators to scale the image.
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();

     /* PlanarImages in JAI are not inherited from the Java Image class and hence
    *  are stored in normal ObjectToken and transferred through actor ports. However
    * normal Java AWT images are stored in AWTImageTokens.
    */

    BufferedImage srcBuffered=null;
    PlanarImage image=null;

    Object inputToken = imgSrc.get(0);
    
    if(inputToken instanceof AWTImageToken)  //then we know its an AWT Image (*not PlanarImage)
    {
    
    	Image src = (Image)((AWTImageToken)inputToken).getValue(); 
        image= JAI.create("awtimage",src);	    


    }
    else if(inputToken instanceof ObjectToken) //then its a PlanarImage probably.	
    {
	if (  ((ObjectToken)inputToken).getValue() instanceof PlanarImage)
	 {
		image=(PlanarImage)((ObjectToken)inputToken).getValue(); 
				  
         } 

    }
    String scale_str="";
    scalePercentage.update();
    Token t = scalePercentage.getToken();
    if(t instanceof IntToken)
        scale_str= new String( ((IntToken)t).intValue() + "");
    else if(t instanceof DoubleToken)
        scale_str = new String(((DoubleToken)t).doubleValue() + "");
 
    float scale = Float.parseFloat(scale_str);

    // Scales the original image    
    
    ParameterBlock pb = new ParameterBlock();
    pb.addSource(image);
    pb.add(scale);
    pb.add(scale);
    pb.add(0.0F);
    pb.add(0.0F);
    pb.add(new InterpolationNearest());
    
    // Creates a new, scaled image and send it to the output
    PlanarImage scaledImage = JAI.create("scale", pb);
    imgDest.broadcast(new ObjectToken(scaledImage));
    
    
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }



}

 
