 /** Image Translate is an image manipulation actor for translating
 * based on the specified information.
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

package util;


import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.StringToken;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.*;
import ptolemy.data.type.BaseType;
import ptolemy.data.ObjectToken;
import ptolemy.data.AWTImageToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.expr.StringParameter;
import ptolemy.actor.parameters.PortParameter;
import ptolemy.data.*;
 
import java.awt.*;
import java.awt.image.*;
import java.net.*;
import javax.imageio.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.lang.Float;

//For JAI usage
import javax.media.jai.JAI;
import javax.media.jai.*;
import javax.media.jai.PlanarImage;
import java.awt.image.renderable.ParameterBlock;
import javax.media.jai.KernelJAI;
import javax.media.jai.InterpolationBicubic;
import javax.media.jai.InterpolationBicubic2;


/**
 * ImageTranslate can be used to translate at the given
 * X,Y translation co-ordinates.
 @author Nandita Mangal
 @version $Id: ImageTranslate.java,v 1.1 2005/11/02 00:21:52 
 @category.name Image Manipulation
 */

public class ImageTranslate extends TypedAtomicActor
{

    ///////////////////////////////////////////////////////////////////////
    ////                   Parameters and Ports                       ////

   
    /*
     * The input image to be translated
     */
    public TypedIOPort imgSrc;

    /*  
     * The output manipulated image
     */
    public TypedIOPort imgDest;  
    
    /*  
     * The translating coordinate X
     */
    public PortParameter x_translation ; 

    /*  
     * The translating coordinate y
     */
    public PortParameter y_translation ;

    /*  
     * The type of interpolation for the
     * above transformation.
     */
     public StringParameter interpolationType ;



  /**
   * constructor for ImageCrop with the given container and name.
   *
   *@param  container                     The container.
    *@param  name                          The name of this actor.
   *@exception  IllegalActionException    If the actor cannot be contained
   *   by the proposed container.
   *@exception  NameDuplicationException  If the container already has an
   *   actor with this name.
   */
  public ImageTranslate(CompositeEntity container, String name)
    throws
      NameDuplicationException, IllegalActionException
  {
    super(container, name);

         imgSrc = new TypedIOPort(this, "Source Image", true, false);  
	 imgSrc.setTypeEquals(BaseType.OBJECT);
         imgSrc.setMultiport(false);
          
         imgDest = new TypedIOPort(this, "Dest Image", false, true);  
	 imgDest.setTypeEquals(BaseType.OBJECT);
         imgDest.setMultiport(false);

         x_translation  = new PortParameter(this,"X Translation"); 
         x_translation.setExpression("50");

         y_translation  = new PortParameter(this,"Y Translation");
         y_translation.setExpression("100");

         interpolationType = new StringParameter(this,"InterpolationType");
         interpolationType.setExpression("INTERPOLATION_NEAREST");
	 interpolationType.addChoice("INTERPOLATION_BILINEAR");
	 interpolationType.addChoice("INTERPOLATION_BICUBIC");
         interpolationType.addChoice("INTERPOLATION_BICUBIC2");
 
        
  }
    ////////////////////////////////////////////////////////////////////////
    ////                 public methods                                ////

 
  /**
   * The source image is retrieved from the input port of the actor
   * and a JAI PlanarImage is created from the input image.
   * After calculating, the cropped image's specifications such 
   * as width,height,Co-ordinates the image is cropped with the JAI.create
   * operator.
   *@exception  IllegalActionException  If there is no director.
   */
  public void fire()
    throws IllegalActionException
  {
    super.fire();
    PlanarImage input=null;
    Object inputToken = imgSrc.get(0);
  
    if(inputToken instanceof AWTImageToken)
    {
    	Image src = (Image)((AWTImageToken)inputToken).getValue(); 
    	input= JAI.create("awtimage",src);	    
    }
    else if (inputToken instanceof ObjectToken)
    {
	input =(PlanarImage)((ObjectToken)inputToken).getValue();

    }

    
    x_translation.update();
    y_translation.update();

    Token t_x = x_translation.getToken();
    Token t_y = y_translation.getToken();
    String x_str="";
    String y_str="";   

    if(t_x instanceof IntToken)
          x_str= new String ( ((IntToken)t_x).intValue() + "");
    else if(t_x instanceof DoubleToken)
	    x_str= new String ( ((DoubleToken)t_x).intValue() + "");

    if(t_y instanceof IntToken)
         y_str= new String ( ((IntToken)t_y).intValue() + "");
    else if (t_y instanceof DoubleToken)
	   y_str= new String ( ((DoubleToken)t_y).intValue() + "");
    

    float x_trans =Float.parseFloat(x_str);
    float y_trans =Float.parseFloat(y_str);

    String interpolation = interpolationType.stringValue();

    Interpolation interp=null;

    if(interpolation.equals("INTERPOLATION_NEAREST"))
	    interp = new InterpolationNearest();
    else if(interpolation.equals("INTERPOLATION_BILINEAR"))
	    interp = new InterpolationBilinear();
    else if(interpolation.equals("INTERPOLATION_BICUBIC"))
	    interp = new InterpolationBicubic(8);
    else if(interpolation.equals("INTERPOLATION_BICUBIC2"))
            interp = new InterpolationBicubic2(8);
  
     // Create a ParameterBlock
     ParameterBlock pb = new ParameterBlock();
          pb.addSource(input);                   // The source image
          pb.add((float)Math.max(x_trans, 0));    // The x translation
          pb.add((float)Math.max(y_trans, 0));    // The y translation
          pb.add(interp); // The interpolation

     // Create the translate operation
     PlanarImage output = JAI.create("translate", pb, null);

     imgDest.broadcast(new ObjectToken(output));

 
  }

  /**
   * Post fire the actor. Return false to indicate that the
   * process has finished. If it returns true, the process will
   * continue indefinitely.
   *
   *@return
   */
  public boolean postfire()
  {
    return false;
  }

  /**
   * Pre fire the actor.
   *  Calls the super class's prefire in case something is set there.
   *
   *@return
   *@exception  IllegalActionException
   */
  public boolean prefire()
    throws IllegalActionException
  {
    return super.prefire();
  }

   


}
