/** WeatherDataSelection actor gives the user the option of selecting
 * between weather data for further contour plotting.
 *
 * Copyright (c) 2004 The Regents of the University of California.
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY
 * FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES
 * ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN
 * IF THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
 * PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY
 * OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT,
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */
package org.ROADnet;

import java.util.HashMap;

import ptolemy.actor.TypedAtomicActor;
import ptolemy.actor.TypedIOPort;
import ptolemy.data.BooleanToken;
import ptolemy.data.ObjectToken;
import ptolemy.data.expr.Parameter;
import ptolemy.data.type.BaseType;
import ptolemy.kernel.CompositeEntity;
import ptolemy.kernel.util.IllegalActionException;
import ptolemy.kernel.util.NameDuplicationException;

//////////////////////////////////////////////////////////////////////////
//// WeatherDataSelection 
 
/** Weather Data Selection allows users to select which contours to plot,
    such as only temperature plots or only humidity plots or both, etc.
  
   @author Nandita Mangal
   @version $Id: WeatherDataSelection.java
   @category.name orb
   @category.name external execution
 */


public class WeatherDataSelection extends TypedAtomicActor {
    /**
     *  @param container The container.
     *  @param name The name of this actor.
     *  @exception IllegalActionException If the actor cannot be contained
     *   by the proposed container.
     *  @exception NameDuplicationException If the container already has an
     *   actor with this name.
     */
    public WeatherDataSelection(CompositeEntity container, String name)
	throws NameDuplicationException, IllegalActionException {
	super(container, name);

	//input weather data
	input =new TypedIOPort(this,"input",true,false);
	input.setMultiport(false);
	input.setTypeEquals(BaseType.OBJECT);

	//output user selected weather data
	output = new TypedIOPort(this, "output", false, true);
	output.setMultiport(false);
	output.setTypeEquals(BaseType.OBJECT);  

	temperature = new Parameter(this, "temperature", new BooleanToken(false));
	temperature.setTypeEquals(BaseType.BOOLEAN);

    
	humidity = new Parameter(this,"humidity",new BooleanToken(false));
	humidity.setTypeEquals(BaseType.BOOLEAN);
	
	pressure = new Parameter(this,"pressure",new BooleanToken(false));
	pressure.setTypeEquals(BaseType.BOOLEAN);
	
	wind = new Parameter(this,"wind",new BooleanToken(false));
	wind.setTypeEquals(BaseType.BOOLEAN);

	

    }
    ////////////////////////////////////////////////////////////
    ///                 public methods                    /////

     /** Change the hash map to reflect user selections on
      *  data contour plotting.
     *
     *  @exception IllegalActionException
     */

    public void fire()  throws IllegalActionException
    {
	
	//get the selection based on user selection
	
	boolean tempSelection=((BooleanToken)temperature.getToken()).booleanValue();
	boolean pressureSelection =((BooleanToken)pressure.getToken()).booleanValue();
	boolean humidSelection =((BooleanToken)humidity.getToken()).booleanValue();
	boolean windSelection =((BooleanToken)wind.getToken()).booleanValue();

	//original weather data
	HashMap inputMatrices = (HashMap) ((ObjectToken)(input.get(0))).getValue();

	//remove data unselected by the user
	if(tempSelection == false)
	  inputMatrices.remove("temperature");
	if(pressureSelection == false)
	   inputMatrices.remove("pressure");
	if(humidSelection == false)
	   inputMatrices.remove("humidity");
	if(windSelection == false)
	    inputMatrices.remove("wind");
	
	output.broadcast(new ObjectToken(inputMatrices));
    }

    /////////////////////////////////////////////////////////////////
    ////                 public ports and Parameters            ////

    /* contour plotting data selections*/
    public Parameter temperature;
    public Parameter pressure;
    public Parameter wind;
    public Parameter humidity;

    /* HashMap containg weather data*/
    public TypedIOPort output;
    public TypedIOPort input;

   
}
